<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use App\Models\Service;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AppointmentController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = Reservation::with(['customer', 'personel', 'room', 'appointmentRequest.service']);

        // Personel sadece kendi randevularını görür
        if (in_array($user->role, ['personel', 'staff'], true)) {
            $query->where('personel_id', $user->id);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }

        if ($request->filled('status')) {
            $query->whereIn('status', $this->statusFilterCandidates((string) $request->status));
        }

        if ($request->has('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }

        if ($request->has('staff_id')) {
            $query->where('personel_id', $request->staff_id);
        }

        $appointments = $query->orderBy('date', 'desc')
            ->orderBy('start_time', 'desc')
            ->paginate($request->get('per_page', 20));

        return response()->json([
            'success' => true,
            'data' => collect($appointments->items())->map(fn($apt) => $this->formatAppointment($apt))->values(),
            'meta' => [
                'current_page' => $appointments->currentPage(),
                'last_page' => $appointments->lastPage(),
                'per_page' => $appointments->perPage(),
                'total' => $appointments->total(),
            ],
        ]);
    }

    public function show(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        $appointment = Reservation::with(['customer', 'personel', 'room', 'appointmentRequest.service'])
            ->find($id);

        if (!$appointment) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu bulunamadı',
            ], 404);
        }

        // Personel sadece kendi randevularını görebilir
        if (in_array($user->role, ['personel', 'staff'], true) && $appointment->personel_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Bu randevuya erişim yetkiniz yok',
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => $this->formatAppointment($appointment),
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'service_id' => 'required|exists:services,id',
            'date' => 'required|date',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'staff_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
            'total_price' => 'nullable|numeric|min:0',
            'deposit_paid' => 'nullable|numeric|min:0',
            'status' => 'nullable|in:open,confirmed,cancelled,completed,beklemede,onaylandi,tamamlandi,iptal',
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();
        [$salonId, $companyId] = $this->resolveSalonScope($user);

        $customer = Customer::find($request->customer_id);
        if (!$customer || ($salonId && (int) $customer->salon_id !== (int) $salonId)) {
            return response()->json([
                'success' => false,
                'message' => 'Müşteri bulunamadı veya bu salona ait değil',
            ], 403);
        }

        $service = Service::find($request->service_id);
        if (!$service || ($salonId && (int) $service->salon_id !== (int) $salonId)) {
            return response()->json([
                'success' => false,
                'message' => 'Hizmet bulunamadı veya bu salona ait değil',
            ], 403);
        }

        $hasConflict = $this->hasAnyConflict($request->date, $request->start_time, $request->staff_id, $request->room_id);
        if ($hasConflict) {
            return response()->json([
                'success' => false,
                'message' => 'Bu saat dilimi uygun değil',
                'code' => 'APPOINTMENT_CONFLICT',
            ], 409);
        }

        try {
            DB::beginTransaction();

            $duration = $service->duration ?? 60;
            $endTime = $request->filled('end_time')
                ? $request->end_time
                : Carbon::createFromFormat('H:i', $request->start_time)->addMinutes($duration)->format('H:i');
            $totalPrice = (float) ($request->total_price ?? $service->price ?? 0);
            $depositPaid = (float) ($request->deposit_paid ?? 0);
            $status = $this->normalizeStatus($request->input('status', 'confirmed'));

            $appointment = Reservation::create([
                'customer_id' => $request->customer_id,
                'beauty_salon_id' => $salonId,
                'company_id' => $companyId,
                'personel_id' => $request->staff_id,
                'room_id' => $request->room_id,
                'date' => $request->date,
                'start_time' => $request->start_time,
                'end_time' => $endTime,
                'total_price' => $totalPrice,
                'deposit_paid' => $depositPaid,
                'remaining_amount' => max(0, $totalPrice - $depositPaid),
                'status' => $status,
                'notes' => $request->notes,
                'created_by' => $user->id,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Randevu başarıyla oluşturuldu',
                'data' => $this->formatAppointment($appointment->load(['customer', 'personel', 'room', 'appointmentRequest.service'])),
            ], 201);
        } catch (\Throwable $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Randevu oluşturulurken hata oluştu',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function update(Request $request, int $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'sometimes|exists:customers,id',
            'service_id' => 'sometimes|exists:services,id',
            'date' => 'sometimes|date',
            'start_time' => 'sometimes|date_format:H:i',
            'end_time' => 'sometimes|nullable|date_format:H:i',
            'staff_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
            'total_price' => 'nullable|numeric|min:0',
            'deposit_paid' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string|max:1000',
            'status' => 'sometimes|in:open,confirmed,cancelled,completed,beklemede,onaylandi,tamamlandi,iptal',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();
        $appointment = Reservation::with(['customer', 'personel', 'room', 'appointmentRequest.service'])->find($id);
        if (!$appointment) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu bulunamadı',
            ], 404);
        }

        if (!$this->canAccessAppointment($user, $appointment)) {
            return response()->json([
                'success' => false,
                'message' => 'Bu randevuyu güncelleme yetkiniz yok',
            ], 403);
        }

        $newDate = $request->input('date', $appointment->date?->format('Y-m-d'));
        $newStart = $request->input('start_time', $appointment->start_time);
        $newStaff = $request->has('staff_id') ? $request->input('staff_id') : $appointment->personel_id;
        $newRoom = $request->has('room_id') ? $request->input('room_id') : $appointment->room_id;

        $serviceId = $request->input('service_id');
        $service = $serviceId ? Service::find($serviceId) : ($appointment->appointmentRequest?->service ?? null);
        $duration = $service->duration ?? 60;
        $newEnd = $request->filled('end_time')
            ? $request->input('end_time')
            : Carbon::createFromFormat('H:i', $newStart)->addMinutes($duration)->format('H:i');

        $hasConflict = $this->hasConflict($newDate, $newStart, $newEnd, $newStaff, $newRoom, $appointment->id);
        if ($hasConflict) {
            return response()->json([
                'success' => false,
                'message' => 'Bu saat dilimi uygun değil',
                'code' => 'APPOINTMENT_CONFLICT',
            ], 409);
        }

        $appointment->update([
            'customer_id' => $request->input('customer_id', $appointment->customer_id),
            'personel_id' => $newStaff,
            'room_id' => $newRoom,
            'date' => $newDate,
            'start_time' => $newStart,
            'end_time' => $newEnd,
            'total_price' => $request->input('total_price', $appointment->total_price),
            'deposit_paid' => $request->input('deposit_paid', $appointment->deposit_paid ?? 0),
            'remaining_amount' => max(
                0,
                (float) $request->input('total_price', $appointment->total_price) -
                (float) $request->input('deposit_paid', $appointment->deposit_paid ?? 0)
            ),
            'status' => $this->normalizeStatus($request->input('status', $appointment->status)),
            'notes' => $request->input('notes', $appointment->notes),
            'updated_by' => $user->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Randevu güncellendi',
            'data' => $this->formatAppointment($appointment->fresh()->load(['customer', 'personel', 'room', 'appointmentRequest.service'])),
        ]);
    }

    public function checkAvailability(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'date' => 'required|date',
            'service_id' => 'required|exists:services,id',
            'staff_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $service = Service::find($request->service_id);
        $duration = $service->duration ?? 60;
        $date = Carbon::parse($request->date);

        $slots = [];
        $start = Carbon::createFromTimeString('09:00');
        $end = Carbon::createFromTimeString('21:00');

        while ($start->lt($end)) {
            $slotStart = $start->format('H:i');
            $slotEnd = $start->copy()->addMinutes($duration)->format('H:i');

            $available = !$this->hasConflict(
                $date->toDateString(),
                $slotStart,
                $slotEnd,
                $request->staff_id,
                $request->room_id
            );

            $slots[] = [
                'time' => $slotStart,
                'available' => $available,
                'reason' => $available ? null : 'Bu saatte başka bir randevu var',
            ];

            $start->addMinutes(30);
        }

        return response()->json([
            'success' => true,
            'data' => $slots,
        ]);
    }

    public function approve(Request $request, int $id): JsonResponse
    {
        return $this->updateStatus($request, $id, 'onaylandi', 'Randevu onaylandı');
    }

    public function cancel(Request $request, int $id): JsonResponse
    {
        return $this->updateStatus($request, $id, 'iptal', 'Randevu iptal edildi');
    }

    public function complete(Request $request, int $id): JsonResponse
    {
        return $this->updateStatus($request, $id, 'tamamlandi', 'Randevu tamamlandı');
    }

    public function calendar(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $startDate = $request->get('start', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $staffId = $request->get('staff_id');

        $query = Reservation::with(['customer', 'personel', 'room', 'appointmentRequest.service'])
            ->whereBetween('date', [$startDate, $endDate])
            ->whereNotIn('status', ['cancelled', 'iptal']);

        // Personel sadece kendi randevularını görür
        if (in_array($user->role, ['personel', 'staff'], true)) {
            $query->where('personel_id', $user->id);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
            
            // Belirli bir personel için filtrele
            if ($staffId) {
                $query->where('personel_id', $staffId);
            }
        }

        $appointments = $query->orderBy('date', 'asc')
            ->orderBy('start_time', 'asc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $appointments->map(fn($apt) => $this->formatAppointment($apt)),
        ]);
    }

    private function formatAppointment(Reservation $appointment): array
    {
        $primaryService = $appointment->appointmentRequest?->service;

        return [
            'id' => $appointment->id,
            'customer_id' => $appointment->customer_id,
            'customer_name' => $appointment->customer->name ?? null,
            'customer_phone' => $appointment->customer->phone ?? null,
            'service_id' => $primaryService?->id,
            'service_name' => $primaryService?->name,
            'staff_id' => $appointment->personel_id,
            'staff_name' => $appointment->personel->name ?? null,
            'room_id' => $appointment->room_id,
            'room_name' => $appointment->room->name ?? null,
            'date' => optional($appointment->date)->format('Y-m-d'),
            'start_time' => $appointment->start_time,
            'end_time' => $appointment->end_time,
            'status' => $this->normalizeStatus((string) $appointment->status),
            'total_price' => (float) $appointment->total_price,
            'deposit' => (float) $appointment->deposit_paid,
            'notes' => $appointment->notes,
            'created_at' => $appointment->created_at?->toISOString(),
            'updated_at' => $appointment->updated_at?->toISOString(),
        ];
    }

    private function updateStatus(Request $request, int $id, string $status, string $message): JsonResponse
    {
        $user = $request->user();
        $appointment = Reservation::with(['customer', 'personel', 'room', 'appointmentRequest.service'])->find($id);
        if (!$appointment) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu bulunamadı',
            ], 404);
        }

        if (!$this->canAccessAppointment($user, $appointment)) {
            return response()->json([
                'success' => false,
                'message' => 'Bu randevuya işlem yetkiniz yok',
            ], 403);
        }

        $appointment->update([
            'status' => $this->normalizeStatus($status),
            'updated_by' => $user->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => $this->formatAppointment($appointment->fresh()->load(['customer', 'personel', 'room', 'appointmentRequest.service'])),
        ]);
    }

    private function canAccessAppointment($user, Reservation $appointment): bool
    {
        if (in_array($user->role, ['personel', 'staff'], true)) {
            return (int) $appointment->personel_id === (int) $user->id;
        }

        if (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            [$salonId] = $this->resolveSalonScope($user);
            return (int) $appointment->beauty_salon_id === (int) $salonId;
        }

        return false;
    }

    private function resolveSalonScope($user): array
    {
        $salonId = $user->beauty_salon_id ?: $user->id;
        return [$salonId, $user->company_id];
    }

    private function hasAnyConflict(string $date, string $startTime, ?int $staffId, ?int $roomId): bool
    {
        return $this->hasConflict($date, $startTime, null, $staffId, $roomId);
    }

    private function hasConflict(string $date, string $startTime, ?string $endTime, ?int $staffId, ?int $roomId, ?int $ignoreReservationId = null): bool
    {
        $query = Reservation::whereNotIn('status', ['cancelled', 'iptal'])
            ->whereDate('date', $date)
            ->where(function ($q) use ($startTime, $endTime) {
                $q->whereBetween('start_time', [$startTime, $endTime ?? $startTime])
                    ->orWhereBetween('end_time', [$startTime, $endTime ?? $startTime])
                    ->orWhere(function ($q2) use ($startTime, $endTime) {
                        $q2->where('start_time', '<=', $startTime)
                            ->where('end_time', '>=', $endTime ?? $startTime);
                    });
            });

        if ($ignoreReservationId) {
            $query->where('id', '!=', $ignoreReservationId);
        }

        if ($staffId) {
            $query->where('personel_id', $staffId);
        }

        if ($roomId) {
            $query->where('room_id', $roomId);
        }

        return $query->exists();
    }

    private function normalizeStatus(string $status): string
    {
        return match ($status) {
            'beklemede' => 'open',
            'onaylandi' => 'confirmed',
            'tamamlandi' => 'completed',
            'iptal' => 'cancelled',
            default => $status,
        };
    }

    private function statusFilterCandidates(string $status): array
    {
        $normalized = $this->normalizeStatus($status);

        return match ($normalized) {
            'open' => ['open', 'beklemede', 'pending'],
            'confirmed' => ['confirmed', 'onaylandi', 'approved'],
            'completed' => ['completed', 'tamamlandi'],
            'cancelled' => ['cancelled', 'iptal', 'rejected'],
            default => [$normalized, $status],
        };
    }
}

