<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\AppointmentRequest;
use App\Models\Customer;
use App\Models\Reservation;
use App\Models\Service;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class AppointmentRequestController extends Controller
{
    private function resolveSalonId(User $user): int
    {
        return $user->beauty_salon_id ?? $user->id;
    }

    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Randevu taleplerine erişim yetkiniz yok.',
            ], 403);
        }

        $salonId = $this->resolveSalonId($user);
        $status = $request->get('status', 'pending');

        $requests = AppointmentRequest::where('beauty_salon_id', $salonId)
            ->where('status', $status)
            ->with(['service', 'personel', 'room'])
            ->orderByDesc('created_at')
            ->paginate((int) $request->get('per_page', 20));

        return response()->json([
            'success' => true,
            'data' => collect($requests->items())->map(function (AppointmentRequest $item) {
                return [
                    'id' => $item->id,
                    'name' => $item->name,
                    'surname' => $item->surname,
                    'phone' => $item->phone,
                    'status' => $item->status,
                    'appointment_date' => $item->appointment_date?->format('Y-m-d'),
                    'appointment_time' => $item->appointment_time ? substr((string) $item->appointment_time, 0, 5) : null,
                    'notes' => $item->notes,
                    'created_at' => $item->created_at?->toISOString(),
                    'service_name' => $item->service?->name,
                    'service_price' => (float) ($item->service?->price ?? 0),
                    'personel_name' => $item->personel?->name,
                    'room_name' => $item->room?->name,
                ];
            })->values(),
            'meta' => [
                'current_page' => $requests->currentPage(),
                'last_page' => $requests->lastPage(),
                'per_page' => $requests->perPage(),
                'total' => $requests->total(),
            ],
        ]);
    }

    public function approve(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Randevu taleplerini onaylama yetkiniz yok.',
            ], 403);
        }

        $salonId = $this->resolveSalonId($user);

        $appointmentRequest = AppointmentRequest::where('id', $id)
            ->where('beauty_salon_id', $salonId)
            ->where('status', 'pending')
            ->with(['service'])
            ->first();

        if (!$appointmentRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu talebi bulunamadı.',
            ], 404);
        }

        $validated = $request->validate([
            'appointment_date' => 'nullable|date',
            'appointment_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'personel_id' => [
                'required',
                'integer',
                Rule::exists('users', 'id')->where(function ($query) use ($salonId) {
                    $query->where('beauty_salon_id', $salonId)
                        ->where('role', 'personel');
                }),
            ],
            'room_id' => 'nullable|exists:rooms,id',
            'total_price' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $customer = null;
            if ($appointmentRequest->customer_id) {
                $customer = Customer::where('id', $appointmentRequest->customer_id)
                    ->where('beauty_salon_id', $salonId)
                    ->first();
            }

            if (!$customer) {
                $customerName = trim(($appointmentRequest->name ?? '') . ' ' . ($appointmentRequest->surname ?? ''));
                $customer = Customer::firstOrCreate(
                    [
                        'phone' => $appointmentRequest->phone,
                        'beauty_salon_id' => $salonId,
                    ],
                    [
                        'name' => $customerName !== '' ? $customerName : 'Müşteri',
                        'email' => null,
                        'created_by' => $user->id,
                    ]
                );
            }

            $service = $appointmentRequest->service;
            if (!$service && is_array($appointmentRequest->service_ids) && count($appointmentRequest->service_ids) > 0) {
                $service = Service::whereIn('id', $appointmentRequest->service_ids)
                    ->where('beauty_salon_id', $salonId)
                    ->first();
            }

            $date = $validated['appointment_date']
                ?? $appointmentRequest->appointment_date?->format('Y-m-d')
                ?? now()->format('Y-m-d');
            $startTimeRaw = $validated['appointment_time']
                ?? ($appointmentRequest->appointment_time ? substr((string) $appointmentRequest->appointment_time, 0, 5) : null)
                ?? now()->addHour()->format('H:i');
            $start = Carbon::parse($date . ' ' . $startTimeRaw);
            $durationMinutes = (int) ($service?->duration ?? 60);
            if ($durationMinutes <= 0) {
                $durationMinutes = 60;
            }
            $end = !empty($validated['end_time'])
                ? Carbon::parse($date . ' ' . $validated['end_time'])
                : $start->copy()->addMinutes($durationMinutes);

            $totalPrice = (float) ($validated['total_price'] ?? ($service?->price ?? 0));

            $reservation = Reservation::create([
                'code' => 'REZ-' . strtoupper(uniqid()),
                'customer_id' => $customer->id,
                'salon_id' => null,
                'beauty_salon_id' => $salonId,
                'date' => $date,
                'start_time' => $start->format('H:i:s'),
                'end_time' => $end->format('H:i:s'),
                'personel_id' => $validated['personel_id'],
                'room_id' => $validated['room_id'] ?? $appointmentRequest->room_id,
                'status' => 'confirmed',
                'total_price' => $totalPrice,
                'deposit_paid' => 0,
                'remaining_amount' => $totalPrice,
                'notes' => $validated['notes'] ?? $appointmentRequest->notes,
                'items_data' => json_encode([
                    'service' => $service ? [
                        'id' => $service->id,
                        'name' => $service->name,
                        'price' => $totalPrice,
                    ] : null,
                ]),
                'created_by' => $user->id,
            ]);

            $appointmentRequest->update([
                'status' => 'approved',
                'reservation_id' => $reservation->id,
                'customer_id' => $customer->id,
                'appointment_date' => $date,
                'appointment_time' => $start->format('H:i:s'),
                'personel_id' => $validated['personel_id'],
                'room_id' => $validated['room_id'] ?? $appointmentRequest->room_id,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Randevu talebi onaylandı.',
                'data' => [
                    'reservation_id' => $reservation->id,
                    'appointment_request_id' => $appointmentRequest->id,
                ],
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('iOS randevu talebi onaylama hatası', [
                'request_id' => $id,
                'user_id' => $user->id ?? null,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Randevu talebi onaylanırken hata oluştu.',
            ], 500);
        }
    }

    public function reject(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Randevu taleplerini reddetme yetkiniz yok.',
            ], 403);
        }

        $salonId = $this->resolveSalonId($user);

        $appointmentRequest = AppointmentRequest::where('id', $id)
            ->where('beauty_salon_id', $salonId)
            ->where('status', 'pending')
            ->first();

        if (!$appointmentRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu talebi bulunamadı.',
            ], 404);
        }

        $reason = trim((string) $request->input('reason', ''));
        $note = $appointmentRequest->notes ? ($appointmentRequest->notes . "\n\n") : '';
        if ($reason !== '') {
            $note .= 'Red Nedeni: ' . $reason;
        }

        $appointmentRequest->update([
            'status' => 'rejected',
            'notes' => $note !== '' ? $note : $appointmentRequest->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Randevu talebi reddedildi.',
            'data' => [
                'appointment_request_id' => $appointmentRequest->id,
            ],
        ]);
    }
}

