<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Services\Sms\OtpService;
use App\Services\SmsService;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    public function login(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required_without:phone|email',
            'phone' => 'required_without:email|string',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = null;
        if ($request->has('email')) {
            $user = User::where('email', $request->email)->first();
        } elseif ($request->has('phone')) {
            $phone = preg_replace('/[^0-9]/', '', $request->phone);
            $user = User::where('phone', $phone)->first();
        }

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'E-posta veya şifre hatalı',
                'code' => 'INVALID_CREDENTIALS',
            ], 401);
        }

        // Sadece admin ve personel giriş yapabilir
        if (!in_array($user->role, ['güzellik_salonu', 'personel', 'admin', 'super_admin'])) {
            return response()->json([
                'success' => false,
                'message' => 'Bu hesap türü mobil uygulamaya giriş yapamaz',
                'code' => 'INVALID_ROLE',
            ], 403);
        }

        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Hesabınız askıya alınmış',
                'code' => 'ACCOUNT_SUSPENDED',
            ], 403);
        }

        if ($user->license_expires_at && $user->license_expires_at->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'Lisans süreniz dolmuş',
                'code' => 'LICENSE_EXPIRED',
            ], 403);
        }

        $user->tokens()->delete();
        $token = $user->createToken('ios-app', ['*'], now()->addDays(180))->plainTextToken;

        return response()->json([
            'success' => true,
            'data' => [
                'access_token' => $token,
                'token_type' => 'Bearer',
                'expires_in' => 180 * 24 * 60 * 60,
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'role' => $user->role,
                    'is_personel_admin' => $user->is_personel_admin ?? false,
                    'salon_id' => $user->beauty_salon_id,
                    'salon_name' => $user->salon_name,
                    'avatar' => $user->profile_image ? asset('storage/' . $user->profile_image) : null,
                ],
            ],
        ]);
    }

    public function logout(Request $request): JsonResponse
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Başarıyla çıkış yapıldı',
        ]);
    }

    public function me(Request $request): JsonResponse
    {
        $user = $request->user();

        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Hesabınız askıya alınmış',
                'code' => 'ACCOUNT_SUSPENDED',
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'role' => $user->role,
                'is_personel_admin' => $user->is_personel_admin ?? false,
                'salon_id' => $user->beauty_salon_id,
                'salon_name' => $user->salon_name,
                'avatar' => $user->profile_image ? asset('storage/' . $user->profile_image) : null,
            ],
        ]);
    }

    public function updateProfile(Request $request): JsonResponse
    {
        $user = $request->user();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|max:255|unique:users,email,' . $user->id,
            'phone' => 'sometimes|string|max:20|unique:users,phone,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $payload = $request->only(['name', 'email', 'phone']);
        if ($request->filled('password')) {
            $payload['password'] = Hash::make($request->password);
        }

        $user->update($payload);
        $user->refresh();

        return response()->json([
            'success' => true,
            'message' => 'Profil başarıyla güncellendi',
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'role' => $user->role,
                'is_personel_admin' => $user->is_personel_admin ?? false,
                'salon_id' => $user->beauty_salon_id,
                'salon_name' => $user->salon_name,
                'avatar' => $user->profile_image ? asset('storage/' . $user->profile_image) : null,
            ],
        ]);
    }

    public function sendDeleteOtp(Request $request, SmsService $smsService): JsonResponse
    {
        $user = $request->user();

        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil. Hesap silme için önce telefon numaranızı güncelleyin.',
            ], 400);
        }

        $result = $smsService->sendOtp($user->phone, 'account_deletion');
        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Doğrulama kodu telefon numaranıza gönderildi.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
        ], 400);
    }

    public function verifyDeleteOtp(Request $request, OtpService $otpService): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'otp_code' => 'required|string|size:6',
        ], [
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();
        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil.',
            ], 400);
        }

        $otpCode = (int) $request->otp_code;
        if (!$otpService->verify((string) $user->phone, $otpCode, 'account_deletion')) {
            return response()->json([
                'success' => false,
                'message' => 'Doğrulama kodu hatalı veya süresi dolmuş.',
            ], 422);
        }

        return response()->json([
            'success' => true,
            'message' => 'Doğrulama başarılı. Hesabı silme işlemini tamamlayabilirsiniz.',
        ]);
    }

    public function deleteAccount(Request $request, OtpService $otpService): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'otp_code' => 'required|string|size:6',
            'password' => 'required|string',
        ], [
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
            'password.required' => 'Şifre zorunludur.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil.',
            ], 400);
        }

        if (!Hash::check((string) $request->password, (string) $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Şifre hatalı.',
            ], 422);
        }

        $otpCode = (int) $request->otp_code;
        if (!$otpService->verify((string) $user->phone, $otpCode, 'account_deletion')) {
            return response()->json([
                'success' => false,
                'message' => 'Doğrulama kodu hatalı veya süresi dolmuş.',
            ], 422);
        }

        try {
            DB::beginTransaction();

            $user->tokens()->delete();
            $deleted = $user->delete();
            if (!$deleted) {
                throw new \RuntimeException('Kullanıcı silinemedi.');
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Hesabınız başarıyla silindi.',
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('iOS hesap silme hatası', [
                'user_id' => $user->id ?? null,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Hesap silinirken bir hata oluştu. Lütfen destek ile iletişime geçin.',
            ], 500);
        }
    }
}

