<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\Transaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CashboxController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $cashAccount = Account::getCashAccount($user->id);
        
        $baseQuery = Transaction::where('account_id', $cashAccount->id);
        
        // Personel sadece kendi işlemlerini görür
        if ($user->role === 'personel') {
            $baseQuery->where('created_by', $user->id);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            // Admin tüm işlemleri görür
        }
        
        // İstatistikler
        $totalIncome = (clone $baseQuery)->where('type', 'income')->sum('amount') ?? 0;
        $totalExpense = (clone $baseQuery)->where('type', 'expense')->sum('amount') ?? 0;
        $totalBalance = $totalIncome - $totalExpense;
        
        $today = Carbon::today();
        $todayIncome = (clone $baseQuery)->where('type', 'income')->whereDate('date', $today)->sum('amount') ?? 0;
        $todayExpense = (clone $baseQuery)->where('type', 'expense')->whereDate('date', $today)->sum('amount') ?? 0;
        
        $monthStart = Carbon::now()->startOfMonth();
        $monthEnd = Carbon::now()->endOfMonth();
        $monthlyIncome = (clone $baseQuery)->where('type', 'income')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;
        $monthlyExpense = (clone $baseQuery)->where('type', 'expense')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;
        
        // İşlemler
        $query = Transaction::where('account_id', $cashAccount->id)
            ->whereIn('type', ['income', 'expense']);
        
        if ($user->role === 'personel') {
            $query->where('created_by', $user->id);
        }
        
        $query->with(['account', 'reservation', 'customer']);
        
        if ($request->has('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }
        if ($request->has('type') && in_array($request->type, ['income', 'expense'])) {
            $query->where('type', $request->type);
        }
        
        $transactions = $query->orderBy('date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));
        
        return response()->json([
            'success' => true,
            'data' => [
                'statistics' => [
                    'total_income' => (float) $totalIncome,
                    'total_expense' => (float) $totalExpense,
                    'total_balance' => (float) $totalBalance,
                    'today_income' => (float) $todayIncome,
                    'today_expense' => (float) $todayExpense,
                    'monthly_income' => (float) $monthlyIncome,
                    'monthly_expense' => (float) $monthlyExpense,
                ],
                'transactions' => collect($transactions->items())->map(function ($transaction) {
                    return [
                        'id' => $transaction->id,
                        'type' => $transaction->type,
                        'amount' => (float) $transaction->amount,
                        'description' => $transaction->description,
                        'date' => $transaction->date?->format('Y-m-d'),
                        'customer_name' => $transaction->customer->name ?? null,
                        'reservation_id' => $transaction->reservation_id,
                        'created_at' => $transaction->created_at?->toISOString(),
                    ];
                })->values(),
                'meta' => [
                    'current_page' => $transactions->currentPage(),
                    'last_page' => $transactions->lastPage(),
                    'per_page' => $transactions->perPage(),
                    'total' => $transactions->total(),
                ],
            ],
        ]);
    }
    
    public function storeIncome(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'description' => 'required|string|max:500',
            'date' => 'required|date',
            'customer_id' => 'nullable|exists:customers,id',
        ]);
        
        $cashAccount = Account::getCashAccount($user->id);
        
        $transaction = Transaction::create([
            'account_id' => $cashAccount->id,
            'type' => 'income',
            'amount' => $request->amount,
            'description' => $request->description,
            'date' => $request->date,
            'customer_id' => $request->customer_id,
            'created_by' => $user->id,
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Gelir kaydedildi',
            'data' => [
                'id' => $transaction->id,
                'type' => $transaction->type,
                'amount' => (float) $transaction->amount,
                'description' => $transaction->description,
                'date' => $transaction->date?->format('Y-m-d'),
            ],
        ], 201);
    }
    
    public function storeExpense(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'description' => 'required|string|max:500',
            'date' => 'required|date',
        ]);
        
        $cashAccount = Account::getCashAccount($user->id);
        
        $transaction = Transaction::create([
            'account_id' => $cashAccount->id,
            'type' => 'expense',
            'amount' => $request->amount,
            'description' => $request->description,
            'date' => $request->date,
            'created_by' => $user->id,
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Gider kaydedildi',
            'data' => [
                'id' => $transaction->id,
                'type' => $transaction->type,
                'amount' => (float) $transaction->amount,
                'description' => $transaction->description,
                'date' => $transaction->date?->format('Y-m-d'),
            ],
        ], 201);
    }
}

