<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Reservation;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = Invoice::with(['reservation.customer', 'customer']);
        
        // Personel sadece kendi randevularına ait faturaları görür
        if ($user->role === 'personel') {
            $query->whereHas('reservation', function ($q) use ($user) {
                $q->where('personel_id', $user->id);
            });
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->whereHas('reservation', function ($q) use ($user) {
                    $q->where('beauty_salon_id', $user->beauty_salon_id);
                })->orWhere('created_by', $user->id);
            } else {
                $query->where('created_by', $user->id);
            }
        }
        
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        
        if ($request->has('date_from')) {
            $query->whereDate('invoice_date', '>=', $request->date_from);
        }
        
        if ($request->has('date_to')) {
            $query->whereDate('invoice_date', '<=', $request->date_to);
        }
        
        $invoices = $query->orderBy('invoice_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));
        
        return response()->json([
            'success' => true,
            'data' => collect($invoices->items())->map(fn($invoice) => $this->formatInvoice($invoice))->values(),
            'meta' => [
                'current_page' => $invoices->currentPage(),
                'last_page' => $invoices->lastPage(),
                'per_page' => $invoices->perPage(),
                'total' => $invoices->total(),
            ],
        ]);
    }
    
    public function show(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        $invoice = Invoice::with(['reservation.customer', 'customer'])->find($id);
        
        if (!$invoice) {
            return response()->json([
                'success' => false,
                'message' => 'Fatura bulunamadı',
            ], 404);
        }
        
        // Personel kontrolü
        if ($user->role === 'personel') {
            if ($invoice->reservation && $invoice->reservation->personel_id !== $user->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu faturaya erişim yetkiniz yok',
                ], 403);
            }
        }
        
        return response()->json([
            'success' => true,
            'data' => $this->formatInvoice($invoice),
        ]);
    }
    
    public function downloadPdf(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        $invoice = Invoice::with(['reservation.customer', 'customer'])->find($id);
        
        if (!$invoice) {
            return response()->json([
                'success' => false,
                'message' => 'Fatura bulunamadı',
            ], 404);
        }
        
        // Personel kontrolü
        if ($user->role === 'personel') {
            if ($invoice->reservation && $invoice->reservation->personel_id !== $user->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu faturaya erişim yetkiniz yok',
                ], 403);
            }
        }
        
        $pdfUrl = $invoice->pdf_path 
            ? asset('storage/' . $invoice->pdf_path)
            : route('admin.invoices.pdf', $invoice->id);
        
        return response()->json([
            'success' => true,
            'data' => [
                'pdf_url' => $pdfUrl,
                'public_url' => $invoice->public_url,
            ],
        ]);
    }
    
    private function formatInvoice(Invoice $invoice): array
    {
        return [
            'id' => $invoice->id,
            'invoice_number' => $invoice->invoice_number,
            'total_amount' => (float) $invoice->total_amount,
            'deposit_paid' => (float) $invoice->deposit_paid,
            'remaining_amount' => (float) $invoice->remaining_amount,
            'invoice_date' => $invoice->invoice_date?->format('Y-m-d'),
            'status' => $invoice->status,
            'customer_name' => $invoice->customer->name ?? $invoice->reservation->customer->name ?? null,
            'customer_phone' => $invoice->customer->phone ?? $invoice->reservation->customer->phone ?? null,
            'reservation_id' => $invoice->reservation_id,
            'public_url' => $invoice->public_url,
            'created_at' => $invoice->created_at?->toISOString(),
        ];
    }
}

