<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Transaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ReportController extends Controller
{
    public function monthlyAppointments(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $month = $request->get('month', Carbon::now()->format('Y-m'));
        $startDate = Carbon::parse($month)->startOfMonth();
        $endDate = Carbon::parse($month)->endOfMonth();
        
        $query = Reservation::whereBetween('date', [$startDate, $endDate]);
        
        if ($user->role === 'personel') {
            $query->where('personel_id', $user->id);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }
        
        $appointments = $query->get();
        
        $byStatus = $appointments->groupBy('status')->map(function ($group) {
            return $group->count();
        });
        
        $byDay = $appointments->groupBy(function ($apt) {
            return Carbon::parse($apt->date)->format('Y-m-d');
        })->map(function ($group) {
            return $group->count();
        });
        
        return response()->json([
            'success' => true,
            'data' => [
                'month' => $month,
                'total' => $appointments->count(),
                'by_status' => $byStatus,
                'by_day' => $byDay,
            ],
        ]);
    }
    
    public function monthlyRevenue(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $month = $request->get('month', Carbon::now()->format('Y-m'));
        $startDate = Carbon::parse($month)->startOfMonth();
        $endDate = Carbon::parse($month)->endOfMonth();
        
        $query = Reservation::whereBetween('date', [$startDate, $endDate])
            ->where('status', 'tamamlandi');
        
        if ($user->role === 'personel') {
            $query->where('personel_id', $user->id);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }
        
        $totalRevenue = $query->sum('total_price') ?? 0;
        
        $byDay = $query->get()->groupBy(function ($apt) {
            return Carbon::parse($apt->date)->format('Y-m-d');
        })->map(function ($group) {
            return $group->sum('total_price');
        });
        
        return response()->json([
            'success' => true,
            'data' => [
                'month' => $month,
                'total_revenue' => (float) $totalRevenue,
                'by_day' => $byDay,
            ],
        ]);
    }
    
    public function weekly(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $weekStart = Carbon::now()->startOfWeek();
        $weekEnd = Carbon::now()->endOfWeek();
        
        $query = Reservation::whereBetween('date', [$weekStart, $weekEnd]);
        
        if ($user->role === 'personel') {
            $query->where('personel_id', $user->id);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }
        
        $appointments = $query->get();
        $revenue = (clone $query)->where('status', 'tamamlandi')->sum('total_price') ?? 0;
        
        $byDay = $appointments->groupBy(function ($apt) {
            return Carbon::parse($apt->date)->format('Y-m-d');
        })->map(function ($group) {
            return [
                'count' => $group->count(),
                'revenue' => $group->where('status', 'tamamlandi')->sum('total_price'),
            ];
        });
        
        return response()->json([
            'success' => true,
            'data' => [
                'week_start' => $weekStart->format('Y-m-d'),
                'week_end' => $weekEnd->format('Y-m-d'),
                'total_appointments' => $appointments->count(),
                'total_revenue' => (float) $revenue,
                'by_day' => $byDay,
            ],
        ]);
    }
}

