<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\AppointmentRequest;
use App\Models\PersonelLeave;
use App\Models\Reservation;
use App\Models\Service;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Throwable;

class StaffController extends Controller
{
    private function resolveSalonId(User $user): int
    {
        return $user->beauty_salon_id ?? $user->id;
    }

    private function toMinutes(?string $time): int
    {
        if (!$time) {
            return 0;
        }
        [$h, $m] = array_pad(explode(':', $time), 2, 0);
        return ((int) $h * 60) + (int) $m;
    }

    private function hasOverlap(array $a, array $b): bool
    {
        $aStart = $this->toMinutes($a['start_time'] ?? null);
        $aEnd = $this->toMinutes($a['end_time'] ?? null);
        $bStart = $this->toMinutes($b['start_time'] ?? null);
        $bEnd = $this->toMinutes($b['end_time'] ?? null);

        if ($aEnd <= $aStart) {
            $aEnd = $aStart + 30;
        }
        if ($bEnd <= $bStart) {
            $bEnd = $bStart + 30;
        }

        return $aStart < $bEnd && $bStart < $aEnd;
    }

    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        // Personel personel listesini göremez
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Personel listesine erişim yetkiniz yok',
            ], 403);
        }
        
        $query = User::where('role', 'personel');
        
        if (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }
        
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }
        
        $staff = $query->orderBy('name', 'asc')
            ->with('services:id')
            ->paginate($request->get('per_page', 20));
        
        return response()->json([
            'success' => true,
            'data' => collect($staff->items())->map(fn($personel) => $this->formatStaff($personel))->values(),
            'meta' => [
                'current_page' => $staff->currentPage(),
                'last_page' => $staff->lastPage(),
                'per_page' => $staff->perPage(),
                'total' => $staff->total(),
            ],
        ]);
    }
    
    public function show(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        // Personel personel detayını göremez
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Personel detayına erişim yetkiniz yok',
            ], 403);
        }
        
        $personel = User::where('role', 'personel')
            ->with('services:id')
            ->find($id);
        
        if (!$personel) {
            return response()->json([
                'success' => false,
                'message' => 'Personel bulunamadı',
            ], 404);
        }
        
        // Salon kontrolü
        if (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            $salonId = $user->beauty_salon_id ?? $user->id;
            if ($personel->beauty_salon_id !== $salonId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu personele erişim yetkiniz yok',
                ], 403);
            }
        }
        
        return response()->json([
            'success' => true,
            'data' => $this->formatStaff($personel, true),
        ]);
    }
    
    public function store(Request $request): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Personel oluşturma yetkiniz yok',
            ], 403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'service_ids' => 'required|array|min:1',
            'service_ids.*' => 'integer|exists:services,id',
        ]);
        
        $salonId = $user->beauty_salon_id ?? $user->id;
        
        $personel = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
            'password' => \Hash::make($validated['password'] ?? 'personel123'),
            'role' => 'personel',
            'beauty_salon_id' => $salonId,
            'is_personel_admin' => false,
        ]);

        $serviceIds = collect($validated['service_ids'] ?? [])->unique()->values();
        $validServiceIds = Service::where('beauty_salon_id', $salonId)
            ->whereIn('id', $serviceIds)
            ->pluck('id')
            ->all();
        $personel->services()->sync($validServiceIds);
        
        return response()->json([
            'success' => true,
            'message' => 'Personel başarıyla oluşturuldu',
            'data' => $this->formatStaff($personel->fresh()->load('services:id')),
        ], 201);
    }
    
    public function update(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Personel güncelleme yetkiniz yok',
            ], 403);
        }

        $salonId = $this->resolveSalonId($user);
        $personel = User::where('role', 'personel')
            ->where('beauty_salon_id', $salonId)
            ->find($id);

        if (!$personel) {
            return response()->json([
                'success' => false,
                'message' => 'Personel bulunamadı',
            ], 404);
        }

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:users,email,' . $personel->id,
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'service_ids' => 'nullable|array|min:1',
            'service_ids.*' => 'integer|exists:services,id',
        ]);

        $updateData = [];
        if (array_key_exists('name', $validated)) {
            $updateData['name'] = $validated['name'];
        }
        if (array_key_exists('email', $validated)) {
            $updateData['email'] = $validated['email'];
        }
        if (array_key_exists('phone', $validated)) {
            $updateData['phone'] = $validated['phone'];
        }
        if (!empty($validated['password'])) {
            $updateData['password'] = \Hash::make($validated['password']);
        }
        $updateData['is_personel_admin'] = false;

        if (!empty($updateData)) {
            $personel->update($updateData);
        }

        if (array_key_exists('service_ids', $validated)) {
            $serviceIds = collect($validated['service_ids'] ?? [])->unique()->values();
            $validServiceIds = Service::where('beauty_salon_id', $salonId)
                ->whereIn('id', $serviceIds)
                ->pluck('id')
                ->all();
            $personel->services()->sync($validServiceIds);
        }

        return response()->json([
            'success' => true,
            'message' => 'Personel başarıyla güncellendi',
            'data' => $this->formatStaff($personel->fresh()->load('services:id')),
        ]);
    }
    
    public function destroy(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Personel silme yetkiniz yok',
            ], 403);
        }

        $salonId = $this->resolveSalonId($user);
        $personel = User::where('role', 'personel')
            ->where('beauty_salon_id', $salonId)
            ->find($id);

        if (!$personel) {
            return response()->json([
                'success' => false,
                'message' => 'Personel bulunamadı',
            ], 404);
        }

        $personel->delete();

        return response()->json([
            'success' => true,
            'message' => 'Personel silindi',
        ]);
    }

    public function leavePreview(Request $request, int $id): JsonResponse
    {
        try {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'İzin yönetimine erişim yetkiniz yok',
            ], 403);
        }

        $validated = $request->validate([
            'date' => 'required|date',
        ]);

        $salonId = $this->resolveSalonId($user);
        $date = $validated['date'];

        $hasLeaveTable = Schema::hasTable('personel_leaves');
        if (!$hasLeaveTable) {
            return response()->json([
                'success' => false,
                'message' => 'İzin sistemi henüz hazır değil. Lütfen yöneticiye bildirin.',
                'code' => 'LEAVE_TABLE_MISSING',
            ], 422);
        }

        $personel = User::where('role', 'personel')
            ->where('beauty_salon_id', $salonId)
            ->find($id);

        if (!$personel) {
            return response()->json([
                'success' => false,
                'message' => 'Personel bulunamadı',
            ], 404);
        }

        $reservations = Reservation::where('beauty_salon_id', $salonId)
            ->where('personel_id', $personel->id)
            ->whereDate('date', $date)
            ->whereNotIn('status', ['cancelled'])
            ->with(['customer:id,name', 'appointmentRequest', 'appointmentRequest.service:id,name'])
            ->orderBy('start_time')
            ->get();

        $appointments = $reservations->map(function ($reservation) {
            $serviceNames = [];
            if ($reservation->appointmentRequest) {
                $req = $reservation->appointmentRequest;
                if (is_array($req->service_ids) && !empty($req->service_ids)) {
                    $serviceNames = Service::whereIn('id', $req->service_ids)->pluck('name')->toArray();
                } elseif ($req->service) {
                    $serviceNames = [$req->service->name];
                }
            }

            return [
                'reservation_id' => $reservation->id,
                'start_time' => $reservation->start_time,
                'end_time' => $reservation->end_time,
                'customer_name' => optional($reservation->customer)->name ?: '-',
                'service_names' => $serviceNames,
            ];
        })->values();

        $leavePersonelIds = PersonelLeave::where('beauty_salon_id', $salonId)
            ->whereDate('leave_date', $date)
            ->pluck('personel_id')
            ->all();

        $otherPersonels = User::where('beauty_salon_id', $salonId)
            ->where('role', 'personel')
            ->where('id', '!=', $personel->id)
            ->orderBy('name')
            ->get(['id', 'name'])
            ->map(function ($p) use ($salonId, $date, $leavePersonelIds) {
                $activeCount = Reservation::where('beauty_salon_id', $salonId)
                    ->where('personel_id', $p->id)
                    ->whereDate('date', $date)
                    ->whereNotIn('status', ['cancelled'])
                    ->count();

                return [
                    'id' => $p->id,
                    'name' => $p->name,
                    'on_leave' => in_array($p->id, $leavePersonelIds, true),
                    'active_appointment_count' => $activeCount,
                ];
            })
            ->values();

        $existingLeave = PersonelLeave::with('transferPersonel:id,name')
            ->where('personel_id', $personel->id)
            ->whereDate('leave_date', $date)
            ->first();

        return response()->json([
            'success' => true,
            'data' => [
                'personel' => ['id' => $personel->id, 'name' => $personel->name],
                'appointments' => $appointments,
                'other_personels' => $otherPersonels,
                'already_on_leave' => (bool) $existingLeave,
                'existing_leave' => $existingLeave ? [
                    'id' => $existingLeave->id,
                    'note' => $existingLeave->note,
                    'transfer_to_personel_id' => $existingLeave->transfer_to_personel_id,
                    'transfer_to_personel_name' => optional($existingLeave->transferPersonel)->name,
                ] : null,
            ],
        ]);
        } catch (Throwable $e) {
            report($e);
            return response()->json([
                'success' => false,
                'message' => 'İzin bilgileri alınırken bir hata oluştu.',
                'code' => 'LEAVE_PREVIEW_ERROR',
            ], 500);
        }
    }

    public function setLeave(Request $request, int $id): JsonResponse
    {
        try {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'İzin yönetimine erişim yetkiniz yok',
            ], 403);
        }

        $validated = $request->validate([
            'leave_date' => 'required|date',
            'note' => 'nullable|string|max:500',
            'transfer_to_personel_id' => 'nullable|integer|exists:users,id',
            'transfer_reservation_ids' => 'nullable|array',
            'transfer_reservation_ids.*' => 'integer|exists:reservations,id',
        ]);

        $salonId = $this->resolveSalonId($user);
        if (!Schema::hasTable('personel_leaves')) {
            return response()->json([
                'success' => false,
                'message' => 'İzin sistemi henüz hazır değil. Lütfen yöneticiye bildirin.',
                'code' => 'LEAVE_TABLE_MISSING',
            ], 422);
        }

        $personel = User::where('role', 'personel')
            ->where('beauty_salon_id', $salonId)
            ->find($id);

        if (!$personel) {
            return response()->json([
                'success' => false,
                'message' => 'Personel bulunamadı',
            ], 404);
        }

        $leaveDate = $validated['leave_date'];
        $targetPersonelId = $validated['transfer_to_personel_id'] ?? null;
        $transferReservationIds = collect($validated['transfer_reservation_ids'] ?? [])
            ->filter()
            ->map(fn ($i) => (int) $i)
            ->unique()
            ->values();

        if ($transferReservationIds->isNotEmpty() && !$targetPersonelId) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu aktarımı için hedef personel seçmelisiniz.',
            ], 422);
        }

        $targetPersonel = null;
        if ($targetPersonelId) {
            $targetPersonel = User::where('role', 'personel')
                ->where('beauty_salon_id', $salonId)
                ->where('id', '!=', $personel->id)
                ->find($targetPersonelId);

            if (!$targetPersonel) {
                return response()->json([
                    'success' => false,
                    'message' => 'Hedef personel bulunamadı.',
                ], 422);
            }

            $targetOnLeave = PersonelLeave::where('beauty_salon_id', $salonId)
                ->where('personel_id', $targetPersonel->id)
                ->whereDate('leave_date', $leaveDate)
                ->exists();

            if ($targetOnLeave) {
                return response()->json([
                    'success' => false,
                    'message' => 'İzinli personele randevu aktarımı yapılamaz.',
                ], 422);
            }
        }

        $activeReservations = Reservation::where('beauty_salon_id', $salonId)
            ->where('personel_id', $personel->id)
            ->whereDate('date', $leaveDate)
            ->whereNotIn('status', ['cancelled'])
            ->get(['id', 'start_time', 'end_time']);

        if ($activeReservations->isNotEmpty() && $transferReservationIds->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Bu tarihte aktif randevular var. Önce randevuları aktarın.',
            ], 422);
        }

        if ($activeReservations->isNotEmpty()) {
            $activeIds = $activeReservations->pluck('id')->map(fn ($v) => (int) $v)->sort()->values();
            $selectedIds = $transferReservationIds->sort()->values();
            if ($selectedIds->count() !== $activeIds->count() || $selectedIds->diff($activeIds)->isNotEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'İzin için o günkü tüm aktif randevuları aktarmalısınız.',
                ], 422);
            }
        }

        if ($targetPersonel && $transferReservationIds->isNotEmpty()) {
            $sourceReservations = $activeReservations->whereIn('id', $transferReservationIds->all())->values();
            $targetReservations = Reservation::where('beauty_salon_id', $salonId)
                ->where('personel_id', $targetPersonel->id)
                ->whereDate('date', $leaveDate)
                ->whereNotIn('status', ['cancelled'])
                ->get(['id', 'start_time', 'end_time']);

            foreach ($sourceReservations as $src) {
                foreach ($targetReservations as $dst) {
                    if ($this->hasOverlap($src->toArray(), $dst->toArray())) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Hedef personelde saat çakışması var.',
                        ], 422);
                    }
                }
            }
        }

        $leaveExisted = PersonelLeave::where('personel_id', $personel->id)
            ->whereDate('leave_date', $leaveDate)
            ->exists();

        DB::transaction(function () use ($salonId, $user, $personel, $targetPersonel, $targetPersonelId, $leaveDate, $validated, $transferReservationIds) {
            $leave = PersonelLeave::firstOrNew([
                'personel_id' => $personel->id,
                'leave_date' => $leaveDate,
            ]);
            $leave->beauty_salon_id = $salonId;
            $leave->note = $validated['note'] ?? null;
            $leave->created_by = $user->id;
            $leave->transfer_to_personel_id = $targetPersonelId;
            $leave->save();

            if ($targetPersonel && $transferReservationIds->isNotEmpty()) {
                $reservations = Reservation::where('beauty_salon_id', $salonId)
                    ->where('personel_id', $personel->id)
                    ->whereDate('date', $leaveDate)
                    ->whereIn('id', $transferReservationIds->all())
                    ->get();

                foreach ($reservations as $reservation) {
                    $reservation->personel_id = $targetPersonel->id;
                    $reservation->save();
                }

                AppointmentRequest::whereIn('reservation_id', $reservations->pluck('id')->all())
                    ->update(['personel_id' => $targetPersonel->id]);
            }
        });

        $message = $leaveExisted ? 'Personel izni güncellendi.' : 'Personel izni kaydedildi.';
        if ($transferReservationIds->isNotEmpty()) {
            $message = $leaveExisted
                ? 'Personel izni güncellendi. Randevular aktarıldı.'
                : 'Personel izni kaydedildi. Randevular aktarıldı.';
        }

        return response()->json([
            'success' => true,
            'message' => $message,
        ]);
        } catch (Throwable $e) {
            report($e);
            return response()->json([
                'success' => false,
                'message' => 'İzin kaydedilirken bir hata oluştu.',
                'code' => 'LEAVE_SAVE_ERROR',
            ], 500);
        }
    }

    public function deleteLeave(Request $request, int $id): JsonResponse
    {
        try {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'İzin yönetimine erişim yetkiniz yok',
            ], 403);
        }

        $validated = $request->validate([
            'leave_date' => 'required|date',
        ]);

        $salonId = $this->resolveSalonId($user);
        if (!Schema::hasTable('personel_leaves')) {
            return response()->json([
                'success' => false,
                'message' => 'İzin sistemi henüz hazır değil. Lütfen yöneticiye bildirin.',
                'code' => 'LEAVE_TABLE_MISSING',
            ], 422);
        }

        $personel = User::where('role', 'personel')
            ->where('beauty_salon_id', $salonId)
            ->find($id);

        if (!$personel) {
            return response()->json([
                'success' => false,
                'message' => 'Personel bulunamadı',
            ], 404);
        }

        $leave = PersonelLeave::where('personel_id', $personel->id)
            ->whereDate('leave_date', $validated['leave_date'])
            ->first();

        if (!$leave) {
            return response()->json([
                'success' => false,
                'message' => 'İzin kaydı bulunamadı.',
            ], 404);
        }

        $leave->delete();

        return response()->json([
            'success' => true,
            'message' => 'İzin iptal edildi.',
        ]);
        } catch (Throwable $e) {
            report($e);
            return response()->json([
                'success' => false,
                'message' => 'İzin iptal edilirken bir hata oluştu.',
                'code' => 'LEAVE_DELETE_ERROR',
            ], 500);
        }
    }
    
    private function formatStaff(User $personel, $detailed = false): array
    {
        $data = [
            'id' => $personel->id,
            'name' => $personel->name,
            'email' => $personel->email,
            'phone' => $personel->phone,
            'is_personel_admin' => $personel->is_personel_admin ?? false,
            'service_ids' => $personel->services->pluck('id')->values()->all(),
            'avatar' => $personel->profile_image ? asset('storage/' . $personel->profile_image) : null,
        ];
        
        if ($detailed) {
            $data['is_active'] = $personel->is_active ?? true;
            $data['created_at'] = $personel->created_at?->toISOString();
        }
        
        return $data;
    }
}

