<?php

namespace App\Console\Commands;

use App\Models\Reservation;
use App\Services\NotificationService;
use Illuminate\Console\Command;
use Carbon\Carbon;

class SendAppointmentReminders extends Command
{
    protected $signature = 'appointments:send-reminders';
    protected $description = 'Send appointment reminders (24h and 2h before)';

    public function handle(NotificationService $notificationService): int
    {
        $now = Carbon::now();

        $target24h = $now->copy()->addHours(24);
        $this->sendRemindersForTime($target24h, 24, $notificationService);

        $target2h = $now->copy()->addHours(2);
        $this->sendRemindersForTime($target2h, 2, $notificationService);

        return Command::SUCCESS;
    }

    private function sendRemindersForTime(Carbon $targetTime, int $hoursBefore, NotificationService $notificationService): void
    {
        $startOfHour = $targetTime->copy()->startOfHour();
        $endOfHour = $targetTime->copy()->endOfHour();

        $reservations = Reservation::where('status', 'onaylandi')
            ->whereDate('date', $targetTime->toDateString())
            ->whereBetween('start_time', [
                $startOfHour->format('H:i:s'),
                $endOfHour->format('H:i:s')
            ])
            ->with(['customer', 'service'])
            ->get();

        foreach ($reservations as $reservation) {
            
            $logKey = "reminder_{$reservation->id}_{$hoursBefore}h";
            if (cache()->has($logKey)) {
                continue;
            }

            $notificationService->sendAppointmentReminder($reservation, $hoursBefore);

            cache()->put($logKey, true, now()->addHours(2));
        }
    }
}

