<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Reservation;
use App\Services\SmsService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;

class SendRecurringAppointmentReminders extends Command
{
    
    protected $signature = 'appointments:send-reminders';

    protected $description = 'Randevu hatırlatma SMS gönder (24 saat ve 3 saat önce)';

    public function handle()
    {
        $this->info('Randevu hatırlatma SMS gönderimi başlatıldı (24s ve 3s)...');

        [$sent24, $failed24, $total24] = $this->sendByHoursBefore(24);
        [$sent3, $failed3, $total3] = $this->sendByHoursBefore(3);

        $this->info("\n📊 Özet:");
        $this->info("24 saat kala -> Toplam: {$total24} | ✅ Gönderilen: {$sent24} | ❌ Başarısız: {$failed24}");
        $this->info("3 saat kala  -> Toplam: {$total3} | ✅ Gönderilen: {$sent3} | ❌ Başarısız: {$failed3}");

        return Command::SUCCESS;
    }

    private function sendByHoursBefore(int $hoursBefore): array
    {
        $target = Carbon::now()->addHours($hoursBefore);
        $startOfHour = $target->copy()->startOfHour();
        $endOfHour = $target->copy()->endOfHour();

        $reservations = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'appointmentRequest'])
            ->whereDate('date', $target->format('Y-m-d'))
            ->whereBetween('start_time', [
                $startOfHour->format('H:i:s'),
                $endOfHour->format('H:i:s')
            ])
            ->whereNotIn('status', ['cancelled', 'completed', 'iptal', 'tamamlandi'])
            ->whereNotNull('customer_id')
            ->get();

        $sentCount = 0;
        $failedCount = 0;
        $smsService = app(SmsService::class);

        foreach ($reservations as $reservation) {
            if (!$reservation->customer || !$reservation->customer->phone) {
                continue;
            }

            $dedupeKey = "sms_reminder_{$reservation->id}_{$hoursBefore}h";
            if (Cache::has($dedupeKey)) {
                continue;
            }

            try {
                $user = \App\Models\User::find($reservation->beauty_salon_id);
                if (!$user) {
                    continue;
                }

                $salonName = $user->salon_name ?? $user->name;
                $formattedDate = Carbon::parse($reservation->date)->format('d/m/Y');
                $formattedTime = Carbon::parse($reservation->start_time)->format('H:i');

                $serviceName = 'Hizmet';
                if ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
                    $serviceName = $reservation->appointmentRequest->service->name;
                }

                $message = "Merhaba {$reservation->customer->name}, ";
                $message .= "{$formattedDate} tarihinde saat {$formattedTime} için {$serviceName} randevunuza {$hoursBefore} saat kaldı. ";
                $message .= "Salon: {$salonName}. Lütfen randevu saatinizden 5 dakika önce salonda olun.";

                $result = $smsService->send(
                    $reservation->customer->phone,
                    $message,
                    null,
                    [
                        'purpose' => 'appointment_reminder',
                        'user_id' => $user->id
                    ]
                );

                if ($result && $result->success) {
                    $sentCount++;
                    Cache::put($dedupeKey, true, now()->addHours(8));
                    $this->info("✅ [{$hoursBefore}s] {$reservation->customer->name} - SMS gönderildi");
                } else {
                    $failedCount++;
                    $this->error("❌ [{$hoursBefore}s] {$reservation->customer->name} - SMS gönderilemedi: " . ($result->error ?? 'Bilinmeyen hata'));
                }
            } catch (\Exception $e) {
                $failedCount++;
                \Log::error('SMS hatırlatma mesajı gönderme hatası', [
                    'reservation_id' => $reservation->id,
                    'customer_id' => $reservation->customer_id,
                    'hours_before' => $hoursBefore,
                    'error' => $e->getMessage(),
                ]);
                $this->error("❌ [{$hoursBefore}s] {$reservation->customer->name} - Hata: {$e->getMessage()}");
            }
        }

        return [$sentCount, $failedCount, $reservations->count()];
    }
}
