<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\SmsService;
use App\Services\Sms\OtpService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class AccountController extends Controller
{
    protected $smsService;
    protected $otpService;

    public function __construct(SmsService $smsService, OtpService $otpService)
    {
        $this->smsService = $smsService;
        $this->otpService = $otpService;
    }

    public function index()
    {
        $user = Auth::user();
        return view('admin.account.index', compact('user'));
    }

    public function sendDeleteOtp(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil. Hesap silme işlemi için telefon numaranızı güncelleyin.',
            ], 400);
        }

        $result = $this->smsService->sendOtp($user->phone, 'account_deletion');

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Doğrulama kodu telefon numaranıza gönderildi.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
        ], 400);
    }

    public function destroy(Request $request)
    {
        $request->validate([
            'otp_code' => 'required|string|size:6',
            'password' => 'required|string',
        ], [
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
            'password.required' => 'Şifre zorunludur.',
        ]);

        $user = Auth::user();

        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors(['password' => 'Şifre hatalı.'])->withInput();
        }

        $otpCode = (int) $request->otp_code;
        if (!$this->otpService->verify($user->phone, $otpCode, 'account_deletion')) {
            return back()->withErrors(['otp_code' => 'Doğrulama kodu hatalı veya süresi dolmuş.'])->withInput();
        }

        try {
            DB::beginTransaction();

            $userId = $user->id;

            // Token'ları sil (Sanctum - foreign key sorunlarını önler)
            $user->tokens()->delete();

            // Kullanıcıyı kalıcı sil (DB cascade ilişkili kayıtları siler)
            $deleted = $user->delete();
            if (!$deleted) {
                throw new \Exception('Kullanıcı silinemedi.');
            }

            DB::commit();

            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();

            return redirect()->route('home')->with('success', 'Hesabınız başarıyla silindi. Üzgünüz ki ayrılıyoruz...');
            
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            \Log::error('Hesap silme DB hatası: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'user_id' => $user->id ?? null,
            ]);
            return back()->withErrors(['error' => 'Hesap silinirken veritabanı hatası. İlişkili kayıtlar silinemiyor olabilir.'])->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Hesap silme hatası: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'user_id' => $user->id ?? null,
            ]);
            return back()->withErrors(['error' => 'Hesap silinirken bir hata oluştu: ' . $e->getMessage()])->withInput();
        }
    }

    public function update(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string',
            'password' => 'nullable|min:8|confirmed',
        ]);

        if ($request->filled('password')) {
            $validated['password'] = Hash::make($validated['password']);
        } else {
            unset($validated['password']);
        }

        $user->update($validated);

        return redirect()->route('admin.account.index')
            ->with('success', 'Profil bilgileri güncellendi.');
    }

    public function sendPasswordChangeOtp(Request $request)
    {
        $user = Auth::user();

        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil. Lütfen önce telefon numaranızı güncelleyin.',
            ], 400);
        }

        $result = $this->smsService->sendOtp($user->phone, 'password_change');

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Doğrulama kodu telefon numaranıza gönderildi.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
        ], 400);
    }

    public function changePassword(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'otp_code' => 'required|string|size:6',
            'password' => 'required|string|min:8|confirmed',
        ], [
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
            'password.required' => 'Yeni şifre zorunludur.',
            'password.min' => 'Şifre en az 8 karakter olmalıdır.',
            'password.confirmed' => 'Şifre tekrarı eşleşmiyor.',
        ]);

        $otpCode = (int) $validated['otp_code'];
        if (!$this->otpService->verify($user->phone, $otpCode, 'password_change')) {
            return back()->withErrors(['otp_code' => 'Doğrulama kodu hatalı veya süresi dolmuş. Lütfen yeni kod alın.'])->withInput();
        }

        $user->update(['password' => Hash::make($validated['password'])]);

        return redirect()->route('admin.account.index')
            ->with('success', 'Şifreniz başarıyla değiştirildi.');
    }

    public function sendPhoneChangeOtp(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'new_phone' => 'required|string',
        ], [
            'new_phone.required' => 'Yeni telefon numarası zorunludur.',
        ]);

        $newPhone = preg_replace('/[^0-9]/', '', $validated['new_phone']);
        if (strlen($newPhone) === 10) {
            $newPhone = '0' . $newPhone;
        } elseif (strlen($newPhone) === 11 && str_starts_with($newPhone, '0')) {
            // ok
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Geçerli bir Türkiye telefon numarası giriniz (05XX XXX XX XX).',
            ], 400);
        }

        if ($newPhone === preg_replace('/[^0-9]/', '', $user->phone ?? '')) {
            return response()->json([
                'success' => false,
                'message' => 'Yeni numara mevcut numaranızla aynı olamaz.',
            ], 400);
        }

        $exists = \App\Models\User::where('phone', $newPhone)->where('id', '!=', $user->id)->exists();
        if ($exists) {
            return response()->json([
                'success' => false,
                'message' => 'Bu telefon numarası zaten kullanımda.',
            ], 400);
        }

        $result = $this->smsService->sendOtp($newPhone, 'phone_change');

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Doğrulama kodu yeni telefon numaranıza gönderildi.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
        ], 400);
    }

    public function changePhone(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'new_phone' => 'required|string',
            'otp_code' => 'required|string|size:6',
            'password' => 'required|string',
        ], [
            'new_phone.required' => 'Yeni telefon numarası zorunludur.',
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
            'password.required' => 'Mevcut şifre zorunludur.',
        ]);

        if (!Hash::check($validated['password'], $user->password)) {
            return back()->withErrors(['password' => 'Şifre hatalı.'])->withInput();
        }

        $newPhone = preg_replace('/[^0-9]/', '', $validated['new_phone']);
        if (strlen($newPhone) === 10) {
            $newPhone = '0' . $newPhone;
        }

        $otpCode = (int) $validated['otp_code'];
        if (!$this->otpService->verify($newPhone, $otpCode, 'phone_change')) {
            return back()->withErrors(['otp_code' => 'Doğrulama kodu hatalı veya süresi dolmuş. Lütfen yeni kod alın.'])->withInput();
        }

        $exists = \App\Models\User::where('phone', $newPhone)->where('id', '!=', $user->id)->exists();
        if ($exists) {
            return back()->withErrors(['new_phone' => 'Bu telefon numarası zaten kullanımda.'])->withInput();
        }

        $user->update(['phone' => $newPhone]);

        return redirect()->route('admin.account.index')
            ->with('success', 'Telefon numaranız başarıyla güncellendi.');
    }
}

