<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\Reservation;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AccountingController extends Controller
{
    public function income(Request $request)
    {
        $user = auth()->user();
        $query = Transaction::where('type', 'income')
            ->with(['account', 'reservation']);

        if ($user->role === 'randevucu') {
            $query->where('created_by', $user->id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }
        if ($request->filled('account_id')) {
            $query->where('account_id', $request->account_id);
        }

        $transactions = $query->orderBy('date', 'desc')->paginate(20);
        $accounts = Account::where('is_active', true)->get();
        
        $totalQuery = Transaction::where('type', 'income');
        if ($user->role === 'randevucu') {
            $totalQuery->where('created_by', $user->id);
        }
        $total = $totalQuery->sum('amount');

        return view('admin.accounting.income', compact('transactions', 'accounts', 'total'));
    }

    public function expense(Request $request)
    {
        $user = auth()->user();
        $query = Transaction::where('type', 'expense')
            ->with(['account']);

        if ($user->role === 'randevucu') {
            $query->where('created_by', $user->id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }
        if ($request->filled('account_id')) {
            $query->where('account_id', $request->account_id);
        }

        $transactions = $query->orderBy('date', 'desc')->paginate(20);
        $accounts = Account::where('is_active', true)->get();
        
        $totalQuery = Transaction::where('type', 'expense');
        if ($user->role === 'randevucu') {
            $totalQuery->where('created_by', $user->id);
        }
        $total = $totalQuery->sum('amount');

        return view('admin.accounting.expense', compact('transactions', 'accounts', 'total'));
    }

    public function storeIncome(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'description' => 'nullable|string',
            'reservation_id' => 'nullable|exists:appointments,id',
        ]);

        $cashAccount = Account::where('type', 'cash')->where('is_active', true)->first();
        if (!$cashAccount) {
            $cashAccount = Account::create([
                'name' => 'Ana Kasa',
                'type' => 'cash',
                'balance' => 0,
                'is_active' => true,
            ]);
        }

        $validated['account_id'] = $cashAccount->id;
        $validated['type'] = 'income';
        $validated['created_by'] = Auth::id();

        Transaction::create($validated);

        $cashAccount->increment('balance', $validated['amount']);

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'create_income',
            'description' => "Gelir eklendi: {$validated['amount']} TL",
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('admin.accounting.income')
            ->with('success', 'Gelir başarıyla eklendi.');
    }

    public function storeExpense(Request $request)
    {
        $validated = $request->validate([
            'account_id' => 'required|exists:accounts,id',
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'description' => 'nullable|string',
        ]);

        $validated['type'] = 'expense';
        $validated['created_by'] = Auth::id();

        Transaction::create($validated);

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'create_expense',
            'description' => "Gider eklendi: {$validated['amount']} TL",
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('admin.accounting.expense')
            ->with('success', 'Gider başarıyla eklendi.');
    }

    public function reservationIncome()
    {
        $user = auth()->user();
        $query = Reservation::where('status', '!=', 'cancelled')
            ->with(['customer', 'salon']);

        if ($user->role === 'randevucu') {
            $query->where('created_by', $user->id);
        }
        
        $appointments = $query->orderBy('date', 'desc')->paginate(20);

        return view('admin.accounting.reservation-income', compact('appointments'));
    }

    public function deposits()
    {
        $deposits = \App\Models\Deposit::with('reservation')
            ->orderBy('due_date', 'asc')
            ->paginate(20);

        return view('admin.accounting.deposits', compact('deposits'));
    }

    public function balances()
    {
        $user = auth()->user();
        $query = Reservation::where('status', '!=', 'cancelled')
            ->where('remaining_amount', '>', 0)
            ->with(['customer', 'salon']);

        if ($user->role === 'randevucu') {
            $query->where('created_by', $user->id);
        }
        
        $appointments = $query->orderBy('date', 'asc')->paginate(20);

        return view('admin.accounting.balances', compact('appointments'));
    }
}
