<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Appointment;
use App\Models\Customer;
use App\Models\User;
use Illuminate\Http\Request;

class AppointmentController extends Controller
{
    public function index(Request $request)
    {
        $query = Appointment::with(['customer', 'assignedUser']);

        if ($request->filled('date')) {
            $query->whereDate('date', $request->date);
        }

        $appointments = $query->orderBy('date')->orderBy('start_time')->paginate(20);

        return view('admin.appointments.index', compact('appointments'));
    }

    public function create()
    {
        $customers = Customer::orderBy('name')->get();
        $users = User::where('is_active', true)->get();

        return view('admin.appointments.create', compact('customers', 'users'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'date' => 'required|date',
            'start_time' => 'required',
            'end_time' => 'required',
            'subject' => 'required|string|max:255',
            'notes' => 'nullable|string',
            'status' => 'required|in:pending,approved,cancelled',
            'assigned_user_id' => 'nullable|exists:users,id',
            'send_sms' => 'nullable|boolean',
        ]);

        $appointment = Appointment::create($validated);

        $smsSent = false;
        $smsBalanceLow = false;
        if ($request->input('send_sms') == '1' && $appointment->customer && $appointment->customer->phone) {
            $user = \Illuminate\Support\Facades\Auth::user();

            if ($user && !$user->isSuperAdmin()) {
                if (!$user->hasSmsBalance(1)) {
                    $smsBalanceLow = true;
                } else {
                    try {
                        $smsService = app(\App\Services\SmsService::class);

                        $formattedDate = \Carbon\Carbon::parse($appointment->date)->format('d.m.Y');
                        $timeRange = $appointment->start_time . ' - ' . $appointment->end_time;
                        $message = "Sayın {$appointment->customer->name}, {$formattedDate} tarihinde {$timeRange} saatleri arasında randevunuz oluşturulmuştur. Konu: {$appointment->subject}. salonay.com";
                        
                        $result = $smsService->send(
                            $appointment->customer->phone,
                            $message,
                            null,
                            ['purpose' => 'appointment', 'user_id' => $user->id]
                        );
                        
                        if ($result && $result->success) {
                            $smsSent = true;
                            \Log::info('✅ Randevu oluşturulduğunda SMS gönderildi', [
                                'appointment_id' => $appointment->id,
                                'phone' => $appointment->customer->phone,
                            ]);
                        }
                    } catch (\Exception $e) {
                        \Log::error('❌ Randevu SMS gönderme hatası: ' . $e->getMessage(), [
                            'appointment_id' => $appointment->id,
                            'trace' => $e->getTraceAsString()
                        ]);
                    }
                }
            } else {
                
                try {
                    $smsService = app(\App\Services\SmsService::class);
                    
                    $formattedDate = \Carbon\Carbon::parse($appointment->date)->format('d.m.Y');
                    $timeRange = $appointment->start_time . ' - ' . $appointment->end_time;
                    $message = "Sayın {$appointment->customer->name}, {$formattedDate} tarihinde {$timeRange} saatleri arasında randevunuz oluşturulmuştur. Konu: {$appointment->subject}. salonay.com";
                    
                    $result = $smsService->send(
                        $appointment->customer->phone,
                        $message,
                        null,
                        ['purpose' => 'appointment']
                    );
                    
                    if ($result && $result->success) {
                        $smsSent = true;
                    }
                } catch (\Exception $e) {
                    \Log::error('❌ Randevu SMS gönderme hatası: ' . $e->getMessage());
                }
            }
        }

        if (request()->ajax() || request()->wantsJson()) {
            $response = [
                'success' => true,
                'message' => 'Randevu başarıyla oluşturuldu.',
                'appointment' => $appointment
            ];
            
            if ($smsBalanceLow) {
                $response['sms_warning'] = 'SMS bakiyeniz kalmamıştır. SMS gönderilemedi.';
            } elseif ($smsSent) {
                $response['sms_sent'] = true;
            }
            
            return response()->json($response);
        }

        $redirect = redirect()->route('admin.appointments.index')
            ->with('success', 'Randevu oluşturuldu.');
            
        if ($smsBalanceLow) {
            $redirect->with('sms_warning', 'SMS bakiyeniz kalmamıştır. SMS gönderilemedi.');
        } elseif ($smsSent) {
            $redirect->with('info', 'Randevu oluşturuldu ve SMS gönderildi.');
        }
        
        return $redirect;
    }

    public function edit($id)
    {
        $appointment = Appointment::findOrFail($id);
        $customers = Customer::orderBy('name')->get();
        $users = User::where('is_active', true)->get();

        return view('admin.appointments.edit', compact('appointment', 'customers', 'users'));
    }

    public function update(Request $request, $id)
    {
        $appointment = Appointment::findOrFail($id);

        $validated = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'date' => 'required|date',
            'start_time' => 'required',
            'end_time' => 'required',
            'subject' => 'required|string|max:255',
            'notes' => 'nullable|string',
            'status' => 'required|in:pending,approved,cancelled',
            'assigned_user_id' => 'nullable|exists:users,id',
        ]);

        $appointment->update($validated);

        return redirect()->route('admin.appointments.index')
            ->with('success', 'Randevu güncellendi.');
    }

    public function destroy($id)
    {
        $appointment = Appointment::findOrFail($id);
        $appointment->delete();

        return redirect()->route('admin.appointments.index')
            ->with('success', 'Randevu silindi.');
    }
}

