<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AppointmentRequest;
use App\Models\Customer;
use App\Models\Reservation;
use App\Models\Invoice;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Barryvdh\DomPDF\Facade\Pdf;

class AppointmentRequestManagementController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }

        $pendingRequests = AppointmentRequest::where('beauty_salon_id', $user->id)
            ->where('status', 'pending')
            ->with(['service', 'customer', 'personel', 'room'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        \Log::info('AppointmentRequestManagementController@index', [
            'user_id' => $user->id,
            'pending_count' => $pendingRequests->count(),
            'is_ajax' => $request->ajax(),
            'wants_json' => $request->wantsJson(),
            'accepts_json' => $request->accepts('application/json'),
            'accept_header' => $request->header('Accept'),
        ]);

        if ($request->ajax() || $request->wantsJson() || $request->accepts('application/json')) {
            $mappedRequests = $pendingRequests->map(function ($req) {
                return [
                    'id' => $req->id,
                    'name' => $req->name ?? '',
                    'surname' => $req->surname ?? '',
                    'phone' => $req->phone ?? '',
                    'service' => $req->service ? [
                        'id' => $req->service->id,
                        'name' => $req->service->name ?? '',
                        'price' => $req->service->price ?? 0,
                    ] : null,
                    'customer' => $req->customer ? [
                        'id' => $req->customer->id,
                        'name' => $req->customer->name ?? '',
                    ] : null,
                    'created_at' => $req->created_at ? $req->created_at->toISOString() : now()->toISOString(),
                    'appointment_date' => $req->appointment_date ? (is_string($req->appointment_date) ? $req->appointment_date : (is_object($req->appointment_date) ? $req->appointment_date->format('Y-m-d') : $req->appointment_date)) : null,
                    'appointment_time' => $req->appointment_time ? (is_string($req->appointment_time) ? substr($req->appointment_time, 0, 5) : (is_object($req->appointment_time) ? $req->appointment_time->format('H:i') : $req->appointment_time)) : null,
                    'personel_id' => $req->personel_id,
                    'room_id' => $req->room_id,
                    'notes' => $req->notes,
                ];
            });
            
            \Log::info('Returning JSON response', [
                'requests_count' => $mappedRequests->count(),
            ]);
            
            return response()->json([
                'success' => true,
                'requests' => $mappedRequests->values()->all(),
            ]);
        }

        return view('admin.appointment-requests.index', compact('pendingRequests'));
    }

    public function show($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $appointmentRequest = AppointmentRequest::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->with(['service', 'customer', 'personel', 'room', 'reservation'])
            ->firstOrFail();

        $personels = \App\Models\User::where('role', 'personel')
            ->where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get();

        $rooms = \App\Models\Room::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get();

        $customer = null;
        $customerReservations = collect();
        
        if ($appointmentRequest->customer_id) {
            $customer = Customer::where('id', $appointmentRequest->customer_id)
                ->where('beauty_salon_id', $user->id)
                ->first();
            
            if ($customer) {
                $customerReservations = Reservation::where('customer_id', $customer->id)
                    ->where('beauty_salon_id', $user->id)
                    ->with(['personel', 'room', 'appointmentRequest.service'])
                    ->orderBy('date', 'desc')
                    ->orderBy('start_time', 'desc')
                    ->get();
            }
        } else {
            
            $customer = Customer::where('phone', $appointmentRequest->phone)
                ->where('beauty_salon_id', $user->id)
                ->first();
            
            if ($customer) {
                $customerReservations = Reservation::where('customer_id', $customer->id)
                    ->where('beauty_salon_id', $user->id)
                    ->with(['personel', 'room', 'appointmentRequest.service'])
                    ->orderBy('date', 'desc')
                    ->orderBy('start_time', 'desc')
                    ->get();
            }
        }

        if (request()->wantsJson() && request()->accepts('application/json') && !request()->accepts('text/html')) {
            return response()->json([
                'request' => [
                    'id' => $appointmentRequest->id,
                    'name' => $appointmentRequest->name,
                    'surname' => $appointmentRequest->surname,
                    'phone' => $appointmentRequest->phone,
                    'age' => $appointmentRequest->age,
                    'service' => $appointmentRequest->service ? [
                        'id' => $appointmentRequest->service->id,
                        'name' => $appointmentRequest->service->name,
                        'price' => $appointmentRequest->service->price,
                        'duration' => $appointmentRequest->service->duration,
                    ] : null,
                    'customer' => $customer ? [
                        'id' => $customer->id,
                        'name' => $customer->name,
                    ] : null,
                    'created_at' => $appointmentRequest->created_at->toISOString(),
                    'appointment_date' => $appointmentRequest->appointment_date ? $appointmentRequest->appointment_date->format('Y-m-d') : null,
                    'appointment_time' => $appointmentRequest->appointment_time ? (is_string($appointmentRequest->appointment_time) ? substr($appointmentRequest->appointment_time, 0, 5) : $appointmentRequest->appointment_time->format('H:i')) : null,
                    'personel_id' => $appointmentRequest->personel_id,
                    'room_id' => $appointmentRequest->room_id,
                    'notes' => $appointmentRequest->notes,
                    'status' => $appointmentRequest->status,
                ],
                'personels' => $personels->map(function($p) {
                    return ['id' => $p->id, 'name' => $p->name];
                }),
                'rooms' => $rooms->map(function($r) {
                    return ['id' => $r->id, 'name' => $r->name];
                }),
                'customer' => $customer ? [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'surname' => $customer->surname,
                    'phone' => $customer->phone,
                    'age' => $customer->age,
                ] : null,
                'customer_reservations' => $customerReservations->map(function($r) {
                    return [
                        'id' => $r->id,
                        'date' => $r->date,
                        'start_time' => $r->start_time,
                        'personel_name' => $r->personel ? $r->personel->name : null,
                        'service_name' => $r->appointmentRequest && $r->appointmentRequest->service ? $r->appointmentRequest->service->name : null,
                        'room_name' => $r->room ? $r->room->name : null,
                        'status' => $r->status,
                        'total_price' => $r->total_price ?? 0,
                    ];
                }),
            ]);
        }

        return view('admin.appointment-requests.show', compact('appointmentRequest', 'personels', 'rooms', 'customer', 'customerReservations'));
    }

    public function approve(Request $request, $id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $appointmentRequest = AppointmentRequest::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->where('status', 'pending')
            ->firstOrFail();

        $validated = $request->validate([
            'appointment_date' => 'required|date',
            'appointment_time' => 'required|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'personel_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
            'notes' => 'nullable|string',
            'total_price' => 'required|numeric|min:0',
            'create_invoice' => 'nullable|boolean',
        ]);

        $customer = null;
        if ($appointmentRequest->customer_id) {
            $customer = \App\Models\Customer::where('id', $appointmentRequest->customer_id)
                ->where('beauty_salon_id', $user->id)
                ->first();
        }
        
        if (!$customer) {
            
            $customerName = trim($appointmentRequest->name . ' ' . ($appointmentRequest->surname ?? ''));
            $customer = \App\Models\Customer::firstOrCreate(
                [
                    'phone' => $appointmentRequest->phone,
                    'beauty_salon_id' => $user->id,
                ],
                [
                    'name' => $customerName,
                    'email' => null,
                    'created_by' => $user->id,
                ]
            );

            if (!$customer->beauty_salon_id) {
                $customer->update(['beauty_salon_id' => $user->id]);
            }
        }

        $code = 'REZ-' . strtoupper(uniqid());

        $totalPrice = $validated['total_price'];

        $itemsData = [];
        if ($appointmentRequest->service) {
            $itemsData = [
                'service' => [
                    'id' => $appointmentRequest->service->id,
                    'name' => $appointmentRequest->service->name,
                    'price' => $totalPrice, 
                ]
            ];
        }

        $endTime = $validated['end_time'] ?? null;
        if (!$endTime && $appointmentRequest->service && $appointmentRequest->service->duration) {
            $startTime = \Carbon\Carbon::parse($validated['appointment_time']);
            $endTime = $startTime->copy()->addMinutes($appointmentRequest->service->duration)->format('H:i:s');
        } else {
            $endTime = $endTime ?? date('H:i:s', strtotime($validated['appointment_time'] . ' +1 hour'));
        }

        $reservation = Reservation::create([
            'code' => $code,
            'customer_id' => $customer->id,
            'salon_id' => null, 
            'beauty_salon_id' => $user->id,
            'date' => $validated['appointment_date'],
            'start_time' => $validated['appointment_time'],
            'end_time' => $endTime,
            'personel_id' => $validated['personel_id'] ?? $appointmentRequest->personel_id,
            'room_id' => $validated['room_id'] ?? $appointmentRequest->room_id,
            'status' => 'confirmed',
            'total_price' => $totalPrice,
            'notes' => $validated['notes'] ?? $appointmentRequest->notes,
            'items_data' => json_encode($itemsData),
            'created_by' => $user->id,
        ]);

        $appointmentRequest->update([
            'status' => 'approved',
            'reservation_id' => $reservation->id,
            'appointment_date' => $validated['appointment_date'],
            'appointment_time' => $validated['appointment_time'],
            'personel_id' => $validated['personel_id'] ?? $appointmentRequest->personel_id,
            'room_id' => $validated['room_id'] ?? $appointmentRequest->room_id,
            'customer_id' => $customer->id,
        ]);

        if ($appointmentRequest->recurring_data) {
            try {
                $recurringData = $appointmentRequest->recurring_data;
                $recurringType = $recurringData['recurring_type'] ?? null;
                $recurringCount = $recurringData['recurring_count'] ?? 1;
                
                if ($recurringType && $recurringCount > 1) {
                    $createdReservations = [];
                    $startDate = \Carbon\Carbon::parse($validated['appointment_date']);
                    $currentDate = $startDate->copy();

                    for ($i = 1; $i < $recurringCount; $i++) {
                        if ($recurringType === 'monthly') {
                            
                            $targetDay = $recurringData['day_of_month'] ?? $startDate->day;
                            $currentDate->addMonth();
                            $currentDate->day($targetDay);
                        } else if ($recurringType === 'weekly') {
                            
                            $targetDayOfWeek = $recurringData['day_of_week'] ?? null;
                            if ($targetDayOfWeek) {

                                $carbonDayOfWeek = $targetDayOfWeek === 7 ? 0 : $targetDayOfWeek;
                                $currentDate->addWeek();
                                
                                $currentDayOfWeek = $currentDate->dayOfWeek;
                                $daysToAdd = ($carbonDayOfWeek - $currentDayOfWeek);
                                if ($daysToAdd < 0) {
                                    $daysToAdd += 7;
                                }
                                $currentDate->addDays($daysToAdd);
                            } else {
                                $currentDate->addWeek();
                            }
                        }

                        $code = 'REZ-' . strtoupper(uniqid());

                        $newReservation = Reservation::create([
                            'code' => $code,
                            'customer_id' => $customer->id,
                            'beauty_salon_id' => $user->id,
                            'date' => $currentDate->format('Y-m-d'),
                            'start_time' => $validated['appointment_time'],
                            'end_time' => $endTime,
                            'personel_id' => $validated['personel_id'] ?? $appointmentRequest->personel_id,
                            'room_id' => $validated['room_id'] ?? $appointmentRequest->room_id,
                            'status' => 'confirmed',
                            'total_price' => $totalPrice,
                            'notes' => ($validated['notes'] ?? $appointmentRequest->notes ?? '') . ' (Tekrarlanan Seans)',
                            'items_data' => json_encode($itemsData),
                            'created_by' => $user->id,
                        ]);

                        $newAppointmentRequest = AppointmentRequest::create([
                            'beauty_salon_id' => $user->id,
                            'customer_id' => $customer->id,
                            'service_id' => $appointmentRequest->service_id,
                            'name' => $appointmentRequest->name,
                            'surname' => $appointmentRequest->surname,
                            'phone' => $appointmentRequest->phone,
                            'age' => $appointmentRequest->age,
                            'appointment_date' => $currentDate->format('Y-m-d'),
                            'appointment_time' => $validated['appointment_time'],
                            'personel_id' => $validated['personel_id'] ?? $appointmentRequest->personel_id,
                            'room_id' => $validated['room_id'] ?? $appointmentRequest->room_id,
                            'status' => 'approved',
                            'reservation_id' => $newReservation->id,
                            'notes' => $appointmentRequest->notes,
                            'phone_verified' => true,
                        ]);
                        
                        $createdReservations[] = $newReservation;
                    }
                    
                    \Log::info('Tekrarlanan seanslar otomatik oluşturuldu', [
                        'original_reservation_id' => $reservation->id,
                        'recurring_type' => $recurringType,
                        'count' => $recurringCount,
                        'created_count' => count($createdReservations),
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Tekrarlanan seanslar oluşturulurken hata', [
                    'appointment_request_id' => $appointmentRequest->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        $invoice = null;
        $invoiceUrl = '';
            try {
            $invoice = $this->createInvoiceForReservation($reservation, $appointmentRequest, $customer, $user);
            if ($invoice) {
                $invoiceUrl = $invoice->public_url;
                \Log::info('✅ Randevu onaylandığında otomatik fatura oluşturuldu', [
                    'reservation_id' => $reservation->id,
                    'invoice_id' => $invoice->id,
                    'invoice_url' => $invoiceUrl,
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('❌ Randevu onaylandığında fatura oluşturma hatası', [
                'reservation_id' => $reservation->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }

        $smsSent = false;
        if ($customer && $customer->phone) {
            try {
                $smsService = app(\App\Services\SmsService::class);

                $salonName = $user->salon_name ?? $user->name;
                $salonAddress = $user->salon_address ?? '';

                $formattedDate = \Carbon\Carbon::parse($validated['appointment_date'])->format('d/m/Y');
                $formattedTime = \Carbon\Carbon::parse($validated['appointment_time'])->format('H:i');

                $contract = \App\Models\Contract::where('user_id', $user->id)
                    ->where('is_default', true)
                    ->first();
                
                if (!$contract) {
                    $contract = \App\Models\Contract::where('user_id', $user->id)
                        ->where('is_active', true)
                        ->first();
                }
                
                $contractUrl = $contract ? $contract->public_url : '';

                $message = $salonName . "\n\n";
                if ($salonAddress) {
                    $message .= "Adres: " . $salonAddress . "\n\n";
                }
                $message .= $formattedDate . " tarihinde, " . $formattedTime . " saatinde randevunuz oluşturulmuştur.\n";
                $message .= "Randevu saatinizden 5 dakika önce salonda hazır bulunmanızı rica ederiz.\n\n";

                if ($invoiceUrl) {
                    $message .= "Fatura:\n";
                    $message .= $invoiceUrl . "\n\n";
                }

                if ($contractUrl) {
                    $message .= "Sözleşme:\n";
                    $message .= $contractUrl . "\n\n";
                }
                
                $message .= "salonay.com iyi günler diler.";

                $result = $smsService->send(
                    $customer->phone,
                    $message,
                    null,
                    ['purpose' => 'default', 'user_id' => $user->id]
                );
                
                if ($result && $result->success) {
                    $smsSent = true;
                    \Log::info('✅ Randevu onaylandığında SMS mesajı gönderildi', [
                        'reservation_id' => $reservation->id,
                        'phone' => $customer->phone,
                        'salon_name' => $salonName,
                    ]);
                } else {
                    \Log::warning('⚠️ Randevu onaylandığında SMS mesajı gönderilemedi', [
                        'reservation_id' => $reservation->id,
                        'phone' => $customer->phone,
                        'error' => $result->message ?? 'Bilinmeyen hata'
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('❌ Randevu SMS mesajı gönderme hatası', [
                    'reservation_id' => $reservation->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        $smsSent = false;
        if ($customer && $customer->phone && ($validated['create_invoice'] ?? false)) {
            try {
                $smsService = app(\App\Services\SmsService::class);
                
                $formattedDate = \Carbon\Carbon::parse($validated['appointment_date'])->format('d.m.Y');
                $time = $validated['appointment_time'];
                $serviceName = $appointmentRequest->service ? $appointmentRequest->service->name : 'Randevu';
                
                $personelName = '';
                $personel = \App\Models\User::find($validated['personel_id'] ?? $appointmentRequest->personel_id);
                if ($personel) {
                    $personelName = $personel->name;
                }
                
                $message = "Sayın {$customer->name}, {$formattedDate} tarihinde {$time} saatinde {$serviceName} randevunuz onaylanmıştır. ";
                if ($personelName) {
                    $message .= "Personel: {$personelName}. ";
                }
                if ($invoice) {
                    $message .= "Faturanız oluşturulmuştur. ";
                }
                $message .= "Salonay";

                $customerPhone = preg_replace('/[^0-9]/', '', $customer->phone);
                
                $result = $smsService->send(
                    $customerPhone,
                    $message,
                    null,
                    ['purpose' => 'reservation', 'user_id' => $user->id]
                );
                
                if ($result && $result->success) {
                    $smsSent = true;
                    \Log::info('✅ Randevu onaylandığında ve faturalandığında SMS gönderildi', [
                        'reservation_id' => $reservation->id,
                        'invoice_id' => $invoice ? $invoice->id : null,
                        'phone' => $customerPhone,
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('❌ Randevu SMS gönderme hatası', [
                    'reservation_id' => $reservation->id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        if ($request->ajax() || $request->wantsJson()) {
            $successMessage = 'Randevu talebi onaylandı ve randevu oluşturuldu.';
            if ($smsSent) {
                $successMessage .= ' SMS mesajı gönderildi.';
            }
            if ($smsSent) {
                $successMessage .= ' SMS gönderildi.';
            }
            
            return response()->json([
                'success' => true,
                'message' => $successMessage,
                'reservation_id' => $reservation->id,
                'invoice_id' => $invoice ? $invoice->id : null,
                'sms_sent' => $smsSent,
                'sms_sent' => $smsSent,
            ]);
        }
        
        $successMessage = 'Randevu talebi onaylandı ve randevu oluşturuldu.';
        if ($whatsappSent) {
            $successMessage .= ' WhatsApp mesajı gönderildi.';
        }
        if ($smsSent) {
            $successMessage .= ' SMS gönderildi.';
        }
        
        return redirect()->back()
            ->with('success', $successMessage);
    }

    public function reject(Request $request, $id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $appointmentRequest = AppointmentRequest::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->where('status', 'pending')
            ->firstOrFail();

        $rejectionReason = $request->input('rejection_reason') ?? $request->json('rejection_reason') ?? 'Belirtilmedi';
        
        $appointmentRequest->update([
            'status' => 'rejected',
            'notes' => ($appointmentRequest->notes ?? '') . "\n\nRed Nedeni: " . $rejectionReason,
        ]);

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Randevu talebi reddedildi.',
            ]);
        }
        
        return redirect()->back()
            ->with('success', 'Randevu talebi reddedildi.');
    }

    public function customerPortfolio(Request $request, $customerId)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $customer = Customer::where('id', $customerId)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
            
        $reservations = Reservation::where('customer_id', $customerId)
            ->where('beauty_salon_id', $user->id)
            ->with(['personel', 'appointmentRequest.service', 'room'])
            ->orderBy('date', 'desc')
            ->get();

        $statistics = [
            'total' => $reservations->count(),
            'completed' => $reservations->where('status', 'completed')->count(),
            'total_spent' => $reservations->sum('total_price'),
        ];

        if ($request->ajax() || $request->wantsJson() || $request->accepts('application/json')) {
            return response()->json([
                'customer' => $customer,
                'reservations' => $reservations->map(function($reservation) {
                    return [
                        'id' => $reservation->id,
                        'date' => $reservation->date,
                        'start_time' => $reservation->start_time,
                        'end_time' => $reservation->end_time,
                        'status' => $reservation->status,
                        'total_price' => $reservation->total_price,
                        'personel_name' => $reservation->personel ? $reservation->personel->name : null,
                        'service_name' => $reservation->appointmentRequest && $reservation->appointmentRequest->service ? $reservation->appointmentRequest->service->name : null,
                        'room_name' => $reservation->room ? $reservation->room->name : null,
                    ];
                }),
                'statistics' => $statistics,
            ]);
        }

            return view('admin.appointment-requests.customer-portfolio', compact('customer', 'reservations', 'statistics'));
    }

    public function getData(Request $request)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }

        $personels = \App\Models\User::where('role', 'personel')
            ->where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get(['id', 'name', 'email', 'phone', 'profile_image']);

        $rooms = \App\Models\Room::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get(['id', 'name']);

        $services = \App\Models\Service::where('beauty_salon_id', $user->id)
            ->with('category')
            ->orderBy('name')
            ->get(['id', 'name', 'price', 'service_category_id']);

        $categories = \App\Models\ServiceCategory::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get(['id', 'name', 'color']);
        
        return response()->json([
            'personels' => $personels->map(function($personel) {
                return [
                    'id' => $personel->id,
                    'name' => $personel->name,
                    'email' => $personel->email ?? '',
                    'phone' => $personel->phone ?? '',
                    'profile_image' => $personel->profile_image,
                    'color' => '#ec4899'
                ];
            }),
            'rooms' => $rooms,
            'services' => $services,
            'categories' => $categories
        ]);
    }

    public function getCustomerInfo(Request $request, $customerId)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $customer = Customer::where('id', $customerId)
            ->where(function($query) use ($user) {
                if (\Schema::hasColumn('customers', 'beauty_salon_id')) {
                    $query->where('beauty_salon_id', $user->id);
                } else {
                    $query->where('created_by', $user->id);
                }
            })
            ->firstOrFail();

        $reservations = Reservation::where('customer_id', $customerId)
            ->where('beauty_salon_id', $user->id)
            ->with(['personel', 'appointmentRequest.service', 'room'])
            ->orderBy('date', 'desc')
            ->limit(3)
            ->get();

        $allReservations = Reservation::where('customer_id', $customerId)
            ->where('beauty_salon_id', $user->id)
            ->get();
        
        $statistics = [
            'total' => $allReservations->count(),
            'completed' => $allReservations->where('status', 'completed')->count(),
            'total_spent' => $allReservations->sum('total_price'),
        ];
        
        return response()->json([
            'customer' => $customer,
            'reservations' => $reservations->map(function($reservation) {
                return [
                    'id' => $reservation->id,
                    'date' => $reservation->date,
                    'start_time' => $reservation->start_time,
                    'status' => $reservation->status,
                    'total_price' => $reservation->total_price,
                    'personel_name' => $reservation->personel ? $reservation->personel->name : null,
                    'service_name' => $reservation->appointmentRequest && $reservation->appointmentRequest->service ? $reservation->appointmentRequest->service->name : null,
                    'room_name' => $reservation->room ? $reservation->room->name : null,
                ];
            }),
            'statistics' => $statistics,
        ]);
    }

    private function createInvoiceForReservation($reservation, $appointmentRequest, $customer, $user)
    {
        try {
            
            $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);

            $publicToken = Str::random(32);

            $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
            $logo = $settings['invoice_logo'] ?? null;
            $address = $user->salon_address ?? ($settings['invoice_address'] ?? '');
            $taxOffice = $settings['invoice_tax_office'] ?? '';
            $taxNumber = $settings['invoice_tax_number'] ?? '';

            $invoicePrice = $reservation->total_price ?? ($appointmentRequest->service ? $appointmentRequest->service->price : 0);
            $items = [];
            
            if ($appointmentRequest->service) {
                $items[] = [
                    'name' => $appointmentRequest->service->name,
                    'quantity' => '1',
                    'price' => $invoicePrice, 
                    'total' => $invoicePrice  
                ];
            } else {
                
                $items[] = [
                    'name' => 'Randevu Hizmeti',
                    'quantity' => '1',
                    'price' => $invoicePrice,
                    'total' => $invoicePrice
                ];
            }

            $invoiceData = [
                'reservation' => $reservation,
                'transaction' => false,
                'invoice' => null,
                'items' => $items,
                'subtotal' => $invoicePrice, 
                'grandTotal' => $invoicePrice, 
                'depositPaid' => 0,
                'logo' => $logo,
                'address' => $address,
                'salonName' => $user->salon_name ?? $user->name,
                'salonAddress' => $address,
                'taxOffice' => $taxOffice,
                'taxNumber' => $taxNumber,
                'customerName' => $customer->name,
                'invoiceDate' => now(),
            ];

            $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);

            $pdfPath = 'invoices/' . $invoiceNumber . '.pdf';
            Storage::disk('public')->put($pdfPath, $pdf->output());

            $publicUrl = url('/fatura/' . $publicToken);
            if (strpos($publicUrl, 'localhost') !== false || strpos($publicUrl, '127.0.0.1') !== false) {
                $publicUrl = str_replace(['http://', 'https://'], '', $publicUrl);
            }

            $invoice = Invoice::create([
                'reservation_id' => $reservation->id,
                'customer_id' => $customer->id,
                'invoice_number' => $invoiceNumber,
                'total_amount' => $invoicePrice, 
                'deposit_paid' => 0,
                'remaining_amount' => $invoicePrice, 
                'invoice_date' => now(),
                'pdf_path' => $pdfPath,
                'public_token' => $publicToken,
                'public_url' => $publicUrl,
                'status' => 'sent',
                'items_json' => json_encode($items),
                'created_by' => $user->id,
            ]);
            
            return $invoice;
        } catch (\Exception $e) {
            \Log::error('Fatura oluşturma hatası', [
                'reservation_id' => $reservation->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }
}

