<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Salon;
use App\Services\SmsService;
use App\Services\Sms\OtpService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class BeautySalonProfileController extends Controller
{
    protected $smsService;
    protected $otpService;

    public function __construct(SmsService $smsService, OtpService $otpService)
    {
        $this->smsService = $smsService;
        $this->otpService = $otpService;
    }

    public function edit()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $isFirstLogin = session('first_login_setup', false) || (empty($user->salon_name) || empty($user->salon_slug));

        $personelCount = \App\Models\User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->count();
        
        $serviceCount = \App\Models\Service::where('beauty_salon_id', $user->id)
            ->count();
        
        $roomCount = \App\Models\Room::where('beauty_salon_id', $user->id)
            ->count();
        
        $customerCount = \App\Models\Customer::where(function($query) use ($user) {
            if (\Schema::hasColumn('customers', 'beauty_salon_id')) {
                $query->where('beauty_salon_id', $user->id);
            } else {
                $query->where('created_by', $user->id);
            }
        })->count();

        $personels = \App\Models\User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->with('services')
            ->orderBy('name')
            ->limit(10)
            ->get();
        
        $services = \App\Models\Service::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->limit(10)
            ->get();
        
        $rooms = \App\Models\Room::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->limit(10)
            ->get();

        return view('admin.beauty-salon.profile', compact('user', 'isFirstLogin', 'personelCount', 'serviceCount', 'roomCount', 'customerCount', 'personels', 'services', 'rooms'));
    }

    public function update(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string|unique:users,phone,' . $user->id,
            'phone_otp' => 'nullable|string|size:6',
            'salon_name' => 'required|string|max:255',
            'salon_short_name' => 'required|string|max:50',
            'short_url' => 'nullable|string|max:50|unique:users,short_url,' . $user->id,
            'salon_phone' => 'required|string|min:10|max:20',
            'salon_city' => 'required|string|max:255',
            'salon_district' => 'required|string|max:255',
            'salon_address' => 'required|string',
            'salon_description' => 'nullable|string',
            'salon_theme' => 'required|in:theme1',
            'salon_color' => 'required|in:pink,blue,purple,orange,green',
            'salon_banner_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:10240',
            'salon_profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'salon_gallery.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'remove_gallery_images' => 'nullable|array',
        ]);

        $workingHoursValidation = $this->normalizeSalonWorkingHours($request->input('working_hours', []));
        if (!$workingHoursValidation['ok']) {
            return back()
                ->withErrors(['working_hours' => $workingHoursValidation['message']])
                ->withInput();
        }

        $newPhone = preg_replace('/[^0-9]/', '', $validated['phone']);
        if (!empty($newPhone) && !str_starts_with($newPhone, '0')) {
            $newPhone = '0' . $newPhone;
        }

        $oldPhone = preg_replace('/[^0-9]/', '', $user->phone ?? '');

        if ($newPhone !== $oldPhone) {
            if (!$request->filled('phone_otp')) {
                return back()->withErrors(['phone_otp' => 'Telefon numarası değiştiği için doğrulama kodu gereklidir. Lütfen "OTP Kodu Gönder" butonuna tıklayın.'])->withInput();
            }

            $otpCode = (int) $request->phone_otp;
            if (!$this->otpService->verify($newPhone, $otpCode, 'phone_change')) {
                return back()->withErrors(['phone_otp' => 'Doğrulama kodu hatalı veya süresi dolmuş. Lütfen tekrar kod gönderin.'])->withInput();
            }
        }

        $salonSlug = Str::slug($validated['salon_short_name']);
        $slugCounter = 1;
        $originalSlug = $salonSlug;

        while (\App\Http\Controllers\Public\ShortUrlController::isUrlTaken($salonSlug, $user->id)) {
            $salonSlug = $originalSlug . '-' . $slugCounter;
            $slugCounter++;
        }

        if ($request->hasFile('salon_profile_image')) {
            try {
                $file = $request->file('salon_profile_image');
                
                if ($file->getSize() > 5120 * 1024) {
                    return back()->withErrors(['salon_profile_image' => 'Dosya boyutu çok büyük. Maksimum 5 MB olmalıdır.'])->withInput();
                }
                
                $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                if (!in_array($file->getMimeType(), $allowedMimes)) {
                    return back()->withErrors(['salon_profile_image' => 'Desteklenmeyen dosya formatı.'])->withInput();
                }
                
                if ($user->salon_profile_image && Storage::disk('public')->exists($user->salon_profile_image)) {
                    try {
                        Storage::disk('public')->delete($user->salon_profile_image);
                    } catch (\Exception $e) {
                        
                    }
                }

                $directory = 'salon-profiles';
                if (!Storage::disk('public')->exists($directory)) {
                    try {
                        Storage::disk('public')->makeDirectory($directory, 0755, true);
                    } catch (\Exception $dirException) {
                        Log::error('Dizin oluşturma hatası (salon-profiles): ' . $dirException->getMessage());
                        return back()->withErrors(['salon_profile_image' => 'Dizin oluşturulamadı. Lütfen storage/app/public dizininin yazılabilir olduğundan emin olun.'])->withInput();
                    }
                }

                try {
                    
                    $filename = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();

                    $stored = $file->store($directory, 'public');
                    
                    if (!$stored) {
                        Log::error('Profil fotoğrafı store() metodu false döndü', [
                            'directory' => $directory,
                            'filename' => $filename,
                            'file_size' => $file->getSize(),
                            'file_mime' => $file->getMimeType(),
                            'file_name' => $file->getClientOriginalName()
                        ]);
                        return back()->withErrors(['salon_profile_image' => 'Dosya yüklenirken bir hata oluştu. Lütfen dosya boyutunu ve formatını kontrol edin.'])->withInput();
                    }
                    
                    $user->salon_profile_image = $stored;
                } catch (\Exception $storeException) {
                    Log::error('Profil fotoğrafı yükleme hatası: ' . $storeException->getMessage(), [
                        'exception' => $storeException,
                        'trace' => $storeException->getTraceAsString(),
                        'directory' => $directory,
                        'file_size' => $file->getSize(),
                        'file_mime' => $file->getMimeType(),
                        'file_name' => $file->getClientOriginalName()
                    ]);
                    return back()->withErrors(['salon_profile_image' => 'Dosya yüklenirken bir hata oluştu: ' . $storeException->getMessage()])->withInput();
                }
            } catch (\Exception $e) {
                Log::error('Profil fotoğrafı yükleme hatası: ' . $e->getMessage(), [
                    'exception' => $e,
                    'trace' => $e->getTraceAsString()
                ]);
                return back()->withErrors(['salon_profile_image' => 'Dosya yüklenirken bir hata oluştu: ' . $e->getMessage()])->withInput();
            }
        }

        if ($request->hasFile('salon_banner_image')) {
            try {
                $file = $request->file('salon_banner_image');
                
                if ($file->getSize() > 10240 * 1024) {
                    return back()->withErrors(['salon_banner_image' => 'Dosya boyutu çok büyük. Maksimum 10 MB olmalıdır.'])->withInput();
                }
                
                $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                if (!in_array($file->getMimeType(), $allowedMimes)) {
                    return back()->withErrors(['salon_banner_image' => 'Desteklenmeyen dosya formatı.'])->withInput();
                }
                
                if ($user->salon_banner_image && Storage::disk('public')->exists($user->salon_banner_image)) {
                    try {
                        Storage::disk('public')->delete($user->salon_banner_image);
                    } catch (\Exception $e) {
                        
                    }
                }

                $directory = 'salon-banners';
                if (!Storage::disk('public')->exists($directory)) {
                    try {
                        Storage::disk('public')->makeDirectory($directory, 0755, true);
                    } catch (\Exception $dirException) {
                        Log::error('Dizin oluşturma hatası (salon-banners): ' . $dirException->getMessage());
                        return back()->withErrors(['salon_banner_image' => 'Dizin oluşturulamadı. Lütfen storage/app/public dizininin yazılabilir olduğundan emin olun.'])->withInput();
                    }
                }

                try {
                    
                    $filename = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();

                    $stored = $file->store($directory, 'public');
                    
                    if (!$stored) {
                        Log::error('Banner fotoğrafı store() metodu false döndü', [
                            'directory' => $directory,
                            'filename' => $filename,
                            'file_size' => $file->getSize(),
                            'file_mime' => $file->getMimeType(),
                            'file_name' => $file->getClientOriginalName()
                        ]);
                        return back()->withErrors(['salon_banner_image' => 'Dosya yüklenirken bir hata oluştu. Lütfen dosya boyutunu ve formatını kontrol edin.'])->withInput();
                    }
                    
                    $user->salon_banner_image = $stored;
                } catch (\Exception $storeException) {
                    Log::error('Banner fotoğrafı yükleme hatası: ' . $storeException->getMessage(), [
                        'exception' => $storeException,
                        'trace' => $storeException->getTraceAsString(),
                        'directory' => $directory,
                        'file_size' => $file->getSize(),
                        'file_mime' => $file->getMimeType(),
                        'file_name' => $file->getClientOriginalName()
                    ]);
                    return back()->withErrors(['salon_banner_image' => 'Dosya yüklenirken bir hata oluştu: ' . $storeException->getMessage()])->withInput();
                }
            } catch (\Exception $e) {
                Log::error('Banner fotoğrafı yükleme hatası: ' . $e->getMessage(), [
                    'exception' => $e,
                    'trace' => $e->getTraceAsString()
                ]);
                return back()->withErrors(['salon_banner_image' => 'Dosya yüklenirken bir hata oluştu: ' . $e->getMessage()])->withInput();
            }
        }

        $gallery = $user->salon_gallery ?? [];
        
        if ($request->has('remove_gallery_images')) {
            foreach ($request->remove_gallery_images as $imagePath) {
                if (Storage::disk('public')->exists($imagePath)) {
                    Storage::disk('public')->delete($imagePath);
                }
                $gallery = array_filter($gallery, function($item) use ($imagePath) {
                    return $item !== $imagePath;
                });
            }
            $gallery = array_values($gallery);
        }
        
        if ($request->hasFile('salon_gallery')) {
            try {
                $directory = 'salon-gallery';
                if (!Storage::disk('public')->exists($directory)) {
                    Storage::disk('public')->makeDirectory($directory, 0755, true);
                }
                
                $maxImages = 10;
                $currentCount = count($gallery);
                
                foreach ($request->file('salon_gallery') as $file) {
                    if ($currentCount >= $maxImages) {
                        return back()->withErrors(['salon_gallery' => "Maksimum {$maxImages} görsel yükleyebilirsiniz."])->withInput();
                    }
                    
                    if ($file->getSize() > 5120 * 1024) {
                        return back()->withErrors(['salon_gallery' => 'Bir veya daha fazla dosya boyutu çok büyük.'])->withInput();
                    }
                    
                    $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                    if (!in_array($file->getMimeType(), $allowedMimes)) {
                        return back()->withErrors(['salon_gallery' => 'Desteklenmeyen dosya formatı.'])->withInput();
                    }
                    
                    $path = $file->store($directory, 'public');
                    if ($path) {
                        $gallery[] = $path;
                        $currentCount++;
                    }
                }
            } catch (\Exception $e) {
                Log::error('Galeri görseli yükleme hatası: ' . $e->getMessage());
                return back()->withErrors(['salon_gallery' => 'Görseller yüklenirken bir hata oluştu.'])->withInput();
            }
        }
        
        $user->salon_gallery = !empty($gallery) ? $gallery : null;

        $salonPhone = preg_replace('/[^0-9]/', '', $validated['salon_phone']);
        if (!empty($salonPhone) && !str_starts_with($salonPhone, '0')) {
            $salonPhone = '0' . $salonPhone;
        }

        $salonSlug = Str::slug($validated['salon_short_name']);
        $slugCounter = 1;
        $originalSlug = $salonSlug;

        while (\App\Http\Controllers\Public\ShortUrlController::isUrlTaken($salonSlug, $user->id)) {
            $salonSlug = $originalSlug . '-' . $slugCounter;
            $slugCounter++;
        }

        if (!empty($validated['short_url'])) {
            $shortUrl = Str::slug($validated['short_url']);
            $shortUrlCounter = 1;
            $originalShortUrl = $shortUrl;

            while (\App\Http\Controllers\Public\ShortUrlController::isUrlTaken($shortUrl, $user->id)) {
                $shortUrl = $originalShortUrl . $shortUrlCounter;
                $shortUrlCounter++;
            }
            $user->short_url = $shortUrl;
        } else {
            if (empty($user->short_url)) {
                $shortUrl = Str::random(6);
                while (\App\Http\Controllers\Public\ShortUrlController::isUrlTaken($shortUrl, $user->id)) {
                    $shortUrl = Str::random(6);
                }
                $user->short_url = $shortUrl;
            }
        }

        $user->name = $validated['name'];
        $user->email = $validated['email'];
        $user->phone = $newPhone;
        $user->salon_name = $validated['salon_name'];
        $user->salon_short_name = $validated['salon_short_name'];
        $user->salon_slug = $salonSlug; 
        $user->salon_phone = $salonPhone;
        $user->salon_city = $validated['salon_city'];
        $user->salon_district = $validated['salon_district'];
        $user->salon_address = $validated['salon_address'];
        $user->salon_description = $validated['salon_description'] ?? null;
        $user->salon_working_hours = $workingHoursValidation['data'];
        // Diger temalar gecici olarak devre disi: tum hesaplar theme1 kullanir.
        $user->salon_theme = 'theme1';
        $user->salon_color = $validated['salon_color'];
        $user->save();

        $salon = Salon::where('created_by', $user->id)->first();
            
        if (!$salon) {
            Salon::create([
                'name' => $validated['salon_name'],
                'created_by' => $user->id,
                'is_active' => true,
                'capacity' => 100,
                'address' => $validated['salon_address'],
                'phone' => $salonPhone,
                'city' => $validated['salon_city'],
                'district' => $validated['salon_district'],
            ]);
        } else {
            $salon->name = $validated['salon_name'];
            $salon->address = $validated['salon_address'];
            $salon->phone = $salonPhone;
            $salon->city = $validated['salon_city'];
            $salon->district = $validated['salon_district'];
            $salon->save();
        }

        session()->forget('first_login_setup');

        return redirect()->route('admin.beauty-salon.profile')
            ->with('success', 'Salon bilgileri başarıyla kaydedildi!');
    }

    public function sendPhoneChangeOtp(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $request->validate([
            'new_phone' => 'required|string|unique:users,phone,' . $user->id,
        ], [
            'new_phone.required' => 'Yeni telefon numarası zorunludur.',
            'new_phone.unique' => 'Bu telefon numarası zaten kullanılıyor.',
        ]);

        $newPhone = preg_replace('/[^0-9]/', '', $request->new_phone);
        if (!empty($newPhone) && !str_starts_with($newPhone, '0')) {
            $newPhone = '0' . $newPhone;
        }

        $result = $this->smsService->sendOtp($newPhone, 'phone_change');

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Doğrulama kodu ' . $newPhone . ' numarasına gönderildi.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
        ], 400);
    }

    private function normalizeSalonWorkingHours(array $input): array
    {
        $days = [
            'monday' => 'Pazartesi',
            'tuesday' => 'Salı',
            'wednesday' => 'Çarşamba',
            'thursday' => 'Perşembe',
            'friday' => 'Cuma',
            'saturday' => 'Cumartesi',
            'sunday' => 'Pazar',
        ];

        $normalized = [];
        $openDayCount = 0;

        foreach ($days as $dayKey => $dayLabel) {
            $dayData = $input[$dayKey] ?? [];
            $isOpen = filter_var($dayData['is_open'] ?? false, FILTER_VALIDATE_BOOLEAN);

            $normalized[$dayKey] = [
                'is_open' => $isOpen,
                'open_time' => null,
                'close_time' => null,
            ];

            if (!$isOpen) {
                continue;
            }

            $openTime = trim((string) ($dayData['open_time'] ?? ''));
            $closeTime = trim((string) ($dayData['close_time'] ?? ''));

            if (!preg_match('/^\d{2}:\d{2}$/', $openTime) || !preg_match('/^\d{2}:\d{2}$/', $closeTime)) {
                return [
                    'ok' => false,
                    'message' => $dayLabel . ' için açılış ve kapanış saatini seçmelisiniz.',
                ];
            }

            if ($openTime >= $closeTime) {
                return [
                    'ok' => false,
                    'message' => $dayLabel . ' için kapanış saati açılış saatinden sonra olmalıdır.',
                ];
            }

            $normalized[$dayKey]['open_time'] = $openTime;
            $normalized[$dayKey]['close_time'] = $closeTime;
            $openDayCount++;
        }

        if ($openDayCount === 0) {
            return [
                'ok' => false,
                'message' => 'En az bir günü açık olarak seçmelisiniz.',
            ];
        }

        return [
            'ok' => true,
            'data' => $normalized,
        ];
    }

}
