<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class BranchController extends Controller
{
    // Şube sistemi bakım modunda
    private const MAINTENANCE_MODE = true;
    
    public function index()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.branches.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $branches = Branch::where('parent_salon_id', $user->id)
            ->with(['branchManager', 'staff'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('admin.branches.index', compact('branches'));
    }

    public function create()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.branches.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $staff = User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->orderBy('name')
            ->get();

        return view('admin.branches.create', compact('staff'));
    }

    public function store(Request $request)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.beauty-salon.branches.index')
                ->with('error', 'Şube sistemi şu anda bakım modunda. Lütfen daha sonra tekrar deneyin.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string',
            'address' => 'nullable|string',
            'city' => 'nullable|string',
            'district' => 'nullable|string',
            'branch_manager_id' => 'nullable|exists:users,id',
            'revenue_share_percentage' => 'nullable|numeric|min:0|max:100',
            'can_create_campaigns' => 'boolean',
        ]);

        $validated['parent_salon_id'] = $user->id;
        $validated['slug'] = Str::slug($validated['name']) . '-' . time();
        $validated['status'] = 'active';
        $validated['can_create_campaigns'] = $request->has('can_create_campaigns');
        $validated['revenue_share_percentage'] = $validated['revenue_share_percentage'] ?? 0;

        Branch::create($validated);

        return redirect()->route('admin.beauty-salon.branches.index')
            ->with('success', 'Şube başarıyla oluşturuldu.');
    }

    public function show($id)
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.branches.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $branch = Branch::where('id', $id)
            ->where('parent_salon_id', $user->id)
            ->with(['branchManager', 'staff', 'revenueSharing'])
            ->firstOrFail();

        return view('admin.branches.show', compact('branch'));
    }

    public function edit($id)
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.branches.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $branch = Branch::where('id', $id)
            ->where('parent_salon_id', $user->id)
            ->firstOrFail();

        $staff = User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->orderBy('name')
            ->get();

        return view('admin.branches.edit', compact('branch', 'staff'));
    }

    public function update(Request $request, $id)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.beauty-salon.branches.index')
                ->with('error', 'Şube sistemi şu anda bakım modunda. Lütfen daha sonra tekrar deneyin.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $branch = Branch::where('id', $id)
            ->where('parent_salon_id', $user->id)
            ->firstOrFail();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string',
            'address' => 'nullable|string',
            'city' => 'nullable|string',
            'district' => 'nullable|string',
            'branch_manager_id' => 'nullable|exists:users,id',
            'revenue_share_percentage' => 'nullable|numeric|min:0|max:100',
            'can_create_campaigns' => 'boolean',
            'status' => 'required|in:active,inactive,suspended',
        ]);

        $validated['can_create_campaigns'] = $request->has('can_create_campaigns');
        
        if ($branch->name !== $validated['name']) {
            $validated['slug'] = Str::slug($validated['name']) . '-' . time();
        }

        $branch->update($validated);

        return redirect()->route('admin.beauty-salon.branches.index')
            ->with('success', 'Şube başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.beauty-salon.branches.index')
                ->with('error', 'Şube sistemi şu anda bakım modunda. Lütfen daha sonra tekrar deneyin.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $branch = Branch::where('id', $id)
            ->where('parent_salon_id', $user->id)
            ->firstOrFail();

        $branch->delete();

        return redirect()->route('admin.beauty-salon.branches.index')
            ->with('success', 'Şube başarıyla silindi.');
    }
}
