<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\Customer;
use App\Models\Reservation;
use App\Models\ActivityLog;
use App\Models\Invoice;
use App\Models\Setting;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CashboxController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            
            if (!$user) {
                return redirect()->route('admin.dashboard')->with('error', 'Kullanıcı bilgisi bulunamadı.');
            }

            try {
                $cashAccount = Account::getCashAccount($user->id);
            } catch (\Exception $e) {
                \Log::error('Kasa hesabı alınamadı: ' . $e->getMessage(), [
                    'user_id' => $user->id,
                    'user_role' => $user->role,
                    'beauty_salon_id' => $user->beauty_salon_id ?? null
                ]);
                return redirect()->route('admin.dashboard')->with('error', 'Kasa hesabı oluşturulamadı: ' . $e->getMessage());
            }
            
            if (!$cashAccount) {
                \Log::error('Kasa hesabı null döndü', [
                    'user_id' => $user->id,
                    'user_role' => $user->role
                ]);
                return redirect()->route('admin.dashboard')->with('error', 'Kasa hesabı bulunamadı.');
            }

            // Kasa hesabı salon bazlı olduğu için, sadece bu salonun kasa hesabına ait işlemleri göster
            $baseQuery = Transaction::where('account_id', $cashAccount->id);
            
            // Personel ise sadece kendi işlemlerini görsün, salon sahibi ise tüm salon işlemlerini görsün
            // Salon sahibi için created_by filtresi YOK - tüm salon işlemlerini görmeli
            if ($user->isPersonel()) {
                $baseQuery->where('created_by', $user->id);
            }
            
            $totalIncome = 0;
            $totalExpense = 0;
            $todayIncome = 0;
            $todayExpense = 0;
            $monthlyIncome = 0;
            $monthlyExpense = 0;
            
            try {
                
                $totalIncome = (clone $baseQuery)->where('type', 'income')->sum('amount') ?? 0;
                $totalExpense = (clone $baseQuery)->where('type', 'expense')->sum('amount') ?? 0;

                $today = date('Y-m-d');
                $todayIncome = (clone $baseQuery)->where('type', 'income')->whereDate('date', $today)->sum('amount') ?? 0;
                $todayExpense = (clone $baseQuery)->where('type', 'expense')->whereDate('date', $today)->sum('amount') ?? 0;

                $monthStart = date('Y-m-01');
                $monthEnd = date('Y-m-t');
                $monthlyIncome = (clone $baseQuery)->where('type', 'income')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;
                $monthlyExpense = (clone $baseQuery)->where('type', 'expense')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;

                $totalBalance = $totalIncome - $totalExpense;
            } catch (\Exception $e) {
                \Log::warning('Gelir/Gider hesaplanamadı: ' . $e->getMessage());
                $totalBalance = 0;
            }

            $query = Transaction::where('account_id', $cashAccount->id)
                ->whereIn('type', ['income', 'expense']);

            // Personel ise sadece kendi işlemlerini görsün, salon sahibi ise tüm salon işlemlerini görsün
            // Salon sahibi için created_by filtresi YOK - tüm salon işlemlerini görmeli
            if ($user->isPersonel()) {
                $query->where('created_by', $user->id);
            }

            try {
                $query->with(['account', 'reservation', 'customer']);
            } catch (\Exception $e) {
                \Log::warning('Transaction ilişkileri yüklenemedi: ' . $e->getMessage());
            }

            try {
                $query->orderBy('date', 'desc')->orderBy('created_at', 'desc');
            } catch (\Exception $e) {
                
                try {
                    $query->orderBy('created_at', 'desc');
                } catch (\Exception $e2) {
                    \Log::warning('Transaction sıralama hatası: ' . $e2->getMessage());
                }
            }

            if ($request->filled('date_from')) {
                try {
                    $query->whereDate('date', '>=', $request->date_from);
                } catch (\Exception $e) {
                    \Log::warning('Tarih filtresi uygulanamadı: ' . $e->getMessage());
                }
            }
            if ($request->filled('date_to')) {
                try {
                    $query->whereDate('date', '<=', $request->date_to);
                } catch (\Exception $e) {
                    \Log::warning('Tarih filtresi uygulanamadı: ' . $e->getMessage());
                }
            }
            if ($request->filled('type') && in_array($request->type, ['income', 'expense'])) {
                $query->where('type', $request->type);
            }
            
            $transactions = $query->paginate(50);

            $customers = collect([]);
            $appointments = collect([]);
            
            try {
                $customerQuery = Customer::query();
                
                // Personel ise sadece kendi müşterilerini görsün, salon sahibi ise tüm salon müşterilerini görsün
                if ($user->isPersonel()) {
                    $customerQuery->where('created_by', $user->id);
                } elseif ($user->isGuzellikSalonu()) {
                    // Salon sahibi ise beauty_salon_id'ye göre filtrele (kendi id'si salon id'sidir)
                    $customerQuery->where('beauty_salon_id', $user->id);
                }
                $customers = $customerQuery->orderBy('name')->get();
            } catch (\Exception $e) {
                \Log::warning('Müşteriler yüklenemedi: ' . $e->getMessage());
            }
            
            try {
                $reservationQuery = Reservation::where('status', '!=', 'cancelled');
                
                // Personel ise sadece kendi randevularını görsün, salon sahibi ise tüm salon randevularını görsün
                if ($user->isPersonel()) {
                    $reservationQuery->where('personel_id', $user->id);
                } elseif ($user->isGuzellikSalonu()) {
                    // Salon sahibi ise beauty_salon_id'ye göre filtrele (kendi id'si salon id'sidir)
                    $reservationQuery->where('beauty_salon_id', $user->id);
                }
                $reservationQuery->with('customer');
                try {
                    $reservationQuery->orderBy('date', 'desc');
                } catch (\Exception $e) {
                    $reservationQuery->orderBy('created_at', 'desc');
                }
                $appointments = $reservationQuery->limit(100)->get();
            } catch (\Exception $e) {
                \Log::warning('Randevular yüklenemedi: ' . $e->getMessage());
                
                try {
                    $reservationQuery = Reservation::where('status', '!=', 'cancelled');
                    
                    if ($user->role === 'randevucu' || $user->role === 'admin') {
                        $reservationQuery->where('created_by', $user->id);
                    }
                    $appointments = $reservationQuery->with('customer')
                        ->orderBy('created_at', 'desc')
                        ->limit(100)
                        ->get();
                } catch (\Exception $e2) {
                    \Log::warning('Randevular yüklenemedi (fallback): ' . $e2->getMessage());
                }
            }
            
            return view('admin.accounting.cashbox', compact(
                'transactions', 
                'totalBalance', 
                'customers', 
                'appointments',
                'totalIncome',
                'totalExpense',
                'todayIncome',
                'todayExpense',
                'monthlyIncome',
                'monthlyExpense'
            ))->with('filters', [
                    'date_from' => $request->date_from,
                    'date_to' => $request->date_to,
                    'type' => $request->type,
                ]);
        } catch (\Exception $e) {
            \Log::error('Cashbox Index Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

            return view('admin.accounting.cashbox', [
                'transactions' => new \Illuminate\Pagination\LengthAwarePaginator([], 0, 50, 1),
                'totalBalance' => 0,
                'totalIncome' => 0,
                'totalExpense' => 0,
                'todayIncome' => 0,
                'todayExpense' => 0,
                'monthlyIncome' => 0,
                'monthlyExpense' => 0,
                'customers' => collect([]),
                'appointments' => collect([]),
                'filters' => [
                    'date_from' => $request->date_from,
                    'date_to' => $request->date_to,
                    'type' => $request->type,
                ]
            ])->with('error', 'Kasa sayfası yüklenirken bir hata oluştu. Lütfen sayfayı yenileyin.');
        }
    }

    public function storeIncome(Request $request)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'date' => 'required|date',
                'time' => 'nullable|string',
                'description' => 'nullable|string',
                'payer_name' => 'nullable|string|max:255',
                'customer_id' => 'nullable|exists:customers,id',
                'reservation_id' => 'nullable|exists:appointments,id',
            ]);

            $validated['date'] = date('Y-m-d', strtotime($validated['date']));

            $cashAccount = Account::getCashAccount(Auth::id());

            if (!empty($validated['payer_name'])) {
                $payerInfo = "Ödeme Yapan: " . $validated['payer_name'];
                if (!empty($validated['description'])) {
                    $validated['description'] = $payerInfo . " | " . $validated['description'];
                } else {
                    $validated['description'] = $payerInfo;
                }
            }

            $validated['account_id'] = $cashAccount->id;
            $validated['type'] = 'income';
            $validated['created_by'] = Auth::id();

        if (!empty($validated['reservation_id']) && empty($validated['customer_id'])) {
            $reservationQuery = Reservation::where('id', $validated['reservation_id']);
            
            // Personel ise sadece kendi randevularını görebilir, salon sahibi ise tüm salon randevularını görebilir
            if (Auth::user()->isPersonel()) {
                $reservationQuery->where('personel_id', Auth::id());
            } elseif (Auth::user()->isGuzellikSalonu()) {
                $reservationQuery->where('beauty_salon_id', Auth::id());
            }
            $reservation = $reservationQuery->first();
            if ($reservation && $reservation->customer_id) {
                $validated['customer_id'] = $reservation->customer_id;
            } else if (!empty($validated['reservation_id'])) {
                
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Seçilen randevu bulunamadı veya yetkiniz yok.');
            }
        }

        if (!empty($validated['customer_id'])) {
            $customerQuery = Customer::where('id', $validated['customer_id']);
            
            // Personel ise sadece kendi müşterilerini görebilir, salon sahibi ise tüm salon müşterilerini görebilir
            if (Auth::user()->isPersonel()) {
                $customerQuery->where('created_by', Auth::id());
            } elseif (Auth::user()->isGuzellikSalonu()) {
                $customerQuery->where('beauty_salon_id', Auth::id());
            }
            $customer = $customerQuery->first();
            if (!$customer) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Seçilen müşteri bulunamadı veya yetkiniz yok.');
            }
        }

        $transaction = Transaction::create($validated);

        if (!empty($validated['customer_id'])) {
            $customer = Customer::find($validated['customer_id']);
            if ($customer) {

                $customer->decrement('balance', $validated['amount']);
            }
        }

            if (!empty($validated['reservation_id'])) {
                $reservationQuery = Reservation::where('id', $validated['reservation_id']);
                
                // Personel ise sadece kendi randevularını görebilir, salon sahibi ise tüm salon randevularını görebilir
                if (Auth::user()->isPersonel()) {
                    $reservationQuery->where('personel_id', Auth::id());
                } elseif (Auth::user()->isGuzellikSalonu()) {
                    $reservationQuery->where('beauty_salon_id', Auth::id());
                }
            $reservation = $reservationQuery->first();
            if ($reservation && isset($reservation->remaining_amount) && $reservation->remaining_amount > 0) {
                $reservation->decrement('remaining_amount', min($validated['amount'], $reservation->remaining_amount));
            }
        }

            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'create_income',
                    'description' => "Gelir eklendi: {$validated['amount']} TL",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }

            try {
                \Log::info('Fatura oluşturma başlatılıyor (Gelir)', [
                    'transaction_id' => $transaction->id,
                    'reservation_id' => $validated['reservation_id'] ?? null,
                    'amount' => $validated['amount'],
                    'type' => $validated['type']
                ]);

                $transaction->refresh();
                
                if (!empty($validated['reservation_id'])) {
                    
                    $invoice = $this->createInvoiceForReservation($validated['reservation_id']);
                    if (!$invoice) {
                        throw new \Exception('Randevu için fatura oluşturulamadı');
                    }
                    \Log::info('Randevu için fatura oluşturuldu (Gelir)', ['invoice_id' => $invoice->id]);
                } else {
                    
                    $invoice = $this->createInvoiceForTransaction($transaction);
                    if (!$invoice) {
                        throw new \Exception('Transaction için fatura oluşturulamadı - createInvoiceForTransaction null döndü');
                    }
                    \Log::info('Transaction için fatura oluşturuldu (Gelir)', [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'transaction_id' => $transaction->id
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Fatura oluşturma hatası (Gelir): ' . $e->getMessage(), [
                    'transaction_id' => $transaction->id,
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ]);
                
            }

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Gelir başarıyla eklendi.',
                    'transaction' => $transaction
                ]);
            }

            return redirect()->route('admin.accounting.cashbox')
                ->with('success', 'Gelir başarıyla eklendi.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation hatası',
                    'errors' => $e->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Store Income Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gelir eklenirken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Gelir eklenirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function storeExpense(Request $request)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'date' => 'required|date',
                'time' => 'nullable|string',
                'description' => 'nullable|string',
                'customer_id' => 'nullable|exists:customers,id',
                'reservation_id' => 'nullable|exists:appointments,id',
            ]);

            $validated['date'] = date('Y-m-d', strtotime($validated['date']));

            $cashAccount = Account::getCashAccount(Auth::id());

            $validated['account_id'] = $cashAccount->id;
            $validated['type'] = 'expense';
            $validated['created_by'] = Auth::id();

            if (!empty($validated['reservation_id']) && (empty($validated['customer_id']) || $validated['customer_id'] == '')) {
                $reservationQuery = Reservation::where('id', $validated['reservation_id']);
                
                // Personel ise sadece kendi randevularını görebilir, salon sahibi ise tüm salon randevularını görebilir
                if (Auth::user()->role === 'personel') {
                    $reservationQuery->where('created_by', Auth::id());
                } elseif (Auth::user()->role === 'güzellik_salonu') {
                    $reservationQuery->where('beauty_salon_id', Auth::id());
                }
                $reservation = $reservationQuery->first();
                if ($reservation && $reservation->customer_id) {
                    $validated['customer_id'] = $reservation->customer_id;
                } else if (!empty($validated['reservation_id'])) {
                    
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Seçilen randevu bulunamadı veya yetkiniz yok.');
                }
            }

            if (!empty($validated['customer_id']) && $validated['customer_id'] != '') {
                $customerQuery = Customer::where('id', $validated['customer_id']);
                
                // Personel ise sadece kendi müşterilerini görebilir, salon sahibi ise tüm salon müşterilerini görebilir
                if (Auth::user()->isPersonel()) {
                    $customerQuery->where('created_by', Auth::id());
                } elseif (Auth::user()->isGuzellikSalonu()) {
                    $customerQuery->where('beauty_salon_id', Auth::id());
                }
                $customer = $customerQuery->first();
                if (!$customer) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Seçilen müşteri bulunamadı veya yetkiniz yok.');
                }
            }

            $transaction = Transaction::create($validated);

            if (!empty($validated['customer_id']) && $validated['customer_id'] != '') {
                $customer = Customer::find($validated['customer_id']);
                if ($customer) {
                    $customer->increment('balance', $validated['amount']);
                }
            }

            if (!empty($validated['reservation_id'])) {
                $reservationQuery = Reservation::where('id', $validated['reservation_id']);
                
                // Personel ise sadece kendi randevularını görebilir, salon sahibi ise tüm salon randevularını görebilir
                if (Auth::user()->isPersonel()) {
                    $reservationQuery->where('personel_id', Auth::id());
                } elseif (Auth::user()->isGuzellikSalonu()) {
                    $reservationQuery->where('beauty_salon_id', Auth::id());
                }
                $reservation = $reservationQuery->first();
                if ($reservation && isset($reservation->remaining_amount)) {
                    $reservation->increment('remaining_amount', $validated['amount']);
                }
            }

            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'create_expense',
                    'description' => "Gider eklendi: {$validated['amount']} TL",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }

            try {
                \Log::info('Fatura oluşturma başlatılıyor (Gider)', [
                    'transaction_id' => $transaction->id,
                    'reservation_id' => $validated['reservation_id'] ?? null,
                    'amount' => $validated['amount'],
                    'type' => $validated['type']
                ]);

                $transaction->refresh();
                
                if (!empty($validated['reservation_id'])) {
                    
                    $invoice = $this->createInvoiceForReservation($validated['reservation_id']);
                    if (!$invoice) {
                        throw new \Exception('Randevu için fatura oluşturulamadı');
                    }
                    \Log::info('Randevu için fatura oluşturuldu (Gider)', ['invoice_id' => $invoice->id]);
                } else {
                    
                    $invoice = $this->createInvoiceForTransaction($transaction);
                    if (!$invoice) {
                        throw new \Exception('Transaction için fatura oluşturulamadı - createInvoiceForTransaction null döndü');
                    }
                    \Log::info('Transaction için fatura oluşturuldu (Gider)', [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'transaction_id' => $transaction->id
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Fatura oluşturma hatası (Gider): ' . $e->getMessage(), [
                    'transaction_id' => $transaction->id,
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ]);
                
            }

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Gider başarıyla eklendi.',
                    'transaction' => $transaction
                ]);
            }

            return redirect()->route('admin.accounting.cashbox')
                ->with('success', 'Gider başarıyla eklendi.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation hatası',
                    'errors' => $e->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Store Expense Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gider eklenirken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Gider eklenirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        try {
            $transaction = Transaction::with(['account', 'customer', 'reservation'])->findOrFail($id);

            $user = Auth::user();
            // Personel ise sadece kendi işlemlerini görebilir, salon sahibi ise tüm salon işlemlerini görebilir
            if ($user->isPersonel() && $transaction->created_by != $user->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işlemi görüntüleme yetkiniz yok.'
                ], 403);
            }
            
            return response()->json([
                'success' => true,
                'transaction' => [
                    'id' => $transaction->id,
                    'amount' => $transaction->amount,
                    'date' => $transaction->date ? $transaction->date->format('Y-m-d') : ($transaction->created_at ? $transaction->created_at->format('Y-m-d') : ''),
                    'description' => $transaction->description,
                    'customer_id' => $transaction->customer_id,
                    'type' => $transaction->type,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'İşlem bulunamadı: ' . $e->getMessage()
            ], 404);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $transaction = Transaction::with(['account', 'customer', 'reservation'])->findOrFail($id);

            $user = Auth::user();
            // Personel ise sadece kendi işlemlerini düzenleyebilir, salon sahibi ise tüm salon işlemlerini düzenleyebilir
            if ($user->isPersonel() && $transaction->created_by != $user->id) {
                return redirect()->back()->with('error', 'Bu işlemi düzenleme yetkiniz yok.');
            }
            
            $oldAmount = $transaction->amount;
            $oldCustomerId = $transaction->customer_id;
            $oldType = $transaction->type;
            
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'date' => 'required|date',
                'description' => 'nullable|string',
                'customer_id' => 'nullable|exists:customers,id',
            ]);
            
            $validated['date'] = date('Y-m-d', strtotime($validated['date']));

            if (!empty($validated['customer_id'])) {
                $customerQuery = Customer::where('id', $validated['customer_id']);
                // Personel ise sadece kendi müşterilerini görebilir, salon sahibi ise tüm salon müşterilerini görebilir
                if ($user->isPersonel()) {
                    $customerQuery->where('created_by', $user->id);
                } elseif ($user->isGuzellikSalonu()) {
                    $customerQuery->where('beauty_salon_id', $user->id);
                }
                $customer = $customerQuery->first();
                if (!$customer) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Seçilen müşteri bulunamadı veya yetkiniz yok.');
                }
            }

            if ($oldCustomerId) {
                $oldCustomer = Customer::find($oldCustomerId);
                if ($oldCustomer) {
                    if ($oldType === 'income') {
                        $oldCustomer->increment('balance', $oldAmount);
                    } else {
                        $oldCustomer->decrement('balance', $oldAmount);
                    }
                }
            }

            $transaction->update($validated);

            if (!empty($validated['customer_id'])) {
                $customer = Customer::find($validated['customer_id']);
                if ($customer) {
                    if ($transaction->type === 'income') {
                        $customer->decrement('balance', $validated['amount']);
                    } else {
                        $customer->increment('balance', $validated['amount']);
                    }
                }
            }

            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'update_transaction',
                    'description' => "İşlem güncellendi: {$transaction->type} - {$validated['amount']} TL",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'İşlem başarıyla güncellendi.',
                    'transaction' => $transaction
                ]);
            }
            
            return redirect()->route('admin.accounting.cashbox')
                ->with('success', 'İşlem başarıyla güncellendi.');
        } catch (\Exception $e) {
            \Log::error('Update Transaction Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'İşlem güncellenirken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'İşlem güncellenirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        $transaction = Transaction::with(['account', 'customer', 'reservation'])->findOrFail($id);

        $user = Auth::user();
        // Personel ise sadece kendi işlemlerini silebilir, salon sahibi ise tüm salon işlemlerini silebilir
        if ($user->isPersonel() && $transaction->created_by != $user->id) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işlemi silme yetkiniz yok.'
                ], 403);
            }
            return redirect()->back()->with('error', 'Bu işlemi silme yetkiniz yok.');
        }
        
        $amount = $transaction->amount;
        $type = $transaction->type;

        if ($type === 'income') {
            
            if ($transaction->customer_id) {
                $customer = Customer::find($transaction->customer_id);
                if ($customer) {
                    $customer->increment('balance', $amount);
                }
            }

            if ($transaction->reservation_id) {
                $reservation = Reservation::find($transaction->reservation_id);
                if ($reservation) {
                    $newRemaining = min($reservation->remaining_amount + $amount, $reservation->total_price);
                    $reservation->update(['remaining_amount' => $newRemaining]);
                }
            }
        } else {
            
            if ($transaction->customer_id) {
                $customer = Customer::find($transaction->customer_id);
                if ($customer) {
                    $customer->decrement('balance', min($amount, $customer->balance));
                }
            }

            if ($transaction->reservation_id) {
                $reservation = Reservation::find($transaction->reservation_id);
                if ($reservation) {
                    $newRemaining = max(0, $reservation->remaining_amount - $amount);
                    $reservation->update(['remaining_amount' => $newRemaining]);
                }
            }
        }
        
        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'delete_transaction',
            'description' => "İşlem silindi: {$type} - {$amount} TL",
            'ip_address' => request()->ip(),
        ]);
        
        $transaction->delete();
        
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'İşlem başarıyla silindi.',
            ]);
        }
        
        return redirect()->route('admin.accounting.cashbox')
            ->with('success', 'İşlem başarıyla silindi.');
    }

    public function report(Request $request)
    {
        try {
            $user = Auth::user();
            $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
            $endDate = $request->get('end_date', now()->format('Y-m-d'));
            
            $query = Transaction::query();

            // Personel ise sadece kendi işlemlerini görsün, salon sahibi ise tüm salon işlemlerini görsün
            if ($user->isPersonel()) {
                $query->where('created_by', $user->id);
            } else {
                // Salon sahibi ise kendi salonunun kasa hesabına ait tüm işlemleri görsün
                $cashAccount = Account::getCashAccount($user->id);
                $query->where('account_id', $cashAccount->id);
            }

            $query->whereBetween('date', [$startDate, $endDate]);

            $totalIncome = (clone $query)->where('type', 'income')->sum('amount');
            $totalExpense = (clone $query)->where('type', 'expense')->sum('amount');
            $netBalance = $totalIncome - $totalExpense;

            $dailyBreakdown = (clone $query)
                ->selectRaw('DATE(date) as day, type, SUM(amount) as total')
                ->groupBy('day', 'type')
                ->orderBy('day')
                ->get()
                ->groupBy('day')
                ->map(function($items, $day) {
                    return [
                        'date' => $day,
                        'income' => $items->where('type', 'income')->sum('total'),
                        'expense' => $items->where('type', 'expense')->sum('total'),
                    ];
                })->values();

            $customerBreakdown = (clone $query)
                ->with('customer:id,name')
                ->whereNotNull('customer_id')
                ->selectRaw('customer_id, type, SUM(amount) as total')
                ->groupBy('customer_id', 'type')
                ->get()
                ->groupBy('customer_id')
                ->map(function($items) {
                    $customer = $items->first()->customer;
                    return [
                        'customer_id' => $items->first()->customer_id,
                        'customer_name' => $customer ? $customer->name : 'Bilinmeyen',
                        'income' => $items->where('type', 'income')->sum('total'),
                        'expense' => $items->where('type', 'expense')->sum('total'),
                    ];
                })->values();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'period' => [
                        'start' => $startDate,
                        'end' => $endDate,
                    ],
                    'summary' => [
                        'total_income' => $totalIncome,
                        'total_expense' => $totalExpense,
                        'net_balance' => $netBalance,
                    ],
                    'daily_breakdown' => $dailyBreakdown,
                    'customer_breakdown' => $customerBreakdown,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Rapor oluşturulurken hata: ' . $e->getMessage()
            ], 500);
        }
    }

    public function recalculate()
    {
        try {
            $user = Auth::user();
            $cashAccount = Account::getCashAccount($user->id);
            $newBalance = $cashAccount->recalculateBalance();
            
            return response()->json([
                'success' => true,
                'message' => 'Kasa bakiyesi yeniden hesaplandı.',
                'balance' => $newBalance
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Hata: ' . $e->getMessage()
            ], 500);
        }
    }

    private function createInvoiceForReservation($reservationId)
    {
        
        $reservation = Reservation::with(['customer', 'salon'])->find($reservationId);
        if (!$reservation) {
            return null;
        }

        $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);

        $publicToken = Str::random(32);

        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $reservation->salon->address ?? ($settings['invoice_address'] ?? '');
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';

        $reservationController = new \App\Http\Controllers\Admin\ReservationController();
        $reflection = new \ReflectionClass($reservationController);
        $method = $reflection->getMethod('prepareInvoiceData');
        $method->setAccessible(true);
        $invoiceData = $method->invoke($reservationController, $reservation, $logo, $address, $taxOffice, $taxNumber);

        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);

        $pdfPath = 'invoices/' . $invoiceNumber . '.pdf';
        Storage::disk('public')->put($pdfPath, $pdf->output());

        $publicUrl = url('/fatura/' . $publicToken);

        $invoice = Invoice::create([
            'reservation_id' => $reservation->id,
            'customer_id' => $reservation->customer_id,
            'invoice_number' => $invoiceNumber,
            'total_amount' => $reservation->total_price,
            'deposit_paid' => $reservation->deposit_paid ?? 0,
            'remaining_amount' => $reservation->remaining_amount ?? ($reservation->total_price - ($reservation->deposit_paid ?? 0)),
            'invoice_date' => now(),
            'pdf_path' => $pdfPath,
            'public_token' => $publicToken,
            'public_url' => $publicUrl,
            'status' => 'sent',
            'items_json' => $invoiceData['items'],
            'created_by' => Auth::id(),
        ]);
        
        return $invoice;
    }

    private function createInvoiceForTransaction($transaction)
    {
        
        $transaction = Transaction::with(['customer', 'account'])->find($transaction->id);
        if (!$transaction) {
            \Log::error('Transaction bulunamadı', ['transaction_id' => $transaction->id ?? 'null']);
            return null;
        }

        $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);

        $publicToken = Str::random(32);

        $itemName = $transaction->type === 'income' ? 'Gelir İşlemi' : 'Gider İşlemi';
        $itemDescription = $transaction->description ?? $itemName;

        $items = [[
            'name' => $itemDescription,
            'quantity' => '-',
            'price' => $transaction->amount,
            'total' => $transaction->amount,
            '_transaction_id' => $transaction->id, 
        ]];

        $publicUrl = url('/fatura/' . $publicToken);

        \Log::info('📝 Fatura oluşturuluyor', [
            'transaction_id' => $transaction->id,
            'invoice_number' => $invoiceNumber,
            'amount' => $transaction->amount,
            'items_json' => $items
        ]);
        
        $invoice = Invoice::create([
            'reservation_id' => null, 
            'customer_id' => $transaction->customer_id,
            'invoice_number' => $invoiceNumber,
            'total_amount' => $transaction->amount,
            'deposit_paid' => $transaction->type === 'income' ? $transaction->amount : 0,
            'remaining_amount' => 0,
            'invoice_date' => $transaction->date ? \Carbon\Carbon::parse($transaction->date) : now(),
            'pdf_path' => null, 
            'public_token' => $publicToken,
            'public_url' => $publicUrl,
            'status' => 'sent',
            'items_json' => $items, 
            'created_by' => Auth::id(),
        ]);
        
        \Log::info('✅ Fatura başarıyla oluşturuldu', [
            'invoice_id' => $invoice->id,
            'invoice_number' => $invoice->invoice_number,
            'transaction_id' => $transaction->id
        ]);
        
        return $invoice;
    }

    public function getInvoiceForTransaction($id)
    {
        try {
            $transaction = Transaction::findOrFail($id);

            $user = Auth::user();
            // Personel ise sadece kendi işlemlerinin faturasını görebilir, salon sahibi ise tüm salon işlemlerinin faturasını görebilir
            if ($user->isPersonel() && $transaction->created_by != $user->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işleme ait faturayı görüntüleme yetkiniz yok.'
                ], 403);
            }
            
            \Log::info('🔍 Fatura aranıyor', [
                'transaction_id' => $transaction->id,
                'amount' => $transaction->amount,
                'created_by' => $transaction->created_by,
                'date' => $transaction->date
            ]);

            $allInvoices = Invoice::where('created_by', $transaction->created_by)
                ->whereNull('reservation_id')
                ->get();
            
            \Log::info('📋 Bulunan faturalar', [
                'count' => $allInvoices->count(),
                'invoice_ids' => $allInvoices->pluck('id')->toArray()
            ]);

            $invoice = $allInvoices->filter(function($inv) use ($transaction) {
                $itemsJson = is_array($inv->items_json) ? $inv->items_json : (is_string($inv->items_json) ? json_decode($inv->items_json, true) : []);

                if (is_array($itemsJson) && isset($itemsJson[0]) && is_array($itemsJson[0])) {
                    if (isset($itemsJson[0]['_transaction_id']) && $itemsJson[0]['_transaction_id'] == $transaction->id) {
                        \Log::info('✅ Fatura bulundu (transaction_id ile)', ['invoice_id' => $inv->id]);
                        return true;
                    }
                }
                return false;
            })->first();

            if (!$invoice) {
                $transactionCreatedAt = $transaction->created_at;
                $invoiceDate = $transaction->date ? $transaction->date->format('Y-m-d') : $transactionCreatedAt->format('Y-m-d');
                
                \Log::info('🔍 Tarih + tutar ile aranıyor', [
                    'invoice_date' => $invoiceDate,
                    'amount' => $transaction->amount,
                    'created_at' => $transactionCreatedAt->format('Y-m-d H:i:s')
                ]);
                
                $invoice = Invoice::where('created_by', $transaction->created_by)
                    ->whereNull('reservation_id')
                    ->where('total_amount', $transaction->amount)
                    ->whereDate('invoice_date', $invoiceDate)
                    ->where('created_at', '>=', $transactionCreatedAt->copy()->subMinutes(10))
                    ->where('created_at', '<=', $transactionCreatedAt->copy()->addHours(2))
                    ->orderBy('created_at', 'asc')
                    ->first();
                
                if ($invoice) {
                    \Log::info('✅ Fatura bulundu (tarih + tutar ile)', ['invoice_id' => $invoice->id]);
                }
            }

            if (!$invoice) {
                \Log::info('🔍 Sadece tutar ile aranıyor');
                
                $invoice = Invoice::where('created_by', $transaction->created_by)
                    ->whereNull('reservation_id')
                    ->where('total_amount', $transaction->amount)
                    ->where('created_at', '>=', $transaction->created_at->copy()->subMinutes(10))
                    ->where('created_at', '<=', $transaction->created_at->copy()->addHours(2))
                    ->orderBy('created_at', 'asc')
                    ->first();
                
                if ($invoice) {
                    \Log::info('✅ Fatura bulundu (sadece tutar ile)', ['invoice_id' => $invoice->id]);
                }
            }

            if (!$invoice) {
                \Log::info('🔍 Son çare: Sadece tutar (tarih yok)');
                
                $invoice = Invoice::where('created_by', $transaction->created_by)
                    ->whereNull('reservation_id')
                    ->where('total_amount', $transaction->amount)
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                if ($invoice) {
                    \Log::info('✅ Fatura bulundu (son çare)', ['invoice_id' => $invoice->id]);
                }
            }
            
            if (!$invoice) {
                \Log::error('❌ Fatura bulunamadı', [
                    'transaction_id' => $transaction->id,
                    'amount' => $transaction->amount,
                    'created_by' => $transaction->created_by
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işlem için fatura bulunamadı. Lütfen yeni bir işlem oluşturun.'
                ], 404);
            }
            
            \Log::info('✅ Fatura bulundu ve döndürülüyor', [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number
            ]);
            
            return response()->json([
                'success' => true,
                'invoice' => [
                    'id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'public_url' => $invoice->public_url,
                    'total_amount' => $invoice->total_amount,
                    'invoice_date' => $invoice->invoice_date->format('d.m.Y'),
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('❌ Get Invoice For Transaction Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Fatura bulunurken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }
}

