<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CustomerSegment;
use App\Models\CustomerClvScore;
use App\Models\PersonalizedOffer;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ClvController extends Controller
{
    // CLV sistemi kaldırıldı
    private const MAINTENANCE_MODE = true;
    
    public function index()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.clv.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        // Müşteri CLV skorları
        $clvScores = CustomerClvScore::where('beauty_salon_id', $user->id)
            ->with(['customer', 'segment'])
            ->orderBy('clv_score', 'desc')
            ->paginate(20);

        // Segment istatistikleri
        $segments = CustomerSegment::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->withCount('clvScores')
            ->get();

        // Toplam istatistikler
        $totalCustomers = Customer::where('beauty_salon_id', $user->id)->count();
        $vipCustomers = CustomerClvScore::where('beauty_salon_id', $user->id)
            ->where('is_vip', true)
            ->count();
        $highRiskCustomers = CustomerClvScore::where('beauty_salon_id', $user->id)
            ->where('churn_risk_level', 'high')
            ->orWhere('churn_risk_level', 'critical')
            ->count();

        return view('admin.clv.index', compact('clvScores', 'segments', 'totalCustomers', 'vipCustomers', 'highRiskCustomers'));
    }

    public function segments()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.clv.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $segments = CustomerSegment::where('beauty_salon_id', $user->id)
            ->withCount('clvScores')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('admin.clv.segments', compact('segments'));
    }

    public function createSegment()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.clv.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        return view('admin.clv.segments.create');
    }

    public function storeSegment(Request $request)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.extensions.index')->with('error', 'Sistem kaldırılmıştır.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'min_clv' => 'nullable|numeric|min:0',
            'max_clv' => 'nullable|numeric|min:0|gte:min_clv',
            'min_visits' => 'nullable|integer|min:0',
            'max_visits' => 'nullable|integer|min:0|gte:min_visits',
            'days_since_last_visit' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $segment = CustomerSegment::create([
            'beauty_salon_id' => $user->id,
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'min_clv' => $validated['min_clv'] ?? null,
            'max_clv' => $validated['max_clv'] ?? null,
            'min_visits' => $validated['min_visits'] ?? null,
            'max_visits' => $validated['max_visits'] ?? null,
            'days_since_last_visit' => $validated['days_since_last_visit'] ?? null,
            'is_active' => $validated['is_active'] ?? true,
            'criteria' => [
                'min_clv' => $validated['min_clv'] ?? null,
                'max_clv' => $validated['max_clv'] ?? null,
                'min_visits' => $validated['min_visits'] ?? null,
                'max_visits' => $validated['max_visits'] ?? null,
                'days_since_last_visit' => $validated['days_since_last_visit'] ?? null,
            ],
        ]);

        return redirect()->route('admin.beauty-salon.clv.segments')
            ->with('success', 'Segment başarıyla oluşturuldu.');
    }

    public function editSegment($id)
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.clv.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $segment = CustomerSegment::where('beauty_salon_id', $user->id)
            ->findOrFail($id);

        return view('admin.clv.segments.edit', compact('segment'));
    }

    public function updateSegment(Request $request, $id)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.extensions.index')->with('error', 'Sistem kaldırılmıştır.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $segment = CustomerSegment::where('beauty_salon_id', $user->id)
            ->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'min_clv' => 'nullable|numeric|min:0',
            'max_clv' => 'nullable|numeric|min:0|gte:min_clv',
            'min_visits' => 'nullable|integer|min:0',
            'max_visits' => 'nullable|integer|min:0|gte:min_visits',
            'days_since_last_visit' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $segment->update([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'min_clv' => $validated['min_clv'] ?? null,
            'max_clv' => $validated['max_clv'] ?? null,
            'min_visits' => $validated['min_visits'] ?? null,
            'max_visits' => $validated['max_visits'] ?? null,
            'days_since_last_visit' => $validated['days_since_last_visit'] ?? null,
            'is_active' => $validated['is_active'] ?? true,
            'criteria' => [
                'min_clv' => $validated['min_clv'] ?? null,
                'max_clv' => $validated['max_clv'] ?? null,
                'min_visits' => $validated['min_visits'] ?? null,
                'max_visits' => $validated['max_visits'] ?? null,
                'days_since_last_visit' => $validated['days_since_last_visit'] ?? null,
            ],
        ]);

        return redirect()->route('admin.beauty-salon.clv.segments')
            ->with('success', 'Segment başarıyla güncellendi.');
    }

    public function destroySegment($id)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.extensions.index')->with('error', 'Sistem kaldırılmıştır.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $segment = CustomerSegment::where('beauty_salon_id', $user->id)
            ->findOrFail($id);

        // Segment'e bağlı CLV skorları varsa segment_id'yi null yap
        CustomerClvScore::where('segment_id', $segment->id)->update(['segment_id' => null]);

        $segment->delete();

        return redirect()->route('admin.beauty-salon.clv.segments')
            ->with('success', 'Segment başarıyla silindi.');
    }

    public function offers()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.clv.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $offers = PersonalizedOffer::where('beauty_salon_id', $user->id)
            ->with(['customer', 'segment'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $segments = CustomerSegment::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->get();

        $services = \App\Models\Service::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('admin.clv.offers', compact('offers', 'segments', 'services'));
    }

    public function calculateClv()
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.extensions.index')->with('error', 'Sistem kaldırılmıştır.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        // Tüm müşteriler için CLV hesapla
        $customers = Customer::where('beauty_salon_id', $user->id)->get();
        
        foreach ($customers as $customer) {
            $this->calculateCustomerClv($customer, $user->id);
        }

        return redirect()->route('admin.beauty-salon.clv.index')
            ->with('success', 'CLV skorları başarıyla hesaplandı.');
    }

    private function calculateCustomerClv($customer, $salonId)
    {
        // Randevu verilerini al
        $appointments = \App\Models\Reservation::where('customer_id', $customer->id)
            ->where('beauty_salon_id', $salonId)
            ->where('status', 'completed')
            ->get();

        $totalSpend = $appointments->sum('total_price') ?? 0;
        $totalVisits = $appointments->count();
        $averageOrderValue = $totalVisits > 0 ? $totalSpend / $totalVisits : 0;
        
        // İlk ve son ziyaret tarihleri
        $firstVisit = $appointments->min('date');
        $lastVisit = $appointments->max('date');
        
        $daysSinceFirstVisit = $firstVisit ? now()->diffInDays($firstVisit) : 0;
        $daysSinceLastVisit = $lastVisit ? now()->diffInDays($lastVisit) : 0;
        
        // Purchase frequency (ayda kaç kez)
        $monthsActive = max(1, $daysSinceFirstVisit / 30);
        $purchaseFrequency = $monthsActive > 0 ? $totalVisits / $monthsActive : 0;
        
        // CLV = Average Order Value × Purchase Frequency × Customer Lifespan (months)
        $customerLifespan = max(1, $monthsActive);
        $clvScore = $averageOrderValue * $purchaseFrequency * $customerLifespan;
        
        // Churn risk hesaplama
        $churnRiskScore = 0;
        if ($daysSinceLastVisit > 90) {
            $churnRiskScore = min(100, 50 + ($daysSinceLastVisit - 90) / 2);
        } elseif ($daysSinceLastVisit > 60) {
            $churnRiskScore = 30;
        } elseif ($daysSinceLastVisit > 30) {
            $churnRiskScore = 15;
        }
        
        $churnRiskLevel = 'low';
        if ($churnRiskScore >= 70) {
            $churnRiskLevel = 'critical';
        } elseif ($churnRiskScore >= 50) {
            $churnRiskLevel = 'high';
        } elseif ($churnRiskScore >= 30) {
            $churnRiskLevel = 'medium';
        }
        
        // VIP kontrolü
        $isVip = $clvScore > 1000 || $totalVisits >= 10;
        
        // Segment belirleme
        $segment = null;
        if ($clvScore > 5000) {
            $segment = CustomerSegment::where('beauty_salon_id', $salonId)
                ->where('slug', 'yuksek-clv')
                ->first();
        } elseif ($clvScore > 1000) {
            $segment = CustomerSegment::where('beauty_salon_id', $salonId)
                ->where('slug', 'orta-clv')
                ->first();
        } else {
            $segment = CustomerSegment::where('beauty_salon_id', $salonId)
                ->where('slug', 'dusuk-clv')
                ->first();
        }
        
        // CLV skorunu kaydet veya güncelle
        CustomerClvScore::updateOrCreate(
            [
                'customer_id' => $customer->id,
                'beauty_salon_id' => $salonId,
            ],
            [
                'segment_id' => $segment?->id,
                'clv_score' => $clvScore,
                'average_order_value' => $averageOrderValue,
                'purchase_frequency' => $purchaseFrequency,
                'total_visits' => $totalVisits,
                'total_spend' => $totalSpend,
                'days_since_first_visit' => $daysSinceFirstVisit,
                'days_since_last_visit' => $daysSinceLastVisit,
                'churn_risk_score' => $churnRiskScore,
                'churn_risk_level' => $churnRiskLevel,
                'is_vip' => $isVip,
                'last_calculated_at' => now(),
            ]
        );
    }
}
