<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Contract;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ContractController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        $globalContract = Contract::getGlobalContract();

        $contracts = Contract::where('user_id', $user->id)
            ->orderBy('is_default', 'desc')
            ->orderBy('name')
            ->get();
            
        return view('admin.settings.contracts.index', compact('contracts', 'globalContract'));
    }

    public function create()
    {
        return view('admin.settings.contracts.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'content' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $validated['user_id'] = Auth::user()->id;
        $validated['company_id'] = Auth::user()->company_id;

        $contract = Contract::create($validated);

        $existingCount = Contract::where('user_id', Auth::user()->id)->count();
        if ($existingCount == 1) {
            $contract->setAsDefault();
        }

        return redirect()->route('admin.settings.contracts.index')
            ->with('success', 'Sözleşme oluşturuldu.');
    }

    public function edit($id)
    {
        $userId = Auth::user()->id;
        $contract = Contract::where('id', $id)
            ->where('user_id', $userId) 
            ->firstOrFail();

        if ($contract->is_required) {
            return redirect()->route('admin.settings.contracts.index')
                ->with('error', 'Bu sözleşme zorunlu sözleşmedir ve düzenlenemez.');
        }
        
        return view('admin.settings.contracts.edit', compact('contract'));
    }

    public function update(Request $request, $id)
    {
        $userId = Auth::user()->id;
        $contract = Contract::where('id', $id)
            ->where('user_id', $userId) 
            ->firstOrFail();

        if ($contract->is_required) {
            return redirect()->route('admin.settings.contracts.index')
                ->with('error', 'Bu sözleşme zorunlu sözleşmedir ve düzenlenemez.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'content' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $contract->update($validated);

        return redirect()->route('admin.settings.contracts.index')
            ->with('success', 'Sözleşme güncellendi.');
    }

    public function destroy($id)
    {
        $userId = Auth::user()->id;
        $contract = Contract::where('id', $id)
            ->where('user_id', $userId) 
            ->firstOrFail();

        if ($contract->is_required) {
            return redirect()->route('admin.settings.contracts.index')
                ->with('error', 'Bu sözleşme zorunlu sözleşmedir ve silinemez.');
        }
        
        if ($contract->is_default) {
            return redirect()->route('admin.settings.contracts.index')
                ->with('error', 'Varsayılan sözleşme silinemez. Önce başka bir sözleşmeyi varsayılan yapın.');
        }
        
        $contract->delete();

        return redirect()->route('admin.settings.contracts.index')
            ->with('success', 'Sözleşme silindi.');
    }

    public function setDefault($id)
    {
        $userId = Auth::user()->id;
        $contract = Contract::where('id', $id)
            ->where('user_id', $userId) 
            ->firstOrFail();
        
        $contract->setAsDefault();

        return redirect()->route('admin.settings.contracts.index')
            ->with('success', 'Sözleşme varsayılan olarak ayarlandı.');
    }

    public function showForReservation($reservationId)
    {
        $reservation = \App\Models\Reservation::with(['customer', 'salon'])->findOrFail($reservationId);
        $userId = Auth::user()->id;

        $contract = Contract::where('user_id', $userId)
            ->where('is_default', true)
            ->where('is_active', true)
            ->first();

        if (!$contract) {
            $contract = Contract::where('user_id', $userId)
                ->where('is_active', true)
                ->first();
        }
        
        if (!$contract) {
            abort(404, 'Aktif sözleşme bulunamadı.');
        }
        
        return view('admin.contracts.show', compact('reservation', 'contract'));
    }

    public function getDefaultContractUrl()
    {
        $userId = Auth::user()->id;

        $contract = Contract::where('user_id', $userId)
            ->where('is_default', true)
            ->where('is_active', true)
            ->first();

        if (!$contract) {
            $contract = Contract::where('user_id', $userId)
                ->where('is_active', true)
                ->first();
        }
        
        if (!$contract) {
            return response()->json(['url' => null, 'message' => 'Varsayılan sözleşme bulunamadı.']);
        }
        
        return response()->json([
            'url' => $contract->public_url,
            'name' => $contract->name,
            'token' => $contract->public_token
        ]);
    }
}

