<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Coupon;
use App\Models\CouponRedemption;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CouponController extends Controller
{
    /**
     * Kupon kodunu uygula (paket seçim sayfasından / modaldan).
     */
    public function apply(Request $request)
    {
        $request->validate([
            'code' => 'required|string|max:64',
        ], [
            'code.required' => 'Kupon kodu giriniz.',
        ]);

        $code = trim(strtoupper($request->input('code')));
        $user = Auth::user();

        if ($user->isSuperAdmin() || $user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Bu işlem için yetkiniz yok.',
            ], 403);
        }

        $coupon = Coupon::where('code', $code)->first();
        if (!$coupon) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz kupon kodu.',
            ], 422);
        }

        if (!$coupon->isValid()) {
            return response()->json([
                'success' => false,
                'message' => 'Bu kupon kodu geçerli değil veya süresi dolmuş.',
            ], 422);
        }

        $alreadyUsed = CouponRedemption::where('user_id', $user->id)->where('coupon_id', $coupon->id)->exists();
        if ($alreadyUsed) {
            return response()->json([
                'success' => false,
                'message' => 'Bu kupon kodunu daha önce kullandınız.',
            ], 422);
        }

        try {
            DB::beginTransaction();

            if ($coupon->hasSubscriptionBenefit()) {
                $base = $user->free_access_ends_at && $user->free_access_ends_at->gt(now())
                    ? $user->free_access_ends_at
                    : now();
                $user->free_access_ends_at = $base->copy()->addDays($coupon->subscription_days);
                $user->save();
            }

            if ($coupon->hasSmsBenefit()) {
                $user->increment('sms_balance', $coupon->sms_credits);
            }

            CouponRedemption::create([
                'user_id' => $user->id,
                'coupon_id' => $coupon->id,
                'used_at' => now(),
            ]);
            $coupon->increment('used_count');

            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            return response()->json([
                'success' => false,
                'message' => 'Kupon uygulanırken bir hata oluştu.',
            ], 500);
        }

        $parts = [];
        if ($coupon->hasSubscriptionBenefit()) {
            $parts[] = $coupon->subscription_days . ' gün ücretsiz kullanım';
        }
        if ($coupon->hasSmsBenefit()) {
            $parts[] = $coupon->sms_credits . ' SMS hakkı';
        }

        return response()->json([
            'success' => true,
            'message' => 'Kupon uygulandı: ' . implode(', ', $parts) . '.',
            'redirect' => route('admin.dashboard'),
        ]);
    }
}
