<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Reservation;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;

class CustomerController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();

            $query = Customer::query();

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }

            if ($user->role === 'güzellik_salonu') {
                
                if (Schema::hasColumn('customers', 'beauty_salon_id')) {
                    $query->where('beauty_salon_id', $user->id);
                } else {
                    
                    $query->where('created_by', $user->id);
                }
            }

            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%");

                    try {
                        $q->orWhere('email', 'like', "%{$search}%");
                    } catch (\Exception $e) {
                        
                    }
                });
            }

            $sortBy = $request->get('sort', 'name_asc'); 
            switch ($sortBy) {
                case 'name_asc':
                    $query->orderBy('name', 'asc');
                    break;
                case 'created_desc':
                    $query->orderBy('created_at', 'desc');
                    break;
                default:
                    $query->orderBy('name', 'asc');
            }

            if ($request->get('format') === 'json' || $request->wantsJson() || $request->ajax()) {
                $customers = $query->get(['id', 'name', 'phone', 'email']);
                return response()->json([
                    'success' => true,
                    'customers' => $customers
                ]);
            }
            
            $customers = $query->paginate(20);

            foreach ($customers as $customer) {
                try {
                    
                    $reservationQuery = Reservation::where('customer_id', $customer->id);
                    if ($user->role === 'randevucu' || $user->role === 'admin') {
                        $reservationQuery->where('created_by', $user->id);
                    }
                    try {
                        $reservationQuery->where('status', '!=', 'cancelled');
                    } catch (\Exception $e) {
                        
                    }
                    $appointments = $reservationQuery->get();

                    $transactionQuery = \App\Models\Transaction::where('customer_id', $customer->id);
                    if ($user->role === 'randevucu' || $user->role === 'admin') {
                        $transactionQuery->where('created_by', $user->id);
                    }

                    $totalDebt = $appointments->sum('total_price') ?? 0;

                    $totalDepositPaid = $appointments->sum('deposit_paid') ?? 0;
                    $totalIncomeTransactions = (clone $transactionQuery)->where('type', 'income')->sum('amount') ?? 0;
                    $totalPaid = $totalDepositPaid + $totalIncomeTransactions;

                    $customer->total_remaining = max(0, $totalDebt - $totalPaid);
                } catch (\Exception $e) {
                    $customer->total_remaining = 0;
                    \Log::warning('Müşteri bakiye hesaplama hatası: ' . $e->getMessage());
                }
            }

            return view('admin.customers.index', compact('customers'));
        } catch (\Exception $e) {
            \Log::error('Customer Index Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);

            try {
                return view('admin.customers.index', [
                    'customers' => new \Illuminate\Pagination\LengthAwarePaginator([], 0, 20, 1)
                ])->with('error', 'Müşteriler yüklenirken bir hata oluştu: ' . $e->getMessage());
            } catch (\Exception $viewError) {
                \Log::error('View render error: ' . $viewError->getMessage());
                return response()->view('errors.500', ['message' => 'Müşteriler yüklenirken bir hata oluştu.'], 500);
            }
        }
    }

    public function create()
    {
        return view('admin.customers.create');
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:20',
                'email' => 'nullable|email|max:255',
                'address' => 'nullable|string',
                'notes' => 'nullable|string',
            ]);

            $customerData = [
                'name' => $validated['name'],
                'phone' => $validated['phone'],
            ];
            
            $user = Auth::user();

            if (Schema::hasColumn('customers', 'created_by')) {
                $customerData['created_by'] = $user->id;
            }

            if (Schema::hasColumn('customers', 'beauty_salon_id') && $user->role === 'güzellik_salonu') {
                $customerData['beauty_salon_id'] = $user->id;
            }

            if (isset($validated['email']) && !empty($validated['email'])) {
                $customerData['email'] = $validated['email'];
            }
            if (isset($validated['address']) && !empty($validated['address'])) {
                $customerData['address'] = $validated['address'];
            }
            if (isset($validated['notes']) && !empty($validated['notes'])) {
                $customerData['notes'] = $validated['notes'];
            }
            
            $customer = Customer::create($customerData);

            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'create_customer',
                    'description' => "Yeni müşteri eklendi: {$customer->name}",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }

            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => true,
                    'message' => 'Müşteri başarıyla eklendi.',
                    'customer' => [
                        'id' => $customer->id,
                        'name' => $customer->name,
                        'phone' => $customer->phone ?? '',
                        'email' => $customer->email ?? '',
                    ]
                ], 200);
            }

            if ($request->has('redirect_back')) {
                return redirect()->back()->with('success', 'Müşteri başarıyla eklendi: ' . $customer->name);
            }
            
            return redirect()->route('admin.customers.index')
                ->with('success', 'Müşteri başarıyla eklendi.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation hatası',
                    'errors' => $e->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Customer Store Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'message' => 'Müşteri oluşturulurken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Müşteri oluşturulurken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function search(Request $request)
    {
        $query = $request->get('q', '');
        
        $user = Auth::user();
        $customerQuery = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $customerQuery->where('created_by', $user->id);
        }

        if (strlen($query) >= 2) {
            $customerQuery->where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('phone', 'like', "%{$query}%");
            });
        }
        
        $customers = $customerQuery
            ->orderBy('name')
            ->limit(20) 
            ->get()
            ->map(function($customer) {
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'phone' => $customer->phone,
                    'email' => $customer->email ?? '',
                    'initials' => $customer->initials ?? '??',
                    'avatar_color' => $customer->avatar_color ?? '#2563eb',
                ];
            });
        
        return response()->json($customers);
    }

    public function show($id)
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $customer = $query->findOrFail($id);

        $reservationQuery = Reservation::where('customer_id', $customer->id)
            ->with(['salon', 'personel', 'room', 'appointmentRequest.service']);
        
        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $reservationQuery->where('created_by', $user->id);
        }
        
        $customer->appointments = $reservationQuery->get();

        $transactionsQuery = \App\Models\Transaction::where('customer_id', $customer->id);
        
        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $transactionsQuery->where('created_by', $user->id);
        }

        $totalAppointments = $customer->appointments->count();
        $totalAmount = $customer->appointments->sum('total_price') ?? 0;

        $totalPaid = $customer->appointments->sum('deposit_paid') ?? 0;
        $transactionPaid = (clone $transactionsQuery)->where('type', 'income')->sum('amount') ?? 0;
        $totalPaid = $totalPaid + $transactionPaid;

        $totalRemaining = max(0, $totalAmount - $totalPaid);
        
        $transactionsOrdered = $transactionsQuery->orderBy('date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        $totalReservationDebt = $customer->appointments->sum('total_price');

        $cumulativePaid = 0;
        foreach ($transactionsOrdered as $index => $transaction) {
            
            $paidBeforeThis = $transactionsOrdered->take($index)->where('type', 'income')->sum('amount');

            if ($transaction->type == 'income') {
                $cumulativePaid = $paidBeforeThis + $transaction->amount;
                
                $transaction->remaining_debt_after = max(0, $totalReservationDebt - $cumulativePaid);
            } else {
                
                $transaction->remaining_debt_after = max(0, $totalReservationDebt - $paidBeforeThis);
            }
        }

        $transactions = $transactionsOrdered->reverse()->values();

        // Referans bilgileri
        $referralCode = \App\Models\SalonRating::where('customer_id', $customer->id)
            ->where('salon_id', $user->beauty_salon_id ?? $user->id)
            ->whereNotNull('referral_code')
            ->latest()
            ->first();
        
        $referrals = \App\Models\Referral::where('referrer_id', $customer->id)
            ->where('beauty_salon_id', $user->beauty_salon_id ?? $user->id)
            ->with('referred')
            ->latest()
            ->get();
        
        $totalReferralRewards = \App\Models\Referral::where('referrer_id', $customer->id)
            ->where('beauty_salon_id', $user->beauty_salon_id ?? $user->id)
            ->where('status', 'completed')
            ->count() * 3; // Her tamamlanan referans için 3 puan

        return view('admin.customers.show', compact('customer', 'totalAppointments', 'totalAmount', 'totalPaid', 'totalRemaining', 'transactions', 'referralCode', 'referrals', 'totalReferralRewards'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        if ($user->role === 'güzellik_salonu' && Schema::hasColumn('customers', 'beauty_salon_id')) {
            $query->where('beauty_salon_id', $user->id);
        }
        
        $customer = $query->findOrFail($id);
        return view('admin.customers.edit', compact('customer'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        if ($user->role === 'güzellik_salonu' && Schema::hasColumn('customers', 'beauty_salon_id')) {
            $query->where('beauty_salon_id', $user->id);
        }
        
        $customer = $query->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        $updateData = [
            'name' => $validated['name'],
            'phone' => $validated['phone'],
        ];

        if (isset($validated['email'])) {
            $updateData['email'] = $validated['email'];
        }
        if (isset($validated['address'])) {
            $updateData['address'] = $validated['address'];
        }
        if (isset($validated['notes'])) {
            $updateData['notes'] = $validated['notes'];
        }
        
        $customer->update($updateData);

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'update_customer',
            'description' => "Müşteri güncellendi: {$customer->name}",
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('admin.customers.index')
            ->with('success', 'Müşteri başarıyla güncellendi.');
    }

    public function toggleLoyaltyCard($id)
    {
        $customer = Customer::findOrFail($id);
        
        // Telefon numarası kontrolü
        if (!$customer->phone) {
            return redirect()->route('admin.customers.show', $id)
                ->with('error', 'Kart programı için telefon numarası gereklidir.');
        }

        $customer->loyalty_card_enabled = !$customer->loyalty_card_enabled;
        $customer->save();

        $status = $customer->loyalty_card_enabled ? 'aktif edildi' : 'devre dışı bırakıldı';
        
        return redirect()->route('admin.customers.show', $id)
            ->with('success', "Kart programı {$status}.");
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        if ($user->role === 'güzellik_salonu' && Schema::hasColumn('customers', 'beauty_salon_id')) {
            $query->where('beauty_salon_id', $user->id);
        }
        
        $customer = $query->findOrFail($id);
        $name = $customer->name;
        $customer->delete();

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'delete_customer',
            'description' => "Müşteri silindi: {$name}",
            'ip_address' => request()->ip(),
        ]);

        return redirect()->route('admin.customers.index')
            ->with('success', 'Müşteri başarıyla silindi.');
    }
}

