<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\CustomerMessage;
use App\Services\SmsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CustomerDirectoryController extends Controller
{
    // Mesaj şablonları
    private function getMessageTemplates()
    {
        return [
            'friday_discount' => [
                'name' => 'Cuma İndirimi',
                'template' => 'Her Cuma %{discount} İndirim Kampanyamızdan Yararlanmayı Unutmayın! 🎉',
                'variables' => ['discount']
            ],
            'weekend_special' => [
                'name' => 'Hafta Sonu Özel',
                'template' => 'Hafta sonu özel fırsatlarımızdan yararlanın! %{discount} indirim sizleri bekliyor. 💅',
                'variables' => ['discount']
            ],
            'new_service' => [
                'name' => 'Yeni Hizmet Duyurusu',
                'template' => 'Yeni hizmetimiz {service_name} artık hizmetinizde! Hemen randevu alın. ✨',
                'variables' => ['service_name']
            ],
            'birthday' => [
                'name' => 'Doğum Günü Mesajı',
                'template' => 'Doğum gününüz kutlu olsun! Size özel %{discount} indirim hediyemiz var. 🎂',
                'variables' => ['discount']
            ],
            'reminder' => [
                'name' => 'Randevu Hatırlatma',
                'template' => 'Sayın {customer_name}, {date} tarihinde {time} saatindeki randevunuzu hatırlatırız. 📅',
                'variables' => ['customer_name', 'date', 'time']
            ],
            'custom' => [
                'name' => 'Özel Mesaj',
                'template' => '',
                'variables' => []
            ],
        ];
    }

    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $customers = Customer::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->paginate(50);

        $templates = $this->getMessageTemplates();
        $sentMessages = CustomerMessage::where('beauty_salon_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.customer-directory.index', compact('customers', 'templates', 'sentMessages'));
    }

    public function sendMessage(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'message_type' => 'required|in:custom,template',
            'template_key' => 'required_if:message_type,template|string',
            'message_content' => 'required|string|max:1000',
            'customer_ids' => 'required|array|min:1',
            'customer_ids.*' => 'exists:customers,id',
            'send_method' => 'required|in:whatsapp,sms',
        ]);

        $smsService = app(SmsService::class);
        $customerIds = $validated['customer_ids'];
        $customers = Customer::whereIn('id', $customerIds)
            ->where('beauty_salon_id', $user->id)
            ->get();

        $totalSent = 0;
        $totalFailed = 0;
        $failedCustomers = [];

        foreach ($customers as $customer) {
            if (!$customer->phone) {
                $totalFailed++;
                $failedCustomers[] = $customer->name;
                continue;
            }

            $message = $validated['message_content'];
            
            // Şablon değişkenlerini değiştir
            $message = str_replace('{customer_name}', $customer->name, $message);
            $message = str_replace('{discount}', $request->input('discount', '10'), $message);
            $message = str_replace('{service_name}', $request->input('service_name', 'Hizmetlerimiz'), $message);
            $message = str_replace('{date}', $request->input('date', date('d.m.Y')), $message);
            $message = str_replace('{time}', $request->input('time', ''), $message);

            try {
                $result = $smsService->send(
                    $customer->phone,
                    $message,
                    null,
                    [
                        'purpose' => 'default',
                        'user_id' => $user->id
                    ]
                );

                if ($result && $result->success) {
                    $totalSent++;
                } else {
                    $totalFailed++;
                    $failedCustomers[] = $customer->name;
                }
            } catch (\Exception $e) {
                $totalFailed++;
                $failedCustomers[] = $customer->name;
            }
        }

        // Mesaj kaydını oluştur
        CustomerMessage::create([
            'beauty_salon_id' => $user->id,
            'message_type' => $validated['message_type'],
            'template_key' => $validated['template_key'] ?? null,
            'message_content' => $validated['message_content'],
            'customer_ids' => $customerIds,
            'total_sent' => $totalSent,
            'total_failed' => $totalFailed,
            'send_method' => $validated['send_method'],
            'sent_at' => now(),
        ]);

        $message = "{$totalSent} müşteriye mesaj gönderildi.";
        if ($totalFailed > 0) {
            $message .= " {$totalFailed} müşteriye mesaj gönderilemedi.";
        }

        return redirect()->route('admin.beauty-salon.customer-directory.index')
            ->with('success', $message)
            ->with('failed_customers', $failedCustomers);
    }

    public function getTemplate(Request $request)
    {
        $templateKey = $request->input('template_key');
        $templates = $this->getMessageTemplates();
        
        if (isset($templates[$templateKey])) {
            return response()->json([
                'success' => true,
                'template' => $templates[$templateKey]['template'],
                'variables' => $templates[$templateKey]['variables']
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Şablon bulunamadı.'
        ], 404);
    }
}
