<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\Customer;
use App\Models\Salon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;

class DashboardController extends Controller
{
    
    protected function getReservationDateColumn(): string
    {
        
        if (\Schema::hasColumn('reservations', 'reservation_date')) {
            return 'reservation_date';
        }
        return 'date';
    }

    public function index()
    {
        try {
            
            Carbon::setLocale('tr');
            date_default_timezone_set('Europe/Istanbul');
            
            $user = auth()->user();

            if ($user && method_exists($user, 'isSuperAdmin') && $user->isSuperAdmin()) {
                return redirect()->route('admin.superadmin.dashboard');
            }

            // Personel hesapları için özel dashboard'a yönlendir
            if ($user->role === 'personel') {
                return redirect()->route('personel.dashboard');
            }

            $hasSalon = false;
            $showSetupWizard = false;
            
            try {
                
                if (!empty($user->salon_name) || !empty($user->salon_slug)) {
                    $hasSalon = true;
                } else {
                    
                    if (class_exists(Salon::class) && \Schema::hasTable('salons')) {
                        $hasSalon = Salon::where('created_by', $user->id)->exists();
                    }
                }
                $showSetupWizard = !$hasSalon;
            } catch (\Exception $e) {
                \Log::warning('Salon kontrolü yapılamadı: ' . $e->getMessage());
                $hasSalon = false;
                $showSetupWizard = false;
            }
            
            $today = now()->setTimezone('Europe/Istanbul')->format('Y-m-d');

            $cashBalance = 0;
            $cashIncome = 0;
            $cashExpense = 0;
            $todayIncome = 0;
            $todayExpense = 0;
            $pendingPayments = 0;
            
            try {
                
                if (class_exists(Account::class) && method_exists(Account::class, 'getCashAccount') && \Schema::hasTable('accounts')) {
                    try {
                        $cashAccount = Account::getCashAccount($user->id);
                        $cashBalance = $cashAccount->balance ?? 0;

                        if (class_exists(Transaction::class) && \Schema::hasTable('transactions')) {
                            $transactionQuery = Transaction::where('account_id', $cashAccount->id);
                            // Personel ise sadece kendi işlemlerini görsün, salon sahibi ise tüm salon işlemlerini görsün
                            if (\Schema::hasColumn('transactions', 'created_by') && $user->role === 'personel') {
                                $transactionQuery->where('created_by', $user->id);
                            }
                            
                            $cashIncome = (clone $transactionQuery)->where('type', 'income')->sum('amount') ?? 0;
                            $cashExpense = (clone $transactionQuery)->where('type', 'expense')->sum('amount') ?? 0;
                            $todayIncome = (clone $transactionQuery)->where('type', 'income')->whereDate('date', $today)->sum('amount') ?? 0;
                            $todayExpense = (clone $transactionQuery)->where('type', 'expense')->whereDate('date', $today)->sum('amount') ?? 0;
                        }

                        if (class_exists(Reservation::class) && \Schema::hasTable('reservations')) {
                            $pendingQuery = Reservation::where('status', '!=', 'cancelled')
                                ->whereRaw('(total_price - COALESCE(deposit_paid, 0)) > 0');
                            if (\Schema::hasColumn('reservations', 'created_by') && ($user->role === 'randevucu' || $user->role === 'admin')) {
                                $pendingQuery->where('created_by', $user->id);
                            }
                            $pendingPayments = $pendingQuery->get()->sum(function($reservation) {
                                return max(0, ($reservation->total_price ?? 0) - ($reservation->deposit_paid ?? 0));
                            });
                        }
                    } catch (\Exception $e) {
                        \Log::warning('Kasa hesabı oluşturulamadı: ' . $e->getMessage());
                    }
                }
            } catch (\Exception $e) {
                \Log::warning('Kasa bilgileri yüklenemedi: ' . $e->getMessage());
            }

            $reservationQuery = null;
            $dateColumn = 'date';
            
            if (class_exists(Reservation::class) && \Schema::hasTable('reservations')) {
                $reservationQuery = Reservation::query();

                if ($user->role === 'güzellik_salonu' && \Schema::hasColumn('reservations', 'beauty_salon_id')) {
                    $reservationQuery->where('beauty_salon_id', $user->id);
                } elseif (\Schema::hasColumn('reservations', 'created_by') && ($user->role === 'randevucu' || $user->role === 'admin')) {
                    $reservationQuery->where('created_by', $user->id);
                }

                $dateColumn = $this->getReservationDateColumn();
            }
            
            $now = Carbon::now()->setTimezone('Europe/Istanbul');
            $stats = [
                'total_appointments' => $reservationQuery ? (clone $reservationQuery)->count() : 0,
                'today_count' => $reservationQuery ? (clone $reservationQuery)->whereDate($dateColumn, $today)->count() : 0,
                'month_count' => $reservationQuery ? (clone $reservationQuery)->whereYear($dateColumn, $now->year)
                    ->whereMonth($dateColumn, $now->month)
                    ->count() : 0,
                'open_count' => $reservationQuery ? (clone $reservationQuery)->where('status', 'open')->count() : 0,
                'confirmed_count' => $reservationQuery ? (clone $reservationQuery)->where('status', 'confirmed')->count() : 0,
                'cancelled_count' => $reservationQuery ? (clone $reservationQuery)->where('status', 'cancelled')->count() : 0,
                'completed_count' => $reservationQuery ? (clone $reservationQuery)->where('status', 'completed')->count() : 0,
                'month_revenue' => $reservationQuery ? (clone $reservationQuery)->whereYear($dateColumn, $now->year)
                    ->whereMonth($dateColumn, $now->month)
                    ->where('status', '!=', 'cancelled')
                    ->sum('total_price') : 0,
            ];

            $last7Days = [];
            $last30Days = [];
            $last12Months = [];
            $last12Weeks = [];

            for ($i = 6; $i >= 0; $i--) {
                $date = now()->setTimezone('Europe/Istanbul')->subDays($i)->format('Y-m-d');
                $count = 0;
                if ($reservationQuery) {
                    try {
                        $query = (clone $reservationQuery)->whereDate($dateColumn, $date);
                        $count = $query->count();
                    } catch (\Exception $e) {
                        $count = 0;
                    }
                }
                $carbonDate = Carbon::now()->setTimezone('Europe/Istanbul')->subDays($i);
                $last7Days[] = [
                    'date' => $carbonDate->locale('tr')->format('d.m'),
                    'count' => $count,
                ];
            }

            for ($i = 29; $i >= 0; $i--) {
                $date = now()->setTimezone('Europe/Istanbul')->subDays($i)->format('Y-m-d');
                $count = 0;
                if ($reservationQuery) {
                    try {
                        $query = (clone $reservationQuery)->whereDate($dateColumn, $date);
                        $count = $query->count();
                    } catch (\Exception $e) {
                        $count = 0;
                    }
                }
                $carbonDate = Carbon::now()->setTimezone('Europe/Istanbul')->subDays($i);
                $last30Days[] = [
                    'date' => $carbonDate->locale('tr')->format('d.m'),
                    'count' => $count,
                ];
            }

            for ($i = 11; $i >= 0; $i--) {
                $weekStart = now()->setTimezone('Europe/Istanbul')->subWeeks($i)->startOfWeek();
                $weekEnd = now()->setTimezone('Europe/Istanbul')->subWeeks($i)->endOfWeek();
                $count = 0;
                if ($reservationQuery) {
                    try {
                        $query = (clone $reservationQuery)
                            ->whereBetween($dateColumn, [$weekStart->format('Y-m-d'), $weekEnd->format('Y-m-d')]);
                        $count = $query->count();
                    } catch (\Exception $e) {
                        $count = 0;
                    }
                }
                $last12Weeks[] = [
                    'date' => $weekStart->locale('tr')->format('d.m') . ' - ' . $weekEnd->locale('tr')->format('d.m'),
                    'count' => $count,
                ];
            }

            for ($i = 11; $i >= 0; $i--) {
                $month = now()->setTimezone('Europe/Istanbul')->subMonths($i);
                $count = 0;
                if ($reservationQuery) {
                    try {
                        $query = (clone $reservationQuery)
                            ->whereYear($dateColumn, $month->year)
                            ->whereMonth($dateColumn, $month->month);
                        $count = $query->count();
                    } catch (\Exception $e) {
                        $count = 0;
                    }
                }
                $last12Months[] = [
                    'date' => $month->locale('tr')->translatedFormat('F Y'),
                    'count' => $count,
                ];
            }

            $recent_appointments = collect([]);
            if ($reservationQuery) {
                try {
                    if ($user->role === 'güzellik_salonu') {
                        $recent_appointments = (clone $reservationQuery)
                            ->with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                            ->orderBy('created_at', 'desc')
                            ->take(10)
                            ->get();
                    } else {
                        $recent_appointments = (clone $reservationQuery)
                            ->with(['customer', 'salon'])
                            ->orderBy('created_at', 'desc')
                            ->take(10)
                            ->get();
                    }
                } catch (\Exception $e) {
                    $recent_appointments = collect([]);
                }
            }

            $monthlyRevenue = 0;
            $monthlyAppointments = 0;
            $weeklyAppointments = 0;
            $weeklyRevenue = 0;
            $totalCustomers = 0;
            $totalRevenue = 0;
            $averageReservationAmount = 0;
            $upcomingAppointments = collect([]);
            $topCustomers = collect([]);
            $last30DaysRevenue = [];
            $lastMonthRevenue = 0;
            $lastMonthAppointments = 0;

            try {
                if ($reservationQuery) {
                    $now = Carbon::now()->setTimezone('Europe/Istanbul');
                    
                    $monthlyAppointments = (clone $reservationQuery)
                        ->whereYear($dateColumn, $now->year)
                        ->whereMonth($dateColumn, $now->month)
                        ->count();

                    $weekStart = $now->copy()->startOfWeek()->toDateString();
                    $weekEnd = $now->copy()->endOfWeek()->toDateString();
                    
                    $weeklyAppointments = (clone $reservationQuery)
                        ->whereBetween($dateColumn, [$weekStart, $weekEnd])
                        ->count();

                    $totalRevenue = (clone $reservationQuery)
                        ->where('status', '!=', 'cancelled')
                        ->sum('total_price') ?? 0;

                    $reservationCount = (clone $reservationQuery)->where('status', '!=', 'cancelled')->count();
                    $averageReservationAmount = $reservationCount > 0 ? ($totalRevenue / $reservationCount) : 0;

                    if ($user->role === 'güzellik_salonu') {
                        $upcomingAppointments = (clone $reservationQuery)
                            ->with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                            ->whereDate($dateColumn, '>=', $today)
                            ->whereDate($dateColumn, '<=', $now->addDays(7)->format('Y-m-d'))
                            ->where('status', '!=', 'cancelled')
                            ->orderBy($dateColumn, 'asc')
                            ->take(5)
                            ->get();
                    } else {
                        $upcomingAppointments = (clone $reservationQuery)
                            ->with(['customer', 'salon'])
                            ->whereDate($dateColumn, '>=', $today)
                            ->whereDate($dateColumn, '<=', $now->addDays(7)->format('Y-m-d'))
                            ->where('status', '!=', 'cancelled')
                            ->orderBy($dateColumn, 'asc')
                            ->take(5)
                            ->get();
                    }

                    $topCustomers = (clone $reservationQuery)
                        ->select('customer_id', DB::raw('count(*) as reservation_count'), DB::raw('sum(total_price) as total_spent'))
                        ->where('status', '!=', 'cancelled')
                        ->groupBy('customer_id')
                        ->orderBy('reservation_count', 'desc')
                        ->take(5)
                        ->get()
                        ->map(function($item) {
                            $customer = Customer::find($item->customer_id);
                            return [
                                'customer' => $customer,
                                'reservation_count' => $item->reservation_count,
                                'total_spent' => $item->total_spent ?? 0
                            ];
                        })
                        ->filter(function($item) {
                            return $item['customer'] !== null;
                        });

                    for ($i = 29; $i >= 0; $i--) {
                        $date = now()->setTimezone('Europe/Istanbul')->subDays($i)->format('Y-m-d');
                        $dayRevenue = (clone $reservationQuery)
                            ->whereDate($dateColumn, $date)
                            ->where('status', '!=', 'cancelled')
                            ->sum('total_price');
                        $carbonDate = Carbon::now()->setTimezone('Europe/Istanbul')->subDays($i);
                        $last30DaysRevenue[] = [
                            'date' => $carbonDate->locale('tr')->format('d.m'),
                            'revenue' => $dayRevenue,
                        ];
                    }

                    $lastMonth = Carbon::now()->setTimezone('Europe/Istanbul')->subMonth();
                    $lastMonthRevenue = (clone $reservationQuery)
                        ->whereYear($dateColumn, $lastMonth->year)
                        ->whereMonth($dateColumn, $lastMonth->month)
                        ->where('status', '!=', 'cancelled')
                        ->sum('total_price');
                    
                    $lastMonthAppointments = (clone $reservationQuery)
                        ->whereYear($dateColumn, $lastMonth->year)
                        ->whereMonth($dateColumn, $lastMonth->month)
                        ->count();
                }

                if (class_exists(Customer::class) && \Schema::hasTable('customers')) {
                    $customerQuery = Customer::query();
                    if (\Schema::hasColumn('customers', 'created_by') && ($user->role === 'randevucu' || $user->role === 'admin')) {
                        $customerQuery->where('created_by', $user->id);
                    }
                    $totalCustomers = $customerQuery->count();
                }
            } catch (\Exception $e) {
                \Log::warning('Ek istatistikler yüklenemedi: ' . $e->getMessage());
            }

            return view('admin.dashboard', compact(
                'stats', 
                'last7Days',
                'last30Days',
                'last12Weeks',
                'last12Months',
                'recent_appointments', 
                'monthlyRevenue', 
                'monthlyAppointments', 
                'weeklyAppointments', 
                'weeklyRevenue',
                'cashBalance',
                'cashIncome',
                'cashExpense',
                'todayIncome',
                'todayExpense',
                'pendingPayments',
                'showSetupWizard',
                'totalCustomers',
                'totalRevenue',
                'averageReservationAmount',
                'upcomingAppointments',
                'topCustomers',
                'last30DaysRevenue',
                'lastMonthRevenue',
                'lastMonthAppointments'
            ));
            
        } catch (\Exception $e) {
            \Log::error('Dashboard Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

            $stats = [
                'total_appointments' => 0,
                'today_count' => 0,
                'month_count' => 0,
                'open_count' => 0,
                'confirmed_count' => 0,
                'cancelled_count' => 0,
                'month_revenue' => 0,
            ];
            $last7Days = [];
            $last30Days = [];
            $last12Weeks = [];
            $last12Months = [];
            $recent_appointments = collect([]);
            $monthlyRevenue = 0;
            $monthlyAppointments = 0;
            $weeklyAppointments = 0;
            $weeklyRevenue = 0;
            $cashBalance = 0;
            $cashIncome = 0;
            $cashExpense = 0;
            $todayIncome = 0;
            $todayExpense = 0;
            $pendingPayments = 0;
            $showSetupWizard = false;
            $totalCustomers = 0;
            $totalRevenue = 0;
            $averageReservationAmount = 0;
            $upcomingAppointments = collect([]);
            $topCustomers = collect([]);
            $last30DaysRevenue = [];
            $lastMonthRevenue = 0;
            $lastMonthAppointments = 0;
            
            return view('admin.dashboard', compact(
                'stats', 
                'last7Days',
                'last30Days',
                'last12Weeks',
                'last12Months',
                'recent_appointments', 
                'monthlyRevenue', 
                'monthlyAppointments', 
                'weeklyAppointments', 
                'weeklyRevenue',
                'cashBalance',
                'cashIncome',
                'cashExpense',
                'todayIncome',
                'todayExpense',
                'pendingPayments',
                'showSetupWizard',
                'totalCustomers',
                'totalRevenue',
                'averageReservationAmount',
                'upcomingAppointments',
                'topCustomers',
                'last30DaysRevenue',
                'lastMonthRevenue',
                'lastMonthAppointments'
            ))->with('error', 'Bazı veriler yüklenirken hata oluştu. Lütfen sayfayı yenileyin.');
        }
    }
}

