<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Reservation;
use App\Models\Setting;
use App\Services\SmsService;
use App\Services\WhatsAppService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class InvoiceController extends Controller
{
    
    public function show($id)
    {
        $reservation = Reservation::with(['customer', 'salon', 'extras'])->findOrFail($id);

        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $settings['invoice_address'] ?? '';
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';

        $salonAddress = $reservation->salon->address ?? $address;

        $invoiceData = $this->prepareInvoiceData($reservation, $logo, $salonAddress, $taxOffice, $taxNumber);

        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        return $pdf->stream('fatura-' . $reservation->id . '.pdf');
    }

    public function showPdf($id)
    {
        $user = Auth::user();
        $invoice = Invoice::with(['reservation.customer', 'reservation.salon', 'reservation.extras', 'customer'])->findOrFail($id);

        if ($user->role !== 'super_admin') {
            if ($user->role === 'randevucu') {
                
                if ($invoice->reservation && $invoice->reservation->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            } else {
                
                if ($invoice->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            }
        }

        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $generalSettings = Setting::where('group', 'general')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $settings['invoice_address'] ?? '';
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        $companyName = $generalSettings['company_name'] ?? 'Salonay';

        if (!$invoice->reservation) {
            
            $invoiceData = $this->prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName);
        } else {
            
            $reservation = $invoice->reservation;
            $salonAddress = $reservation->salon->address ?? $address;
            $invoiceData = $this->prepareInvoiceData($reservation, $logo, $salonAddress, $taxOffice, $taxNumber, $invoice);
        }

        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        return $pdf->stream('fatura-' . $invoice->invoice_number . '.pdf');
    }

    public function downloadPdf($id)
    {
        $user = Auth::user();
        $invoice = Invoice::with(['reservation.customer', 'reservation.salon', 'reservation.extras', 'customer'])->findOrFail($id);

        if ($user->role !== 'super_admin') {
            if ($user->role === 'randevucu') {
                
                if ($invoice->reservation && $invoice->reservation->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            } else {
                
                if ($invoice->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            }
        }

        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $generalSettings = Setting::where('group', 'general')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $settings['invoice_address'] ?? '';
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        $companyName = $generalSettings['company_name'] ?? 'Salonay';

        if (!$invoice->reservation) {
            
            $invoiceData = $this->prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName);
        } else {
            
            $reservation = $invoice->reservation;
            $salonAddress = $reservation->salon->address ?? $address;
            $invoiceData = $this->prepareInvoiceData($reservation, $logo, $salonAddress, $taxOffice, $taxNumber, $invoice);
        }

        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        return $pdf->download('fatura-' . $invoice->invoice_number . '.pdf');
    }

    public function sendSms(Request $request, $id)
    {
        $invoice = Invoice::with(['reservation.customer', 'reservation.salon', 'customer'])->findOrFail($id);
        
        // Müşteri telefon numarasını kontrol et (reservation veya direct customer)
        $customerPhone = null;
        if ($invoice->reservation && $invoice->reservation->customer) {
            $customerPhone = $invoice->reservation->customer->phone;
        } elseif ($invoice->customer) {
            $customerPhone = $invoice->customer->phone;
        }
        
        if (!$customerPhone) {
            return response()->json([
                'success' => false,
                'message' => 'Müşterinin telefon numarası bulunamadı.'
            ], 400);
        }
        
        try {
            $smsService = app(SmsService::class);

            $contract = null;
            
            if ($invoice->reservation && $invoice->reservation->beauty_salon_id) {
                $contract = \App\Models\Contract::where('user_id', $invoice->reservation->beauty_salon_id)
                    ->where('is_default', true)
                    ->where('is_active', true)
                    ->first();
                
                if (!$contract) {
                    $contract = \App\Models\Contract::where('user_id', $invoice->reservation->beauty_salon_id)
                        ->where('is_active', true)
                        ->first();
                }
            } elseif (auth()->user()) {
                $contract = \App\Models\Contract::where('user_id', auth()->user()->id)
                    ->where('is_default', true)
                    ->where('is_active', true)
                    ->first();
                
                if (!$contract) {
                    $contract = \App\Models\Contract::where('user_id', auth()->user()->id)
                        ->where('is_active', true)
                        ->first();
                }
            }

            $invoiceUrl = $invoice->public_url;
            $contractUrl = $contract ? $contract->public_url : null;

            $message = "Fatura bilgileriniz:\n\n";
            $message .= "Fatura No: {$invoice->invoice_number}\n";
            $message .= "Tutar: " . number_format($invoice->total_amount, 2, ',', '.') . " ₺\n";
            
            if ($invoiceUrl) {
                $message .= "\nFatura Linki:\n{$invoiceUrl}\n";
            }
            
            if ($contractUrl) {
                $message .= "\nSözleşme Linki:\n{$contractUrl}\n";
            }
            
            $message .= "\nsalonay.com";

            $result = $smsService->send(
                $customerPhone,
                $message,
                null,
                [
                    'purpose' => 'default',
                    'user_id' => auth()->id()
                ]
            );

            if ($result && $result->success) {
                return response()->json([
                    'success' => true,
                    'message' => 'SMS mesajı başarıyla gönderildi.'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result->message ?? 'SMS mesajı gönderilemedi. Lütfen tekrar deneyin.'
                ], 500);
            }
        } catch (\Exception $e) {
            \Log::error('Fatura SMS gönderme hatası: ' . $e->getMessage(), [
                'invoice_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'SMS mesajı gönderilirken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    public function sendWhatsApp(Request $request, $id)
    {
        // WhatsApp kaldırıldı, SMS'e yönlendir
        return $this->sendSms($request, $id);
    }
    
    private function oldSendWhatsApp(Request $request, $id)
    {
        $invoice = Invoice::with(['reservation.customer', 'reservation.salon', 'customer'])->findOrFail($id);
        
        // Müşteri telefon numarasını kontrol et (reservation veya direct customer)
        $customerPhone = null;
        if ($invoice->reservation && $invoice->reservation->customer) {
            $customerPhone = $invoice->reservation->customer->phone;
        } elseif ($invoice->customer) {
            $customerPhone = $invoice->customer->phone;
        }
        
        if (!$customerPhone) {
            return response()->json([
                'success' => false,
                'message' => 'Müşterinin telefon numarası bulunamadı.'
            ], 400);
        }
        
        try {
            $smsService = app(SmsService::class);

            $contract = null;
            
            if ($invoice->reservation && $invoice->reservation->beauty_salon_id) {
                $contract = \App\Models\Contract::where('user_id', $invoice->reservation->beauty_salon_id)
                    ->where('is_default', true)
                    ->where('is_active', true)
                    ->first();
                
                if (!$contract) {
                    $contract = \App\Models\Contract::where('user_id', $invoice->reservation->beauty_salon_id)
                        ->where('is_active', true)
                        ->first();
                }
            } elseif (auth()->user()) {
                $contract = \App\Models\Contract::where('user_id', auth()->user()->id)
                    ->where('is_default', true)
                    ->where('is_active', true)
                    ->first();
                
                if (!$contract) {
                    $contract = \App\Models\Contract::where('user_id', auth()->user()->id)
                        ->where('is_active', true)
                        ->first();
                }
            }
            
            $contractUrl = $contract ? $contract->public_url : '';

            $reservation = $invoice->reservation;
            $reservationDate = '';
            $reservationTime = '';
            
            if ($reservation) {
                $reservationDate = $reservation->date ? \Carbon\Carbon::parse($reservation->date)->format('d/m/Y') : '';
                if ($reservation->start_time && $reservation->end_time) {
                    $reservationTime = $reservation->start_time . ' - ' . $reservation->end_time;
                } elseif ($reservation->start_time) {
                    $reservationTime = $reservation->start_time;
                }
            }

            $user = auth()->user();
            $salonName = $user->salon_name ?? $user->name ?? 'Salonay';
            $salonAddress = $user->salon_address ?? '';

            $message = $salonName . "\n\n";
            if ($salonAddress) {
                $message .= "Adres: " . $salonAddress . "\n\n";
            }
            if ($reservationDate && $reservationTime) {
                $message .= "Randevu: {$reservationDate} - {$reservationTime}\n\n";
            }
            $message .= "Fatura:\n";
            $message .= $invoice->public_url . "\n\n";
            if ($contractUrl) {
                $message .= "Sözleşme:\n";
                $message .= $contractUrl . "\n\n";
            }
            $message .= "salonay.com iyi günler diler.";

            $result = $smsService->send(
                $customerPhone,
                $message,
                null,
                ['purpose' => 'appointment', 'user_id' => $user->id]
            );
            
            if ($result && $result->success) {
                return response()->json([
                    'success' => true,
                    'message' => 'WhatsApp mesajı başarıyla gönderildi.'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result->message ?? 'WhatsApp mesajı gönderilemedi. Lütfen tekrar deneyin.'
                ], 500);
            }
        } catch (\Exception $e) {
            \Log::error('Fatura WhatsApp gönderme hatası: ' . $e->getMessage(), [
                'invoice_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'WhatsApp mesajı gönderilirken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $query = Invoice::query();

        if ($user->role !== 'super_admin') {
            $query->where('created_by', $user->id);
        }
        
        $invoice = $query->findOrFail($id);

        if ($invoice->pdf_path && Storage::disk('public')->exists($invoice->pdf_path)) {
            Storage::disk('public')->delete($invoice->pdf_path);
        }
        
        $invoice->delete();
        
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Fatura başarıyla silindi.'
            ]);
        }
        
        return redirect()->route('admin.settings.invoices.index')
            ->with('success', 'Fatura başarıyla silindi.');
    }

    private function getPublicUrl($token)
    {
        $url = url('/fatura/' . $token);
        
        // localhost için salonay/public yolunu kaldır
        if (strpos($url, 'localhost') !== false || strpos($url, '127.0.0.1') !== false) {
            $url = str_replace('/salonay/public', '', $url);
        }
        return $url;
    }

    private function prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber, $invoice = null)
    {
        // Salon bilgilerini al
        $salon = null;
        $salonName = 'Güzellik Salonu';
        $salonAddress = $address;
        
        if ($reservation->beauty_salon_id) {
            $salon = \App\Models\User::find($reservation->beauty_salon_id);
            if ($salon) {
                $salonName = $salon->salon_name ?? $salon->name ?? 'Güzellik Salonu';
                $salonAddress = $salon->salon_address ?? $salon->address ?? $address;
            }
        } elseif ($reservation->salon) {
            $salon = $reservation->salon;
            $salonName = $salon->salon_name ?? $salon->name ?? 'Güzellik Salonu';
            $salonAddress = $salon->salon_address ?? $salon->address ?? $address;
        } else {
            // Eğer salon bilgisi yoksa, faturayı oluşturan kullanıcıdan al
            $user = Auth::user();
            if ($user) {
                $salonName = $user->salon_name ?? $user->name ?? 'Güzellik Salonu';
                $salonAddress = $user->salon_address ?? $user->address ?? $address;
            }
        }
        
        $items = [];
        $itemNumber = 0;

        // Hizmet bilgisini bul
        $serviceName = 'Hizmet';
        $reservation->load('appointmentRequest.service');
        
        if ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
            $serviceName = $reservation->appointmentRequest->service->name;
        } else {
            $itemsData = json_decode($reservation->items_data ?? '{}', true);
            if (isset($itemsData['service']) && is_array($itemsData['service'])) {
                $serviceName = $itemsData['service']['name'] ?? 'Hizmet';
            } elseif (isset($itemsData['services']) && is_array($itemsData['services']) && !empty($itemsData['services'])) {
                $firstService = $itemsData['services'][0];
                $serviceName = $firstService['name'] ?? 'Hizmet';
            }
        }

        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if (!is_array($itemsData) || empty($itemsData)) {
            $itemsData = json_decode($reservation->notes ?? '{}', true);
            if (!is_array($itemsData)) {
                $itemsData = [];
            }
        }

        $mealTotal = 0;
        $organizationPrice = $itemsData['organization_price'] ?? 0;
        $extraTotal = 0;

        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $mealTotal += $total;
                }
            }
        }

        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $extraTotal += $extra['price'];
                }
            }
        }

        $salonPrice = $reservation->total_price - ($mealTotal + $organizationPrice + $extraTotal);

        // Hizmet kalemini ekle (toplam fiyat - yemek - organizasyon - extra)
        if ($salonPrice > 0) {
            $items[] = [
                'name' => $serviceName,
                'quantity' => 1,
                'price' => $salonPrice,
                'total' => $salonPrice
            ];
        }

        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $items[] = [
                        'name' => $meal['name'],
                        'quantity' => $meal['quantity'] . ' Kisi',
                        'price' => $meal['price'],
                        'total' => $total
                    ];
                }
            }
        }

        if ($organizationPrice > 0) {
            $items[] = [
                'name' => 'Organizasyon',
                'quantity' => '-',
                'price' => $organizationPrice,
                'total' => $organizationPrice
            ];
        }

        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $items[] = [
                        'name' => $extra['name'],
                        'quantity' => '-',
                        'price' => $extra['price'],
                        'total' => $extra['price']
                    ];
                }
            }
        }

        $grandTotal = $reservation->total_price;
        $depositPaid = $reservation->deposit_paid ?? 0;

        // Salon bilgilerini al (beauty_salon_id üzerinden User modelinden)
        $salonName = 'Güzellik Salonu';
        $salonAddress = $address;
        
        if ($reservation->beauty_salon_id) {
            $beautySalon = \App\Models\User::find($reservation->beauty_salon_id);
            if ($beautySalon) {
                $salonName = $beautySalon->salon_name ?? $beautySalon->name ?? 'Güzellik Salonu';
                $salonAddress = $beautySalon->salon_address ?? $beautySalon->address ?? $address;
            }
        } elseif ($reservation->salon) {
            // Fallback: eski salon ilişkisi varsa (Salon modeli)
            $salonName = $reservation->salon->name ?? 'Güzellik Salonu';
            $salonAddress = $reservation->salon->address ?? $address;
        } else {
            // Son çare: faturayı oluşturan kullanıcıdan al
            $user = Auth::user();
            if ($user) {
                $salonName = $user->salon_name ?? $user->name ?? 'Güzellik Salonu';
                $salonAddress = $user->salon_address ?? $user->address ?? $address;
            }
        }
        
        return [
            'reservation' => $reservation,
            'invoice' => $invoice,
            'items' => $items,
            'subtotal' => $grandTotal,
            'grandTotal' => $grandTotal,
            'depositPaid' => $depositPaid,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $salonName,
            'salonAddress' => $salonAddress,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
        ];
    }

    private function prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName)
    {
        
        $items = $invoice->items_json ?? [];

        $cleanItems = [];
        foreach ($items as $item) {
            if (is_array($item)) {
                $cleanItems[] = [
                    'name' => $item['name'] ?? 'İşlem',
                    'quantity' => $item['quantity'] ?? '-',
                    'price' => $item['price'] ?? 0,
                    'total' => $item['total'] ?? ($item['price'] ?? 0),
                ];
            }
        }

        if (empty($cleanItems)) {
            $cleanItems[] = [
                'name' => 'İşlem',
                'quantity' => '-',
                'price' => $invoice->total_amount,
                'total' => $invoice->total_amount,
            ];
        }

        $customerName = $invoice->customer ? $invoice->customer->name : 'Müşteri';

        $invoiceDate = $invoice->invoice_date ?? $invoice->created_at;
        
        return [
            'reservation' => null,
            'transaction' => true,
            'invoice' => $invoice,
            'items' => $cleanItems,
            'subtotal' => $invoice->total_amount,
            'grandTotal' => $invoice->total_amount,
            'depositPaid' => $invoice->deposit_paid ?? 0,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $companyName,
            'salonAddress' => $address,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
            'customerName' => $customerName,
            'invoiceDate' => $invoiceDate,
        ];
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'reservation_id' => 'required|exists:appointments,id',
        ]);

        $reservation = Reservation::with(['customer', 'salon'])->findOrFail($validated['reservation_id']);

        $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);

        $publicToken = Str::random(32);

        $invoiceData = $this->prepareInvoiceData(
            $reservation,
            Setting::where('group', 'invoice')->where('key', 'invoice_logo')->value('value'),
            $reservation->salon->address ?? Setting::where('group', 'invoice')->where('key', 'invoice_address')->value('value') ?? '',
            Setting::where('group', 'invoice')->where('key', 'invoice_tax_office')->value('value') ?? '',
            Setting::where('group', 'invoice')->where('key', 'invoice_tax_number')->value('value') ?? ''
        );

        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);

        $pdfPath = 'invoices/' . $invoiceNumber . '.pdf';
        Storage::disk('public')->put($pdfPath, $pdf->output());

        $invoice = Invoice::create([
            'reservation_id' => $reservation->id,
            'customer_id' => $reservation->customer_id,
            'invoice_number' => $invoiceNumber,
            'total_amount' => $reservation->total_price,
            'deposit_paid' => $reservation->deposit_paid ?? 0,
            'remaining_amount' => $reservation->remaining_amount ?? ($reservation->total_price - ($reservation->deposit_paid ?? 0)),
            'invoice_date' => now(),
            'pdf_path' => $pdfPath,
            'public_token' => $publicToken,
            'public_url' => $this->getPublicUrl($publicToken),
            'status' => 'sent', 
            'items_json' => $invoiceData['items'],
            'created_by' => Auth::id(),
        ]);

        if ($request->has('send_sms') && $request->send_sms == '1' && $reservation->customer->phone) {
            $user = Auth::user();
            if ($user && !$user->isSuperAdmin()) {
                if ($user->hasSmsBalance(1)) {
                    $smsService = app(\App\Services\SmsService::class);
                    $invoiceUrl = $invoice->public_url; 
                    $message = "Fatura linkiniz: {$invoiceUrl}";
                    $smsService->send(
                        $reservation->customer->phone,
                        $message,
                        null,
                        ['purpose' => 'invoice', 'user_id' => $user->id]
                    );
                }
            } else {
                $smsService = app(\App\Services\SmsService::class);
                $invoiceUrl = $invoice->public_url; 
                $message = "Fatura linkiniz: {$invoiceUrl}";
                $smsService->send(
                    $reservation->customer->phone,
                    $message,
                    null,
                    ['purpose' => 'invoice', 'user_id' => $user->id]
                );
            }
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Fatura başarıyla oluşturuldu.',
            'invoice_id' => $invoice->id,
            'invoice_number' => $invoice->invoice_number,
            'public_url' => $invoice->public_url,
        ]);
    }

    public function index(Request $request)
    {
        $user = Auth::user();
        $query = Invoice::with(['reservation', 'customer', 'creator']);

        $sortBy = $request->get('sort', 'created_desc'); 
        $needsJoin = ($sortBy === 'name_asc');
        
        if ($needsJoin) {
            
            $query->leftJoin('customers', 'invoices.customer_id', '=', 'customers.id')
                  ->select('invoices.*');
        }

        if ($request->filled('start_date')) {
            $startDate = $request->get('start_date');
            if ($needsJoin) {
                $query->whereDate('invoices.invoice_date', '>=', $startDate);
            } else {
                $query->whereDate('invoice_date', '>=', $startDate);
            }
        }
        
        if ($request->filled('end_date')) {
            $endDate = $request->get('end_date');
            if ($needsJoin) {
                $query->whereDate('invoices.invoice_date', '<=', $endDate);
            } else {
                $query->whereDate('invoice_date', '<=', $endDate);
            }
        }

        if ($user->role !== 'super_admin') {
            if ($user->role === 'randevucu') {

                $query->where(function($q) use ($user, $needsJoin) {
                    
                    if ($needsJoin) {
                        $q->where('invoices.created_by', $user->id);
                    } else {
                        $q->where('created_by', $user->id);
                    }
                });
            } else {

                if ($needsJoin) {
                    $query->where('invoices.created_by', $user->id);
                } else {
                    $query->where('created_by', $user->id);
                }
            }
        }

        switch ($sortBy) {
            case 'name_asc':
                
                $query->orderBy('customers.name', 'asc');
                break;
            case 'created_desc':
                
                if ($needsJoin) {
                    $query->orderBy('invoices.created_at', 'desc');
                } else {
                    $query->orderBy('created_at', 'desc');
                }
                break;
            default:
                if ($needsJoin) {
                    $query->orderBy('invoices.created_at', 'desc');
                } else {
                    $query->orderBy('created_at', 'desc');
                }
        }
        
        $invoices = $query->paginate(10);

        $totalAmountQuery = Invoice::query();
        if ($user->role !== 'super_admin') {
            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $totalAmountQuery->where('created_by', $user->id);
            }
        }
        $totalAmount = $totalAmountQuery->sum('total_amount');
        
        return view('admin.settings.invoices.index', compact('invoices', 'totalAmount'));
    }
}
