<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class LoyaltyController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = auth()->user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $tiers = \App\Models\LoyaltyTier::where('beauty_salon_id', $user->id)->orderBy('sort_order')->get();
        $rewards = \App\Models\LoyaltyReward::where('beauty_salon_id', $user->id)->where('is_active', true)->get();
        $customerLoyalties = \App\Models\CustomerLoyalty::where('beauty_salon_id', $user->id)
            ->with(['customer', 'loyaltyTier'])
            ->orderBy('total_points', 'desc')
            ->paginate(20);

        $totalPoints = \App\Models\CustomerLoyalty::where('beauty_salon_id', $user->id)->sum('total_points');
        $totalCustomers = \App\Models\CustomerLoyalty::where('beauty_salon_id', $user->id)->count();
        $activeTiers = $tiers->where('is_active', true)->count();

        return view('admin.loyalty.index', compact('tiers', 'rewards', 'customerLoyalties', 'totalPoints', 'totalCustomers', 'activeTiers'));
    }

    /**
     * Display loyalty tiers.
     */
    public function tiers()
    {
        $user = auth()->user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $tiers = \App\Models\LoyaltyTier::where('beauty_salon_id', $user->id)
            ->orderBy('sort_order')
            ->get();

        return view('admin.loyalty.tiers', compact('tiers'));
    }

    /**
     * Display loyalty rewards.
     */
    public function rewards()
    {
        $user = auth()->user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $rewards = \App\Models\LoyaltyReward::where('beauty_salon_id', $user->id)
            ->orderBy('points_required')
            ->get();

        return view('admin.loyalty.rewards', compact('rewards'));
    }

    /**
     * Display loyalty customers.
     */
    public function customers()
    {
        $user = auth()->user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $customerLoyalties = \App\Models\CustomerLoyalty::where('beauty_salon_id', $user->id)
            ->with(['customer', 'loyaltyTier'])
            ->orderBy('total_points', 'desc')
            ->paginate(20);

        return view('admin.loyalty.customers', compact('customerLoyalties'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Store a newly created loyalty tier.
     */
    public function storeTier(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'level' => 'required|integer|min:1',
            'min_visits' => 'nullable|integer|min:0',
            'min_spend' => 'nullable|numeric|min:0',
            'points_per_visit' => 'nullable|numeric|min:0',
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
        ]);

        $tier = \App\Models\LoyaltyTier::create([
            'beauty_salon_id' => $user->id,
            'name' => $validated['name'],
            'level' => $validated['level'],
            'min_visits' => $validated['min_visits'] ?? 0,
            'min_spend' => $validated['min_spend'] ?? 0,
            'points_per_visit' => $validated['points_per_visit'] ?? 1,
            'discount_percentage' => $validated['discount_percentage'] ?? 0,
            'is_active' => true,
            'sort_order' => $validated['level'],
        ]);

        return redirect()->route('admin.beauty-salon.loyalty.tiers')
            ->with('success', 'Kart seviyesi başarıyla oluşturuldu.');
    }

    /**
     * Store a newly created loyalty reward.
     */
    public function storeReward(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isBeautySalon()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'points_required' => 'required|numeric|min:0',
            'reward_type' => 'nullable|string|in:discount,free_service,points,cash',
            'reward_value' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
        ]);

        $reward = \App\Models\LoyaltyReward::create([
            'beauty_salon_id' => $user->id,
            'name' => $validated['name'],
            'points_required' => $validated['points_required'],
            'reward_type' => $validated['reward_type'] ?? 'discount',
            'reward_value' => $validated['reward_value'] ?? 0,
            'description' => $validated['description'] ?? null,
            'is_active' => true,
        ]);

        return redirect()->route('admin.beauty-salon.loyalty.rewards')
            ->with('success', 'Kart ödülü başarıyla oluşturuldu.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
