<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MembershipPackage;
use App\Models\CustomerMembership;
use App\Models\Customer;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MembershipController extends Controller
{
    // Üyelik sistemi kaldırıldı
    private const MAINTENANCE_MODE = true;
    
    public function index()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.memberships.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $packages = MembershipPackage::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('price')
            ->get();
        
        $activeMemberships = CustomerMembership::where('beauty_salon_id', $user->id)
            ->where('status', 'active')
            ->with(['customer', 'membershipPackage'])
            ->latest()
            ->paginate(20);

        return view('admin.memberships.index', compact('packages', 'activeMemberships'));
    }

    public function packages()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.memberships.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $packages = MembershipPackage::where('beauty_salon_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();
        
        $services = Service::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('admin.memberships.packages', compact('packages', 'services'));
    }

    public function customers()
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.memberships.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $memberships = CustomerMembership::where('beauty_salon_id', $user->id)
            ->with(['customer', 'membershipPackage'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.memberships.customers', compact('memberships'));
    }

    public function storePackage(Request $request)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.extensions.index')->with('error', 'Sistem kaldırılmıştır.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'period' => 'required|in:monthly,quarterly,yearly',
            'price' => 'required|numeric|min:0',
            'service_limit' => 'nullable|integer|min:0',
            'amount_limit' => 'nullable|numeric|min:0',
            'allow_rollover' => 'boolean',
            'rollover_percentage' => 'nullable|integer|min:0|max:100',
            'priority_booking' => 'boolean',
            'included_services' => 'nullable|array',
            'excluded_services' => 'nullable|array',
        ]);

        $validated['beauty_salon_id'] = $user->id;
        $validated['allow_rollover'] = $request->has('allow_rollover');
        $validated['priority_booking'] = $request->has('priority_booking');

        MembershipPackage::create($validated);

        return redirect()->route('admin.beauty-salon.memberships.packages')
            ->with('success', 'Üyelik paketi başarıyla oluşturuldu.');
    }

    public function updatePackage(Request $request, $id)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.extensions.index')->with('error', 'Sistem kaldırılmıştır.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $package = MembershipPackage::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'period' => 'required|in:monthly,quarterly,yearly',
            'price' => 'required|numeric|min:0',
            'service_limit' => 'nullable|integer|min:0',
            'amount_limit' => 'nullable|numeric|min:0',
            'allow_rollover' => 'boolean',
            'rollover_percentage' => 'nullable|integer|min:0|max:100',
            'priority_booking' => 'boolean',
            'included_services' => 'nullable|array',
            'excluded_services' => 'nullable|array',
            'is_active' => 'boolean',
        ]);

        $validated['allow_rollover'] = $request->has('allow_rollover');
        $validated['priority_booking'] = $request->has('priority_booking');
        $validated['is_active'] = $request->has('is_active');

        $package->update($validated);

        return redirect()->route('admin.beauty-salon.memberships.packages')
            ->with('success', 'Üyelik paketi başarıyla güncellendi.');
    }

    public function showPackage($id)
    {
        if (self::MAINTENANCE_MODE) {
            return view('admin.memberships.maintenance');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $package = MembershipPackage::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();

        if (request()->wantsJson() || request()->ajax()) {
            return response()->json($package);
        }

        return view('admin.memberships.package-show', compact('package'));
    }

    public function destroyPackage($id)
    {
        if (self::MAINTENANCE_MODE) {
            return redirect()->route('admin.extensions.index')->with('error', 'Sistem kaldırılmıştır.');
        }
        
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $package = MembershipPackage::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();

        // Aktif üyelikleri kontrol et
        $activeMemberships = CustomerMembership::where('membership_package_id', $package->id)
            ->where('status', 'active')
            ->count();

        if ($activeMemberships > 0) {
            return redirect()->route('admin.beauty-salon.memberships.packages')
                ->with('error', 'Bu pakete ait aktif üyelikler bulunmaktadır. Önce üyelikleri iptal edin.');
        }

        $package->delete();

        return redirect()->route('admin.beauty-salon.memberships.packages')
            ->with('success', 'Üyelik paketi başarıyla silindi.');
    }
}
