<?php

namespace App\Http\Controllers\Admin\Mobile;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Reservation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MobileCustomerController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $customers = $query->orderBy('name')->get();
        
        return view('admin.mobile.customers.index', compact('customers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        $customer = Customer::create([
            'name' => $validated['name'],
            'phone' => $validated['phone'],
            'email' => $validated['email'] ?? null,
            'address' => $validated['address'] ?? null,
            'notes' => $validated['notes'] ?? null,
            'created_by' => Auth::id(),
            'beauty_salon_id' => Auth::id(),
        ]);

        return redirect()->route('mobile.customers.show', $customer->id)
            ->with('success', 'Müşteri başarıyla oluşturuldu.');
    }

    public function show($id)
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $customer = $query->findOrFail($id);

        $reservationQuery = Reservation::where('customer_id', $customer->id)
            ->with(['salon', 'personel', 'room', 'appointmentRequest.service']);
        
        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $reservationQuery->where('created_by', $user->id);
        }
        
        $appointments = $reservationQuery->orderBy('date', 'desc')->get();

        $transactionsQuery = \App\Models\Transaction::where('customer_id', $customer->id);
        
        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $transactionsQuery->where('created_by', $user->id);
        }

        $totalAppointments = $appointments->count();
        $totalAmount = $appointments->sum('total_price') ?? 0;

        $totalPaid = $appointments->sum('deposit_paid') ?? 0;
        $transactionPaid = (clone $transactionsQuery)->where('type', 'income')->sum('amount') ?? 0;
        $totalPaid = $totalPaid + $transactionPaid;

        $totalRemaining = max(0, $totalAmount - $totalPaid);
        
        $transactionsOrdered = $transactionsQuery->orderBy('date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        $totalReservationDebt = $appointments->sum('total_price');

        $cumulativePaid = 0;
        foreach ($transactionsOrdered as $index => $transaction) {
            
            $paidBeforeThis = $transactionsOrdered->take($index)->where('type', 'income')->sum('amount');

            if ($transaction->type == 'income') {
                $cumulativePaid = $paidBeforeThis + $transaction->amount;
                
                $transaction->remaining_debt_after = max(0, $totalReservationDebt - $cumulativePaid);
            } else {
                
                $transaction->remaining_debt_after = max(0, $totalReservationDebt - $paidBeforeThis);
            }
        }

        $transactions = $transactionsOrdered->reverse()->values();

        return view('admin.mobile.customers.show', compact('customer', 'appointments', 'totalAppointments', 'totalAmount', 'totalPaid', 'totalRemaining', 'transactions'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $customer = $query->findOrFail($id);
        return view('admin.mobile.customers.edit', compact('customer'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = Customer::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $customer = $query->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        $updateData = [
            'name' => $validated['name'],
            'phone' => $validated['phone'],
        ];

        if (isset($validated['email'])) {
            $updateData['email'] = $validated['email'];
        }
        if (isset($validated['address'])) {
            $updateData['address'] = $validated['address'];
        }
        if (isset($validated['notes'])) {
            $updateData['notes'] = $validated['notes'];
        }
        
        $customer->update($updateData);

        return redirect()->route('mobile.customers.show', $customer->id)
            ->with('success', 'Müşteri başarıyla güncellendi.');
    }
}

