<?php

namespace App\Http\Controllers\Admin\Mobile;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MobileReservationController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        if ($user->role === 'güzellik_salonu') {
            $reservations = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                ->where('beauty_salon_id', $user->id)
                ->orderBy('date', 'desc')
                ->orderBy('start_time', 'desc')
                ->get();
        } else {
            
            $query = Reservation::with(['customer', 'salon', 'appointmentRequest.service', 'personel'])
                ->orderBy('date', 'desc')
                ->orderBy('start_time', 'desc');
            
            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $reservations = $query->get();
        }
        
        return view('admin.mobile.appointments.index', compact('reservations'));
    }

    public function create()
    {
        $user = Auth::user();

        if ($user->role === 'güzellik_salonu') {
            $customers = Customer::where('beauty_salon_id', $user->id)
                ->orderBy('name')
                ->get();
        } else {
            $customers = Customer::query();
            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $customers->where('created_by', $user->id);
            }
            $customers = $customers->orderBy('name')->get();
        }
        
        return view('admin.mobile.appointments.create', compact('customers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'date' => 'required|date',
            'start_time' => 'required',
            'end_time' => 'required',
            'total_price' => 'required|numeric|min:0',
            'deposit_paid' => 'nullable|numeric|min:0',
            'status' => 'nullable|in:open,confirmed,cancelled,completed',
            'notes' => 'nullable|string',
        ]);

        $user = Auth::user();

        $customer = Customer::findOrFail($validated['customer_id']);
        if ($user->role === 'güzellik_salonu' && $customer->beauty_salon_id !== $user->id) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Bu müşteriye erişim yetkiniz yok.');
        }
        if (($user->role === 'randevucu' || $user->role === 'admin') && $customer->created_by !== $user->id) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Bu müşteriye erişim yetkiniz yok.');
        }

        $reservation = Reservation::create([
            'customer_id' => $validated['customer_id'],
            'beauty_salon_id' => $user->role === 'güzellik_salonu' ? $user->id : null,
            'date' => $validated['date'],
            'start_time' => $validated['start_time'],
            'end_time' => $validated['end_time'],
            'total_price' => $validated['total_price'],
            'deposit_paid' => $validated['deposit_paid'] ?? 0,
            'remaining_amount' => $validated['total_price'] - ($validated['deposit_paid'] ?? 0),
            'status' => $validated['status'] ?? 'open',
            'notes' => $validated['notes'] ?? null,
            'created_by' => $user->id,
        ]);

        try {
            ActivityLog::create([
                'user_id' => $user->id,
                'action' => 'create_reservation',
                'description' => "Randevu oluşturuldu: ID {$reservation->id}",
                'ip_address' => request()->ip(),
            ]);
        } catch (\Exception $e) {
            \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
        }

        return redirect()->route('mobile.appointments.show', $reservation->id)
            ->with('success', 'Randevu başarıyla oluşturuldu.');
    }

    public function show($id)
    {
        $user = Auth::user();
        $query = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room']);
        
        if ($user->role === 'güzellik_salonu') {
            $query->where('beauty_salon_id', $user->id);
        } elseif ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        
        return view('admin.mobile.appointments.show', compact('reservation'));
    }

    public function complete($id)
    {
        $user = Auth::user();
        $query = Reservation::query();
        
        if ($user->role === 'güzellik_salonu') {
            $query->where('beauty_salon_id', $user->id);
        } elseif ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        
        $reservation->update([
            'status' => 'completed'
        ]);
        
        try {
            ActivityLog::create([
                'user_id' => $user->id,
                'action' => 'complete_reservation',
                'description' => "Randevu tamamlandı: ID {$reservation->id}",
                'ip_address' => request()->ip(),
            ]);
        } catch (\Exception $e) {
            \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
        }
        
        return redirect()->back()->with('success', 'Randevu başarıyla tamamlandı.');
    }

    public function cancel($id)
    {
        $user = Auth::user();
        $query = Reservation::query();
        
        if ($user->role === 'güzellik_salonu') {
            $query->where('beauty_salon_id', $user->id);
        } elseif ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        
        $reservation->update([
            'status' => 'cancelled'
        ]);
        
        try {
            ActivityLog::create([
                'user_id' => $user->id,
                'action' => 'cancel_reservation',
                'description' => "Randevu iptal edildi: ID {$reservation->id}",
                'ip_address' => request()->ip(),
            ]);
        } catch (\Exception $e) {
            \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
        }
        
        return redirect()->back()->with('success', 'Randevu başarıyla iptal edildi.');
    }
}

