<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Reservation;
use App\Models\AppointmentRequest;
use App\Models\Service;
use Carbon\Carbon;

class NotificationController extends Controller
{
    public function index()
    {
        try {
            $user = auth()->user();
            $notifications = [];

        // Laravel notifications tablosundan bildirimleri al
        if ($user) {
            $dbNotifications = $user->notifications()
                ->where('type', 'App\Notifications\AppointmentRequestCreated')
                ->orderBy('created_at', 'desc')
                ->limit(50)
                ->get();
            
            foreach ($dbNotifications as $dbNotification) {
                $data = $dbNotification->data;
                $notifications[] = [
                    'type' => 'appointment_request',
                    'icon' => 'calendar-plus',
                    'color' => 'info',
                    'title' => $data['title'] ?? 'Yeni Randevu Talebi',
                    'message' => $data['message'] ?? 'Yeni randevu talebi geldi',
                    'date' => $dbNotification->created_at,
                    'url' => route('admin.appointment-requests.index'),
                    'read_at' => $dbNotification->read_at,
                ];
            }
        }

        // Randevu talepleri (pending status)
        if ($user) {
            $appointmentRequests = AppointmentRequest::where('created_at', '>=', Carbon::now()->subDays(7))
                ->where('status', 'pending')
                ->where('beauty_salon_id', $user->id)
                ->with(['beautySalon', 'service'])
                ->orderBy('created_at', 'desc')
                ->get();
                
            foreach ($appointmentRequests as $request) {
                $customerName = trim($request->name . ' ' . ($request->surname ?? ''));
                $serviceName = 'Hizmet seçilmedi';
                
                if ($request->service_ids && is_array($request->service_ids) && count($request->service_ids) > 0) {
                    $services = Service::whereIn('id', $request->service_ids)->get();
                    if ($services->count() > 0) {
                        $serviceName = $services->pluck('name')->join(', ');
                    }
                } elseif ($request->service) {
                    $serviceName = $request->service->name;
                }
            
                $notifications[] = [
                    'type' => 'appointment_request_pending',
                    'icon' => 'calendar-plus',
                    'color' => 'info',
                    'title' => 'Bekleyen Randevu Talebi',
                    'message' => "{$customerName} - {$serviceName}",
                    'date' => $request->created_at,
                    'url' => route('admin.appointment-requests.index')
                ];
            }
        }

        if ($user) {
            $newAppointments = Reservation::where('created_at', '>=', Carbon::now()->subDays(7))
                ->where('beauty_salon_id', $user->id)
                ->with(['customer', 'salon'])
                ->orderBy('created_at', 'desc')
                ->get();
                
            foreach ($newAppointments as $reservation) {
                $notifications[] = [
                    'type' => 'new_reservation',
                    'icon' => 'calendar-check',
                    'color' => 'primary',
                    'title' => 'Yeni randevu oluşturuldu',
                    'message' => ($reservation->customer ? $reservation->customer->name : 'Müşteri') . ' için randevu oluşturuldu',
                    'date' => $reservation->created_at,
                    'url' => route('admin.appointments.show', $reservation->id)
                ];
            }

            $upcomingAppointments = Reservation::whereBetween('date', [
                    Carbon::now()->toDateString(),
                    Carbon::now()->addDays(3)->toDateString()
                ])
                ->where('beauty_salon_id', $user->id)
                ->where('status', '!=', 'cancelled')
                ->with(['customer', 'salon'])
                ->orderBy('date', 'asc')
                ->get();
                
            foreach ($upcomingAppointments as $reservation) {
                $notifications[] = [
                    'type' => 'upcoming',
                    'icon' => 'clock',
                    'color' => 'warning',
                    'title' => 'Yaklaşan randevu',
                    'message' => ($reservation->customer ? $reservation->customer->name : 'Müşteri') . ' - ' . Carbon::parse($reservation->date)->format('d.m.Y'),
                    'date' => $reservation->date,
                    'url' => route('admin.appointments.show', $reservation->id)
                ];
            }
        }

        usort($notifications, function($a, $b) {
            return $b['date'] <=> $a['date'];
        });
        
        return view('admin.notifications.index', compact('notifications'));
        } catch (\Exception $e) {
            \Log::error('NotificationController index error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response('Bildirimler yüklenirken bir hata oluştu.', 500);
        }
    }

    public function todayAppointments(Request $request)
    {
        try {
            $user = auth()->user();
            $today = Carbon::today()->toDateString();

            if ($user->role === 'güzellik_salonu') {
                $query = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                    ->where('beauty_salon_id', $user->id)
                    ->where('date', $today)
                    ->whereNotIn('status', ['cancelled', 'completed'])
                    ->orderBy('start_time', 'asc');
            }
            
            elseif ($user->isPersonel()) {
                $query = Reservation::with(['customer', 'appointmentRequest.service', 'room'])
                    ->where('personel_id', $user->id)
                    ->where('date', $today)
                    ->whereNotIn('status', ['cancelled', 'completed'])
                    ->orderBy('start_time', 'asc');
            }
            
            else {
                $query = Reservation::with(['customer', 'salon'])
                    ->where('date', $today)
                    ->whereNotIn('status', ['cancelled', 'completed'])
                    ->orderBy('start_time', 'asc');
                
                if ($user->role === 'randevucu' || $user->role === 'admin') {
                    $query->where('created_by', $user->id);
                }
            }
            
            $appointments = $query->get();
            
            return response()->json([
                'appointments' => $appointments->map(function($reservation) use ($user) {
                    return [
                        'id' => $reservation->id,
                        'customer_name' => $reservation->customer ? $reservation->customer->name : 'Müşteri Yok',
                        'start_time' => $reservation->start_time,
                        'end_time' => $reservation->end_time,
                        'status' => $reservation->status,
                        'service_name' => $reservation->appointmentRequest && $reservation->appointmentRequest->service ? $reservation->appointmentRequest->service->name : null,
                        'personel_name' => $reservation->personel ? $reservation->personel->name : null,
                        'room_name' => $reservation->room ? $reservation->room->name : null,
                        'total_price' => $reservation->total_price,
                        'url' => route('admin.appointments.show', $reservation->id),
                    ];
                }),
                'count' => $appointments->count()
            ]);
        } catch (\Exception $e) {
            \Log::error('Today Appointments Error: ' . $e->getMessage());
            return response()->json([
                'appointments' => [],
                'count' => 0,
                'error' => 'Randevular yüklenirken bir hata oluştu.'
            ], 500);
        }
    }

    public function upcomingAppointments(Request $request)
    {
        try {
            $user = auth()->user();
            $now = Carbon::now();
            $oneHourLater = $now->copy()->addHour();
            $thirtyMinutesLater = $now->copy()->addMinutes(30);
            
            $query = Reservation::with(['customer', 'salon'])
                ->where('status', '!=', 'cancelled')
                ->where('status', '!=', 'completed')
                ->where('date', '>=', $now->toDateString());

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $appointments = $query->get()->filter(function($reservation) use ($now, $oneHourLater) {
                try {
                    if (!$reservation->date || !$reservation->start_time) {
                        return false;
                    }
                    
                    $reservationDateTime = Carbon::parse($reservation->date . ' ' . $reservation->start_time);

                    if ($reservationDateTime < $now) {
                        return false;
                    }
                    
                    if ($reservationDateTime > $oneHourLater) {
                        return false;
                    }
                    
                    return true;
                } catch (\Exception $e) {
                    \Log::warning('Randevu tarih/saat parse hatası: ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id ?? null,
                        'date' => $reservation->date ?? null,
                        'start_time' => $reservation->start_time ?? null
                    ]);
                    return false;
                }
            });
            
            return response()->json([
                'appointments' => $appointments->values()->map(function($reservation) {
                    return [
                        'id' => $reservation->id,
                        'date' => $reservation->date,
                        'start_time' => $reservation->start_time,
                        'end_time' => $reservation->end_time,
                        'status' => $reservation->status,
                        'customer' => $reservation->customer ? [
                            'id' => $reservation->customer->id,
                            'name' => $reservation->customer->name,
                            'phone' => $reservation->customer->phone
                        ] : null,
                        'salon' => $reservation->salon ? [
                            'id' => $reservation->salon->id,
                            'name' => $reservation->salon->name
                        ] : null
                    ];
                })
            ]);
        } catch (\Exception $e) {
            \Log::error('Upcoming Appointments Error: ' . $e->getMessage());
            return response()->json([
                'appointments' => [],
                'error' => 'Yaklaşan randevular yüklenirken bir hata oluştu.'
            ], 500);
        }
    }

    public function getNotifications(Request $request)
    {
        try {
            $user = auth()->user();
            $notifications = [];
            $totalUnread = 0;
            
            if (!$user) {
                return response()->json([
                    'notifications' => [],
                    'count' => 0,
                    'unread_count' => 0,
                ]);
            }
            
            // Laravel notifications tablosundan bildirimleri al (okunmamış)
            try {
                $dbNotifications = $user->unreadNotifications()
                    ->where('type', 'App\Notifications\AppointmentRequestCreated')
                    ->orderBy('created_at', 'desc')
                    ->limit(10)
                    ->get();
                
                foreach ($dbNotifications as $dbNotification) {
                    $data = is_array($dbNotification->data) ? $dbNotification->data : json_decode($dbNotification->data, true);
                    $notifications[] = [
                        'id' => $dbNotification->id,
                        'type' => 'appointment_request',
                        'icon' => 'calendar-plus',
                        'color' => 'info',
                        'title' => $data['title'] ?? 'Yeni Randevu Talebi',
                        'message' => $data['message'] ?? 'Yeni randevu talebi geldi',
                        'date' => $dbNotification->created_at ? $dbNotification->created_at->toISOString() : now()->toISOString(),
                        'url' => '/admin/randevu-talepleri',
                        'read_at' => $dbNotification->read_at ? $dbNotification->read_at->toISOString() : null,
                    ];
                }
            } catch (\Exception $e) {
                \Log::warning('Notifications DB error: ' . $e->getMessage());
            }
            
            // Bekleyen randevu talepleri
            try {
                $appointmentRequests = AppointmentRequest::where('status', 'pending')
                    ->where('beauty_salon_id', $user->id)
                    ->where('created_at', '>=', Carbon::now()->subDays(7))
                    ->orderBy('created_at', 'desc')
                    ->limit(10)
                    ->get();
                    
                foreach ($appointmentRequests as $request) {
                    $customerName = trim($request->name . ' ' . ($request->surname ?? ''));
                    $serviceName = 'Hizmet seçilmedi';
                    
                    if ($request->service_ids && is_array($request->service_ids) && count($request->service_ids) > 0) {
                        $services = Service::whereIn('id', $request->service_ids)->get();
                        if ($services->count() > 0) {
                            $serviceName = $services->pluck('name')->join(', ');
                        }
                    } elseif ($request->service) {
                        $serviceName = $request->service->name;
                    }
                    
                    $notifications[] = [
                        'id' => 'request_' . $request->id,
                        'type' => 'appointment_request_pending',
                        'icon' => 'calendar-plus',
                        'color' => 'info',
                        'title' => 'Bekleyen Randevu Talebi',
                        'message' => "{$customerName} - {$serviceName}",
                        'date' => $request->created_at ? $request->created_at->toISOString() : now()->toISOString(),
                        'url' => '/admin/randevu-talepleri'
                    ];
                }
            } catch (\Exception $e) {
                \Log::warning('Appointment requests error: ' . $e->getMessage());
            }
            
            // Tarihe göre sırala
            if (!empty($notifications)) {
                usort($notifications, function($a, $b) {
                    $dateA = strtotime($a['date'] ?? '1970-01-01');
                    $dateB = strtotime($b['date'] ?? '1970-01-01');
                    return $dateB - $dateA;
                });
            }
            
            // Sayıları hesapla
            try {
                $unreadCount = $user->unreadNotifications()
                    ->where('type', 'App\Notifications\AppointmentRequestCreated')
                    ->count();
                
                $pendingRequestsCount = AppointmentRequest::where('status', 'pending')
                    ->where('beauty_salon_id', $user->id)
                    ->where('created_at', '>=', Carbon::now()->subDays(7))
                    ->count();
                
                $totalUnread = $unreadCount + $pendingRequestsCount;
            } catch (\Exception $e) {
                \Log::warning('Count calculation error: ' . $e->getMessage());
                $totalUnread = 0;
            }
            
            return response()->json([
                'notifications' => array_slice($notifications, 0, 10),
                'count' => $totalUnread,
                'unread_count' => $totalUnread,
            ]);
        } catch (\Exception $e) {
            \Log::error('Notifications Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'notifications' => [],
                'count' => 0,
                'unread_count' => 0,
                'error' => 'Bildirimler yüklenirken bir hata oluştu.'
            ], 500);
        }
    }
}
