<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use App\Models\PersonelLeave;

class PersonelController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personels = User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->with('services')
            ->orderBy('name')
            ->get();
        
        return view('admin.personel.index', compact('personels'));
    }

    public function create()
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $services = Service::where('beauty_salon_id', $user->id)
            ->get();
        
        return view('admin.personel.create', compact('services'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'service_ids' => 'nullable|array',
            'service_ids.*' => 'exists:services,id',
        ]);

        $password = $request->filled('password') ? $validated['password'] : 'personel123';
        
        $personelData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
            'password' => Hash::make($password),
            'role' => 'personel',
            'beauty_salon_id' => $user->id,
        ];

        if ($request->hasFile('profile_image')) {
            try {
                $file = $request->file('profile_image');
                
                if ($file->getSize() > 5120 * 1024) {
                    if ($request->ajax() || $request->wantsJson()) {
                        return response()->json(['success' => false, 'message' => 'Dosya boyutu çok büyük. Maksimum 5 MB olmalıdır.'], 422);
                    }
                    return back()->withErrors(['profile_image' => 'Dosya boyutu çok büyük. Maksimum 5 MB olmalıdır.'])->withInput();
                }
                
                $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                if (!in_array($file->getMimeType(), $allowedMimes)) {
                    if ($request->ajax() || $request->wantsJson()) {
                        return response()->json(['success' => false, 'message' => 'Desteklenmeyen dosya formatı.'], 422);
                    }
                    return back()->withErrors(['profile_image' => 'Desteklenmeyen dosya formatı.'])->withInput();
                }
                
                $directory = 'personel-profiles';
                if (!Storage::disk('public')->exists($directory)) {
                    Storage::disk('public')->makeDirectory($directory, 0755, true);
                }
                
                $stored = $file->store($directory, 'public');
                
                if ($stored) {
                    $personelData['profile_image'] = $stored;
                }
            } catch (\Exception $e) {
                Log::error('Profil fotoğrafı yükleme hatası (create): ' . $e->getMessage());
            }
        }
        
        try {
            $personel = User::create($personelData);

            $serviceIds = $request->input('service_ids', []);
            if (!empty($serviceIds)) {
                
                $validServiceIds = Service::where('beauty_salon_id', $user->id)
                    ->whereIn('id', $serviceIds)
                    ->pluck('id')
                    ->toArray();
                
                if (!empty($validServiceIds)) {
                    $personel->services()->sync($validServiceIds);
                }
            }

            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => true,
                    'message' => 'Personel başarıyla eklendi.',
                    'personel' => [
                        'id' => $personel->id,
                        'name' => $personel->name,
                        'email' => $personel->email,
                        'phone' => $personel->phone ?? '',
                        'profile_image' => $personel->profile_image,
                    ]
                ], 200);
            }
            
            return redirect()->route('admin.beauty-salon.personel.index')
                ->with('success', 'Personel başarıyla eklendi.');
        } catch (\Exception $e) {
            Log::error('Personel oluşturma hatası: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'message' => 'Personel oluşturulurken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->withErrors(['error' => 'Personel oluşturulurken bir hata oluştu.'])->withInput();
        }
    }

    public function show($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personel = User::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->with('services')
            ->firstOrFail();
        
        $personelServices = $personel->services;

        $allServices = Service::where('beauty_salon_id', $user->id)
            ->get();

        $appointments = \App\Models\Reservation::where('personel_id', $personel->id)
            ->with(['customer', 'room'])
            ->orderBy('date', 'desc')
            ->orderBy('start_time', 'desc')
            ->paginate(10);

        $stats = [
            'total_appointments' => \App\Models\Reservation::where('personel_id', $personel->id)->count(),
            'completed_appointments' => \App\Models\Reservation::where('personel_id', $personel->id)
                ->where('status', 'completed')
                ->count(),
            'total_revenue' => \App\Models\Reservation::where('personel_id', $personel->id)
                ->where('status', 'completed')
                ->sum('total_price') ?? 0,
        ];
        
        return view('admin.personel.show', compact('personel', 'personelServices', 'allServices', 'appointments', 'stats'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personel = User::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->with('services')
            ->firstOrFail();

        $allServices = Service::where('beauty_salon_id', $user->id)
            ->get();

        $personelServices = $personel->services;
        
        return view('admin.personel.edit', compact('personel', 'allServices', 'personelServices'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personel = User::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->firstOrFail();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $personel->id,
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
        ]);
        
        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
        ];
        
        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        if ($request->hasFile('profile_image')) {
            try {
                $file = $request->file('profile_image');
                
                if ($file->getSize() > 5120 * 1024) {
                    return back()->withErrors(['profile_image' => 'Dosya boyutu çok büyük. Maksimum 5 MB olmalıdır.'])->withInput();
                }
                
                $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                if (!in_array($file->getMimeType(), $allowedMimes)) {
                    return back()->withErrors(['profile_image' => 'Desteklenmeyen dosya formatı.'])->withInput();
                }
                
                if ($personel->profile_image && Storage::disk('public')->exists($personel->profile_image)) {
                    try {
                        Storage::disk('public')->delete($personel->profile_image);
                    } catch (\Exception $e) {
                        
                    }
                }
                
                $directory = 'personel-profiles';
                if (!Storage::disk('public')->exists($directory)) {
                    try {
                        Storage::disk('public')->makeDirectory($directory, 0755, true);
                    } catch (\Exception $dirException) {
                        Log::error('Dizin oluşturma hatası (personel-profiles): ' . $dirException->getMessage());
                        return back()->withErrors(['profile_image' => 'Dizin oluşturulamadı. Lütfen storage/app/public dizininin yazılabilir olduğundan emin olun.'])->withInput();
                    }
                }
                
                try {
                    $stored = $file->store($directory, 'public');
                    
                    if (!$stored) {
                        Log::error('Profil fotoğrafı store() metodu false döndü', [
                            'directory' => $directory,
                        ]);
                        return back()->withErrors(['profile_image' => 'Dosya yüklenirken bir hata oluştu.'])->withInput();
                    }
                    
                    $updateData['profile_image'] = $stored;
                } catch (\Exception $storeException) {
                    Log::error('Profil fotoğrafı yükleme hatası: ' . $storeException->getMessage());
                    return back()->withErrors(['profile_image' => 'Dosya yüklenirken bir hata oluştu: ' . $storeException->getMessage()])->withInput();
                }
            } catch (\Exception $e) {
                Log::error('Profil fotoğrafı yükleme genel hatası: ' . $e->getMessage());
                return back()->withErrors(['profile_image' => 'Dosya yüklenirken bir hata oluştu: ' . $e->getMessage()])->withInput();
            }
        }
        
        try {
            $personel->update($updateData);

            $serviceIds = $request->input('service_ids', []);
            if (!empty($serviceIds)) {
                
                $validServiceIds = Service::where('beauty_salon_id', $user->id)
                    ->whereIn('id', $serviceIds)
                    ->pluck('id')
                    ->toArray();
                
                $personel->services()->sync($validServiceIds);
            } else {
                
                $personel->services()->sync([]);
            }
            
            return redirect()->route('admin.beauty-salon.personel.index')
                ->with('success', 'Personel başarıyla güncellendi.');
        } catch (\Exception $e) {
            Log::error('Personel güncelleme hatası: ' . $e->getMessage(), [
                'personel_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->withErrors(['error' => 'Personel güncellenirken bir hata oluştu: ' . $e->getMessage()])->withInput();
        }
    }

    public function destroy($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personel = User::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->firstOrFail();
        
        try {
            
            if ($personel->profile_image && Storage::disk('public')->exists($personel->profile_image)) {
                try {
                    Storage::disk('public')->delete($personel->profile_image);
                } catch (\Exception $e) {
                    Log::warning('Personel profil fotoğrafı silinemedi: ' . $e->getMessage());
                }
            }

            $personel->services()->detach();
            
            $personel->delete();
            
            return redirect()->route('admin.beauty-salon.personel.index')
                ->with('success', 'Personel başarıyla silindi.');
        } catch (\Exception $e) {
            Log::error('Personel silme hatası: ' . $e->getMessage(), [
                'personel_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->withErrors(['error' => 'Personel silinirken bir hata oluştu.']);
        }
    }

    public function assignServices(Request $request, $personelId)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personel = User::where('id', $personelId)
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->firstOrFail();

        $serviceIds = $request->input('service_ids', []);

        \Log::info('Personel Hizmet Atama', [
            'personel_id' => $personelId,
            'service_ids' => $serviceIds,
            'request_all' => $request->all()
        ]);

        $validated = $request->validate([
            'service_ids' => 'nullable|array',
            'service_ids.*' => 'exists:services,id',
        ]);

        $servicesToSync = $validated['service_ids'] ?? [];
        
        \Log::info('Hizmetler senkronize ediliyor', [
            'personel_id' => $personelId,
            'services_to_sync' => $servicesToSync
        ]);
        
        $personel->services()->sync($servicesToSync);

        $syncedServices = $personel->services()->pluck('services.id')->toArray();
        \Log::info('Senkronizasyon sonrası', [
            'personel_id' => $personelId,
            'synced_services' => $syncedServices
        ]);
        
        return redirect()->back()
            ->with('success', 'Hizmetler başarıyla atandı.');
    }

    public function removeService($personelId, $serviceId)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personel = User::where('id', $personelId)
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->firstOrFail();
        
        $personel->services()->detach($serviceId);
        
        return redirect()->back()
            ->with('success', 'Hizmet başarıyla kaldırıldı.');
    }

    public function getBusyHours(Request $request, $personelId)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $personel = User::where('id', $personelId)
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->firstOrFail();
        
        $date = $request->input('date');
        if (!$date) {
            return response()->json([
                'busy_hours' => []
            ]);
        }

        $onLeave = PersonelLeave::where('personel_id', $personelId)
            ->whereDate('leave_date', $date)
            ->exists();

        if ($onLeave) {
            $busyHours = [];
            for ($hour = 8; $hour < 22; $hour++) {
                $busyHours[str_pad($hour, 2, '0', STR_PAD_LEFT) . ':00'] = true;
                $busyHours[str_pad($hour, 2, '0', STR_PAD_LEFT) . ':30'] = true;
            }
            $busyHours['22:00'] = true;

            return response()->json([
                'busy_hours' => $busyHours,
                'date' => $date,
                'personel_id' => $personelId,
                'on_leave' => true,
            ]);
        }

        $reservations = \App\Models\Reservation::where('personel_id', $personelId)
            ->where('beauty_salon_id', $user->id)
            ->whereDate('date', $date)
            ->whereIn('status', ['confirmed', 'pending', 'in_progress', 'open'])
            ->with('services')
            ->get();

        $busyHours = [];
        foreach ($reservations as $reservation) {
            $startTime = \Carbon\Carbon::parse($date . ' ' . $reservation->start_time);
            
            // Eğer end_time varsa onu kullan, yoksa hizmet sürelerine göre hesapla
            if ($reservation->end_time) {
                $endTime = \Carbon\Carbon::parse($date . ' ' . $reservation->end_time);
            } else {
                // Hizmet sürelerini topla
                $totalDuration = 30; // Varsayılan
                if ($reservation->services && $reservation->services->count() > 0) {
                    $totalDuration = $reservation->services->sum('duration') ?: 30;
                }
                $endTime = $startTime->copy()->addMinutes($totalDuration);
            }

            // 30 dakikalık slotlara göre dolu saatleri işaretle
            $current = $startTime->copy();
            while ($current < $endTime) {
                $timeSlot = $current->format('H:i');
                $busyHours[$timeSlot] = true;
                $current->addMinutes(30);
            }
            
            // Başlangıç saatini de dolu olarak işaretle
            $busyHours[$startTime->format('H:i')] = true;
        }
        
        return response()->json([
            'busy_hours' => $busyHours,
            'date' => $date,
            'personel_id' => $personelId
        ]);
    }
}

