<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AppointmentRequest;
use App\Models\PersonelLeave;
use App\Models\Reservation;
use App\Models\Service;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PersonelLeaveController extends Controller
{
    private function toMinutes(?string $time): int
    {
        if (!$time) {
            return 0;
        }
        [$h, $m] = array_pad(explode(':', $time), 2, 0);
        return ((int) $h * 60) + (int) $m;
    }

    private function reservationRange(array $reservation): array
    {
        $start = $this->toMinutes($reservation['start_time'] ?? null);
        $end = $reservation['end_time'] ? $this->toMinutes($reservation['end_time']) : ($start + 30);
        if ($end <= $start) {
            $end = $start + 30;
        }
        return [$start, $end];
    }

    private function hasOverlap(array $a, array $b): bool
    {
        [$aStart, $aEnd] = $this->reservationRange($a);
        [$bStart, $bEnd] = $this->reservationRange($b);
        return $aStart < $bEnd && $bStart < $aEnd;
    }

    public function preview(Request $request, $personelId)
    {
        $salon = Auth::user();
        if ($salon->role !== 'güzellik_salonu') {
            abort(403);
        }

        $validated = $request->validate([
            'date' => 'required|date',
        ]);

        $personel = User::where('id', $personelId)
            ->where('beauty_salon_id', $salon->id)
            ->where('role', 'personel')
            ->firstOrFail();

        $assignedServices = collect();
        try {
            $assignedServices = $personel->services()->get(['services.id', 'services.name']);
        } catch (\Throwable $e) {
            \Log::warning('Personel hizmetleri yüklenemedi', [
                'personel_id' => $personel->id,
                'error' => $e->getMessage(),
            ]);
        }

        $date = $validated['date'];

        $reservations = Reservation::where('beauty_salon_id', $salon->id)
            ->where('personel_id', $personel->id)
            ->whereDate('date', $date)
            ->whereNotIn('status', ['cancelled'])
            ->with([
                'customer:id,name',
                'appointmentRequest:id,reservation_id,service_id,service_ids',
                'appointmentRequest.service:id,name',
            ])
            ->orderBy('start_time')
            ->get();

        $appointmentItems = $reservations->map(function ($reservation) {
            $serviceNames = [];
            if ($reservation->appointmentRequest) {
                $req = $reservation->appointmentRequest;
                if (is_array($req->service_ids) && !empty($req->service_ids)) {
                    $serviceNames = Service::whereIn('id', $req->service_ids)->pluck('name')->toArray();
                } elseif ($req->service) {
                    $serviceNames = [$req->service->name];
                }
            }

            return [
                'reservation_id' => $reservation->id,
                'start_time' => $reservation->start_time,
                'end_time' => $reservation->end_time,
                'customer_name' => optional($reservation->customer)->name ?: '-',
                'service_names' => $serviceNames,
            ];
        })->values();

        $otherPersonels = User::where('beauty_salon_id', $salon->id)
            ->where('role', 'personel')
            ->where('id', '!=', $personel->id)
            ->orderBy('name')
            ->get(['id', 'name']);

        $leavePersonelIds = PersonelLeave::where('beauty_salon_id', $salon->id)
            ->whereDate('leave_date', $date)
            ->pluck('personel_id')
            ->all();

        $otherPersonels = $otherPersonels->map(function ($p) use ($salon, $date, $leavePersonelIds) {
            $activeCount = Reservation::where('beauty_salon_id', $salon->id)
                ->where('personel_id', $p->id)
                ->whereDate('date', $date)
                ->whereNotIn('status', ['cancelled'])
                ->count();

            return [
                'id' => $p->id,
                'name' => $p->name,
                'on_leave' => in_array($p->id, $leavePersonelIds, true),
                'active_appointment_count' => $activeCount,
            ];
        })->values();

        $existingLeave = PersonelLeave::with('transferPersonel:id,name')
            ->where('personel_id', $personel->id)
            ->whereDate('leave_date', $date)
            ->first();

        return response()->json([
            'success' => true,
            'personel' => [
                'id' => $personel->id,
                'name' => $personel->name,
            ],
            'has_active_appointments' => $appointmentItems->isNotEmpty(),
            'appointments' => $appointmentItems,
            'assigned_services' => $assignedServices->map(function ($service) {
                return [
                    'id' => $service->id,
                    'name' => $service->name,
                ];
            })->values(),
            'other_personels' => $otherPersonels,
            'already_on_leave' => (bool) $existingLeave,
            'existing_leave' => $existingLeave ? [
                'id' => $existingLeave->id,
                'note' => $existingLeave->note,
                'transfer_to_personel_id' => $existingLeave->transfer_to_personel_id,
                'transfer_to_personel_name' => optional($existingLeave->transferPersonel)->name,
            ] : null,
        ]);
    }

    public function store(Request $request, $personelId)
    {
        $salon = Auth::user();
        if ($salon->role !== 'güzellik_salonu') {
            abort(403);
        }

        $personel = User::where('id', $personelId)
            ->where('beauty_salon_id', $salon->id)
            ->where('role', 'personel')
            ->firstOrFail();

        $validated = $request->validate([
            'leave_date' => 'required|date',
            'note' => 'nullable|string|max:500',
            'transfer_to_personel_id' => 'nullable|integer|exists:users,id',
            'transfer_reservation_ids' => 'nullable|array',
            'transfer_reservation_ids.*' => 'integer|exists:reservations,id',
        ]);

        $transferReservationIds = collect($validated['transfer_reservation_ids'] ?? [])->filter()->map(fn ($id) => (int) $id)->unique()->values();
        $targetPersonelId = $validated['transfer_to_personel_id'] ?? null;
        $leaveDate = $validated['leave_date'];

        if ($transferReservationIds->isNotEmpty() && !$targetPersonelId) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu aktarımı için hedef personel seçmelisiniz.',
            ], 422);
        }

        $targetPersonel = null;
        if ($targetPersonelId) {
            $targetPersonel = User::where('id', $targetPersonelId)
                ->where('beauty_salon_id', $salon->id)
                ->where('role', 'personel')
                ->where('id', '!=', $personel->id)
                ->first();

            if (!$targetPersonel) {
                return response()->json([
                    'success' => false,
                    'message' => 'Hedef personel bulunamadı.',
                ], 422);
            }

            $targetOnLeave = PersonelLeave::where('beauty_salon_id', $salon->id)
                ->where('personel_id', $targetPersonel->id)
                ->whereDate('leave_date', $leaveDate)
                ->exists();

            if ($targetOnLeave) {
                return response()->json([
                    'success' => false,
                    'message' => 'İzinli personele randevu aktarımı yapılamaz. Lütfen başka personel seçin.',
                ], 422);
            }
        }

        $activeReservations = Reservation::where('beauty_salon_id', $salon->id)
            ->where('personel_id', $personel->id)
            ->whereDate('date', $leaveDate)
            ->whereNotIn('status', ['cancelled'])
            ->get(['id', 'start_time', 'end_time']);

        if ($activeReservations->isNotEmpty() && $transferReservationIds->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Bu tarihte personelin aktif randevuları var. Lütfen aktarılacak randevuları seçin.',
            ], 422);
        }

        if ($activeReservations->isNotEmpty()) {
            $activeIds = $activeReservations->pluck('id')->map(fn ($id) => (int) $id)->sort()->values();
            $selectedIds = $transferReservationIds->sort()->values();

            if ($selectedIds->count() !== $activeIds->count() || $selectedIds->diff($activeIds)->isNotEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'İzin kaydı için o günkü tüm aktif randevuları eksiksiz aktarmalısınız.',
                ], 422);
            }
        }

        if ($targetPersonel && $transferReservationIds->isNotEmpty()) {
            $sourceReservations = $activeReservations->whereIn('id', $transferReservationIds->all())->values();

            $targetReservations = Reservation::where('beauty_salon_id', $salon->id)
                ->where('personel_id', $targetPersonel->id)
                ->whereDate('date', $leaveDate)
                ->whereNotIn('status', ['cancelled'])
                ->get(['id', 'start_time', 'end_time']);

            foreach ($sourceReservations as $sourceReservation) {
                $sourceArray = $sourceReservation->toArray();
                foreach ($targetReservations as $targetReservation) {
                    $targetArray = $targetReservation->toArray();
                    if ($this->hasOverlap($sourceArray, $targetArray)) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Hedef personelde saat çakışması var. Lütfen farklı personel seçin.',
                        ], 422);
                    }
                }
            }
        }

        $leaveExisted = PersonelLeave::where('personel_id', $personel->id)
            ->whereDate('leave_date', $leaveDate)
            ->exists();

        DB::transaction(function () use (
            $salon,
            $personel,
            $targetPersonel,
            $targetPersonelId,
            $leaveDate,
            $validated,
            $transferReservationIds
        ) {
            $leave = PersonelLeave::firstOrNew([
                'personel_id' => $personel->id,
                'leave_date' => $leaveDate,
            ]);

            $leave->beauty_salon_id = $salon->id;
            $leave->note = $validated['note'] ?? null;
            $leave->created_by = $salon->id;
            $leave->transfer_to_personel_id = $targetPersonelId;
            $leave->save();

            if ($targetPersonel && $transferReservationIds->isNotEmpty()) {
                $reservations = Reservation::where('beauty_salon_id', $salon->id)
                    ->where('personel_id', $personel->id)
                    ->whereDate('date', $leaveDate)
                    ->whereIn('id', $transferReservationIds->all())
                    ->get();

                foreach ($reservations as $reservation) {
                    $reservation->personel_id = $targetPersonel->id;
                    $reservation->save();
                }

                AppointmentRequest::whereIn('reservation_id', $reservations->pluck('id')->all())
                    ->update(['personel_id' => $targetPersonel->id]);
            }
        });

        if ($transferReservationIds->isNotEmpty()) {
            $message = $leaveExisted
                ? 'Personel izni güncellendi. Seçilen randevular başarıyla aktarıldı.'
                : 'Personel izni kaydedildi. Seçilen randevular başarıyla aktarıldı.';
        } else {
            $message = $leaveExisted
                ? 'Personel izni güncellendi.'
                : 'Personel izni kaydedildi.';
        }

        return response()->json([
            'success' => true,
            'message' => $message,
        ]);
    }

    public function destroy(Request $request, $personelId)
    {
        $salon = Auth::user();
        if ($salon->role !== 'güzellik_salonu') {
            abort(403);
        }

        $validated = $request->validate([
            'leave_date' => 'required|date',
        ]);

        $personel = User::where('id', $personelId)
            ->where('beauty_salon_id', $salon->id)
            ->where('role', 'personel')
            ->firstOrFail();

        $leave = PersonelLeave::where('personel_id', $personel->id)
            ->whereDate('leave_date', $validated['leave_date'])
            ->first();

        if (!$leave) {
            return response()->json([
                'success' => false,
                'message' => 'İptal edilecek izin kaydı bulunamadı.',
            ], 404);
        }

        $leave->delete();

        return response()->json([
            'success' => true,
            'message' => 'Personel izni başarıyla iptal edildi.',
        ]);
    }
}

