<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PersonelLeave;
use App\Models\Reservation;
use Illuminate\Http\Request;

class ReservationCalendarController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $salons = collect([]);
        $personels = collect([]);

        if ($user->role !== 'güzellik_salonu') {
            $query = \App\Models\Salon::where('is_active', true);

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $salons = $query->get();
        }

        if ($user->role === 'güzellik_salonu') {
            $personels = \App\Models\User::where('role', 'personel')
                ->where('beauty_salon_id', $user->id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();
        }
        
        return view('admin.appointments.calendar', compact('salons', 'personels'));
    }

    public function feed(Request $request)
    {
        $start = $request->get('start');
        $end = $request->get('end');
        $salonId = $request->get('salon_id');
        $personelId = $request->get('personel_id');

        $user = auth()->user();

        if ($user->role === 'güzellik_salonu') {
            $query = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                ->where('beauty_salon_id', $user->id)
                ->whereBetween('date', [$start, $end])
                ->whereNotIn('status', ['cancelled', 'completed']);
            
            if ($personelId) {
                $query->where('personel_id', $personelId);
            }
        } else {
            
            $query = Reservation::with(['customer', 'salon'])
                ->whereBetween('date', [$start, $end])
                ->whereNotIn('status', ['cancelled', 'completed']);

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }

            if ($salonId) {
                $query->where('salon_id', $salonId);
            }
        }

        $appointments = $query->get()->map(function($reservation) use ($user) {
            
            $color = '#3b82f6'; 
            switch($reservation->status) {
                case 'confirmed':
                    $color = '#3b82f6'; 
                    break;
                case 'open':
                    $color = '#f59e0b'; 
                    break;
                case 'completed':
                    $color = '#22c55e'; 
                    break;
                case 'cancelled':
                    $color = '#ef4444'; 
                    break;
            }

            if ($user->role === 'güzellik_salonu') {
                $personelName = $reservation->personel ? $reservation->personel->name : 'Personel Yok';
                $serviceName = $reservation->appointmentRequest && $reservation->appointmentRequest->service 
                    ? $reservation->appointmentRequest->service->name 
                    : 'Hizmet';
                
                $title = $personelName . ' - ' . $serviceName;
                $salonName = '';
            } else {
                
                if ($reservation->personel) {
                    $title = $reservation->personel->name . ' - ' . ($reservation->salon->name ?? 'Salon');
                } else {
                    $title = ($reservation->customer->name ?? 'Müşteri') . ' - ' . ($reservation->salon->name ?? 'Salon');
                }
                $salonName = $reservation->salon->name ?? '';
            }

            return [
                'id' => $reservation->id,
                'title' => $title,
                'start' => $reservation->date->format('Y-m-d') . 'T' . $reservation->start_time,
                'end' => $reservation->date->format('Y-m-d') . 'T' . $reservation->end_time,
                'color' => $color,
                'extendedProps' => [
                    'reservation_id' => $reservation->id,
                    'status' => $reservation->status,
                    'customer' => $reservation->customer->name ?? '',
                    'personel' => $reservation->personel ? $reservation->personel->name : '',
                    'personel_id' => $reservation->personel_id,
                    'service' => $reservation->appointmentRequest && $reservation->appointmentRequest->service 
                        ? $reservation->appointmentRequest->service->name 
                        : '',
                    'room' => $reservation->room ? $reservation->room->name : '',
                    'salon' => $salonName,
                    'start_time' => $reservation->start_time,
                    'end_time' => $reservation->end_time,
                    'total_price' => $reservation->total_price,
                ],
                'url' => route('admin.appointments.show', $reservation->id),
            ];
        });

        if ($user->role === 'güzellik_salonu') {
            $leaveQuery = PersonelLeave::with('personel:id,name')
                ->where('beauty_salon_id', $user->id)
                ->whereBetween('leave_date', [$start, $end]);

            if ($personelId) {
                $leaveQuery->where('personel_id', $personelId);
            }

            $leaveEvents = $leaveQuery->get()->map(function ($leave) {
                $leaveDate = $leave->leave_date instanceof \Carbon\Carbon
                    ? $leave->leave_date->format('Y-m-d')
                    : (string) $leave->leave_date;

                return [
                    'id' => 'leave-' . $leave->id,
                    'title' => 'İzinli',
                    'start' => $leaveDate . 'T08:00:00',
                    'end' => $leaveDate . 'T22:00:00',
                    'color' => '#94a3b8',
                    'extendedProps' => [
                        'status' => 'leave',
                        'personel' => optional($leave->personel)->name,
                        'personel_id' => $leave->personel_id,
                        'note' => $leave->note,
                    ],
                ];
            });

            $appointments = $appointments->concat($leaveEvents)->values();
        }

        return response()->json($appointments);
    }

    public function year()
    {
        $user = auth()->user();
        $query = Reservation::select('date', 'status')
            ->whereYear('date', now()->year);

        if ($user->role === 'güzellik_salonu') {
            $query->where('beauty_salon_id', $user->id);
        } elseif ($user->role === 'randevucu' || $user->role === 'admin') {
            
            $query->where('created_by', $user->id);
        }
        
        $appointments = $query->get()
            ->groupBy(function($item) {
                return $item->date->format('Y-m-d');
            });

        return view('admin.appointments.year-calendar', compact('appointments'));
    }

    public function yearV2()
    {
        $user = auth()->user();

        if ($user->role === 'güzellik_salonu') {
            $query = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                ->where('beauty_salon_id', $user->id)
                ->whereYear('date', now()->year);
        } else {
            $query = Reservation::with(['customer', 'salon'])
                ->whereYear('date', now()->year);

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
        }
        
        $appointments = $query->get()
            ->groupBy(function($item) {
                return $item->date->format('Y-m-d');
            });

        return view('admin.appointments.year-calendar-v2', compact('appointments'));
    }
}

