<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use App\Models\Salon;
use App\Models\Menu;
use App\Models\Extra;
use App\Models\ActivityLog;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\Invoice;
use App\Models\Contract;
use App\Models\User;
use App\Services\SmsService;
use App\Services\PremiumFeaturesIntegrationService;
use App\Models\Setting;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ReservationController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();

            if ($user->role === 'güzellik_salonu') {
                $query = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                    ->where('beauty_salon_id', $user->id);
            } else {
                
                $query = Reservation::query()->with(['customer', 'salon', 'menu', 'appointmentRequest.service', 'personel']);

                $dateColumn = $this->getReservationDateColumn();

                $sortBy = $request->get('sort', 'date_asc');
                $needsJoin = ($sortBy === 'name_asc');
                
                if ($needsJoin) {
                    $query->leftJoin('customers', 'reservations.customer_id', '=', 'customers.id')
                          ->select('reservations.*');
                }

                if ($user->role === 'randevucu' || $user->role === 'admin') {
                    if ($needsJoin) {
                        $query->where('reservations.created_by', $user->id);
                    } else {
                        $query->where('created_by', $user->id);
                    }
                }
            }

            if ($user->role !== 'güzellik_salonu') {
                $dateColumn = $this->getReservationDateColumn();
            } else {
                $dateColumn = 'date';
            }

            $sortBy = $request->get('sort', 'date_asc');
            $needsJoin = ($sortBy === 'name_asc' && $user->role !== 'güzellik_salonu');

            if ($request->filled('date')) {
                if ($user->role === 'güzellik_salonu') {
                    $query->whereDate($dateColumn, $request->date);
                } elseif ($needsJoin) {
                    $query->whereDate('reservations.' . $dateColumn, $request->date);
                } else {
                    $query->whereDate($dateColumn, $request->date);
                }
            }
            if ($request->filled('salon_id') && $user->role !== 'güzellik_salonu') {
                if ($needsJoin) {
                    $query->where('reservations.salon_id', $request->salon_id);
                } else {
                    $query->where('salon_id', $request->salon_id);
                }
            }
            if ($request->filled('status')) {
                if ($user->role === 'güzellik_salonu') {
                    $query->where('status', $request->status);
                } elseif ($needsJoin) {
                    $query->where('reservations.status', $request->status);
                } else {
                    $query->where('status', $request->status);
                }
            }
            if ($request->filled('customer_name')) {
                try {
                    $query->whereHas('customer', function($q) use ($request, $user) {
                        $q->where('name', 'like', '%' . $request->customer_name . '%');
                        
                        if ($user->role === 'randevucu' || $user->role === 'admin') {
                            $q->where('created_by', $user->id);
                        }
                    });
                } catch (\Exception $e) {
                    
                }
            }

            if ($user->role === 'güzellik_salonu') {
                
                $query->orderByRaw("CASE WHEN status = 'completed' THEN 1 ELSE 0 END");
                switch ($sortBy) {
                    case 'date_asc':
                        $query->orderBy($dateColumn, 'asc')->orderBy('start_time', 'asc');
                        break;
                    case 'date_desc':
                        $query->orderBy($dateColumn, 'desc')->orderBy('start_time', 'desc');
                        break;
                    case 'name_asc':
                        $query->join('customers', 'reservations.customer_id', '=', 'customers.id')
                              ->select('reservations.*')
                              ->orderBy('customers.name', 'asc');
                        break;
                    default:
                        $query->orderBy($dateColumn, 'asc')->orderBy('start_time', 'asc');
                }
            } else {
                
                if ($needsJoin) {
                    $query->orderByRaw("CASE WHEN reservations.status = 'completed' THEN 1 ELSE 0 END");
                } else {
                    $query->orderByRaw("CASE WHEN status = 'completed' THEN 1 ELSE 0 END");
                }
                
                switch ($sortBy) {
                    case 'date_asc':
                        if ($needsJoin) {
                            $query->orderBy('reservations.' . $dateColumn, 'asc');
                        } else {
                            $query->orderBy($dateColumn, 'asc');
                        }
                        break;
                    case 'name_asc':
                        $query->orderBy('customers.name', 'asc');
                        break;
                    case 'date_desc':
                        if ($needsJoin) {
                            $query->orderBy('reservations.' . $dateColumn, 'desc');
                        } else {
                            $query->orderBy($dateColumn, 'desc');
                        }
                        break;
                    default:
                        if ($needsJoin) {
                            $query->orderBy('reservations.' . $dateColumn, 'asc');
                        } else {
                            $query->orderBy($dateColumn, 'asc');
                        }
                }
            }

            $appointments = $query->paginate(20);

            $salonQuery = Salon::where('is_active', true);
            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $salonQuery->where('created_by', $user->id);
            }
            $salons = $salonQuery->get();

            return view('admin.appointments.index', compact('appointments', 'salons'));
        } catch (\Exception $e) {
            \Log::error('Reservation Index Error: ' . $e->getMessage());
            $appointments = new \Illuminate\Pagination\LengthAwarePaginator([], 0, 20, 1);
            $salons = collect([]);
            return view('admin.appointments.index', compact('appointments', 'salons'))
                ->with('error', 'Randevular yüklenirken bir hata oluştu.');
        }
    }

    public function create()
    {
        try {
            $user = Auth::user();

            if ($user->role === 'personel') {
                abort(403, 'Personel randevu oluşturamaz.');
            }
            $query = Customer::query();

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $customers = $query->orderBy('name')->get();
            
            $salonQuery = Salon::where('is_active', true);
            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $salonQuery->where('created_by', $user->id);
            }
            $salons = $salonQuery->get();

            try {
                $menus = \App\Models\Menu::where('is_active', true)->orderBy('name')->get();
            } catch (\Exception $e) {
                $menus = collect([]);
            }
            
            try {
                $extras = \App\Models\Extra::where('is_active', true)->orderBy('name')->get();
            } catch (\Exception $e) {
                $extras = collect([]);
            }

            return view('admin.appointments.create', compact('customers', 'salons', 'menus', 'extras'));
        } catch (\Exception $e) {
            \Log::error('Reservation Create Error: ' . $e->getMessage());
            return view('admin.appointments.create', [
                'customers' => collect([]),
                'salons' => collect([]),
                'menus' => collect([]),
                'extras' => collect([])
            ])->with('error', 'Sayfa yüklenirken bir hata oluştu.');
        }
    }

    public function checkAvailability(Request $request)
    {
        try {
            $date = $request->get('date');
            $salonId = $request->get('salon_id');
            
            if (!$date || !$salonId) {
                return response()->json(['busy_hours' => []]);
            }
            
            $user = Auth::user();

            $query = Reservation::where('salon_id', $salonId)
                ->whereDate('date', $date)
                ->whereIn('status', ['open', 'confirmed']);

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $salon = Salon::where('id', $salonId)->where('created_by', $user->id)->first();
                if (!$salon) {
                    return response()->json(['busy_hours' => [], 'error' => 'Salon bulunamadı']);
                }
            }
            
            $appointments = $query->select('start_time', 'end_time', 'customer_id')
                ->with('customer:id,name')
                ->get();
            
            $busyHours = $appointments->map(function($res) {
                return [
                    'start_time' => $res->start_time,
                    'end_time' => $res->end_time,
                    'customer' => $res->customer->name ?? 'Müşteri'
                ];
            });
            
            return response()->json([
                'busy_hours' => $busyHours,
                'date' => $date,
                'salon_id' => $salonId
            ]);
        } catch (\Exception $e) {
            \Log::error('Check Availability Error: ' . $e->getMessage());
            return response()->json(['busy_hours' => [], 'error' => $e->getMessage()]);
        }
    }

    public function store(Request $request)
    {
        try {
            $user = Auth::user();

            if ($user->role === 'personel') {
                abort(403, 'Personel randevu oluşturamaz.');
            }

            if ($user->role === 'güzellik_salonu') {
                $validated = $request->validate([
                    'customer_id' => 'required|exists:customers,id',
                    'service_id' => 'required|array|min:1',
                    'service_id.*' => 'required|exists:services,id',
                    'personel_id' => 'required|exists:users,id',
                    'room_id' => 'nullable|exists:rooms,id',
                    'date' => 'required|date',
                    'start_time' => 'required',
                    'end_time' => 'nullable', 
                    'total_price' => 'required|numeric|min:0',
                    'notes' => 'nullable|string',
                ]);

                $customer = Customer::where('id', $validated['customer_id'])
                    ->where(function($query) use ($user) {
                        if (\Schema::hasColumn('customers', 'beauty_salon_id')) {
                            $query->where('beauty_salon_id', $user->id);
                        } else {
                            $query->where('created_by', $user->id);
                        }
                    })
                    ->firstOrFail();

                $serviceIds = $validated['service_id'];
                $services = \App\Models\Service::whereIn('id', $serviceIds)
                    ->where('beauty_salon_id', $user->id)
                    ->get();
                
                if ($services->count() !== count($serviceIds)) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Seçilen hizmetlerden biri veya birkaçı bulunamadı.');
                }

                $personel = \App\Models\User::where('id', $validated['personel_id'])
                    ->where('beauty_salon_id', $user->id)
                    ->where('role', 'personel')
                    ->firstOrFail();

                if (empty($validated['end_time']) && !empty($validated['start_time'])) {
                    // Bitiş saati her zaman 30 dakika sonrası olarak ayarla
                    $startTime = \Carbon\Carbon::createFromFormat('H:i:s', $validated['start_time']);
                    $validated['end_time'] = $startTime->addMinutes(30)->format('H:i:s');
                }

                $code = 'REZ-' . strtoupper(uniqid());

                $totalPrice = $validated['total_price'];

                $itemsData = [
                    'services' => $services->map(function($service) {
                        return [
                            'id' => $service->id,
                            'name' => $service->name,
                            'price' => $service->price,
                            'duration' => $service->duration,
                        ];
                    })->toArray()
                ];

                $reservation = Reservation::create([
                    'code' => $code,
                    'customer_id' => $customer->id,
                    'salon_id' => null,
                    'beauty_salon_id' => $user->id,
                    'date' => $validated['date'],
                    'start_time' => $validated['start_time'],
                    'end_time' => $validated['end_time'],
                    'personel_id' => $validated['personel_id'],
                    'room_id' => $validated['room_id'] ?? null,
                    'status' => 'confirmed', 
                    'total_price' => $totalPrice, 
                    'deposit_paid' => 0, 
                    'remaining_amount' => $totalPrice, 
                    'notes' => $validated['notes'] ?? null,
                    'items_data' => json_encode($itemsData),
                    'created_by' => $user->id,
                ]);

                // Hizmetleri iliştir
                $reservation->services()->attach($serviceIds);

                // Premium özellikler entegrasyonu - Randevu oluşturulurken
                try {
                    $premiumService = app(PremiumFeaturesIntegrationService::class);
                    $premiumService->onAppointmentCreated($reservation, [
                        'campaign_id' => $request->input('campaign_id'),
                    ]);
                } catch (\Exception $e) {
                    \Log::error('Premium özellik entegrasyonu hatası (store): ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'trace' => $e->getTraceAsString(),
                    ]);
                }

                try {
                    // Birden fazla hizmet varsa ilkini al, yoksa null
                    $firstService = $services->first();
                    $this->createInvoiceForReservation($reservation, $firstService, $customer, $user);
                } catch (\Exception $e) {
                    \Log::error('Fatura oluşturma hatası (yeni randevu)', [
                        'reservation_id' => $reservation->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
                
                return redirect()->route('admin.appointments.index')
                    ->with('success', 'Randevu başarıyla oluşturuldu ve fatura kesildi.');
            }

            $validated = $request->validate([
                'customer_id' => 'required|exists:customers,id',
                'salon_id' => 'required|exists:salons,id',
                'date' => 'required|date',
                'start_time' => 'required',
                'end_time' => 'required',
                'guest_count' => 'nullable|integer|min:1',
                'total_price' => 'required|numeric|min:0|max:99999999.99',
                'deposit_paid' => 'nullable|numeric|min:0|max:99999999.99',
                'status' => 'nullable|in:open,confirmed,cancelled,completed',
                'notes' => 'nullable|string',
                'meals' => 'nullable|array',
                'extras' => 'nullable|array',
                'organization_price' => 'nullable|numeric|min:0|max:99999999.99',
            ]);

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $customer = Customer::findOrFail($validated['customer_id']);
                if ($customer->created_by !== $user->id) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Bu müşteriye erişim yetkiniz yok.');
                }
            }

            $reservationPrice = (float)($validated['total_price'] ?? 0);
            $mealTotal = 0;
            
            if ($request->filled('meals') && is_array($request->meals)) {
                foreach ($request->meals as $meal) {
                    if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                        $mealTotal += (float)$meal['quantity'] * (float)$meal['price'];
                    }
                }
            }

            $organizationTotal = 0;
            if ($request->filled('organizations') && is_array($request->organizations)) {
                foreach ($request->organizations as $org) {
                    if (isset($org['quantity']) && isset($org['price'])) {
                        $organizationTotal += (float)$org['quantity'] * (float)$org['price'];
                    }
                }
            }
            
            $extraTotal = 0;
            
            if ($request->filled('extras') && is_array($request->extras)) {
                foreach ($request->extras as $extra) {
                    if (isset($extra['name']) && isset($extra['price'])) {
                        $extraTotal += (float)$extra['price'];
                    }
                }
            }

            $totalPrice = $reservationPrice + $mealTotal + $organizationTotal + $extraTotal;

            if ($totalPrice <= 0) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Randevu oluşturmak için en az bir fiyat girişi yapmanız gerekmektedir. Lütfen salon ücreti, yemek, organizasyon veya ekstra fiyatlarından en az birini giriniz.');
            }

            $maxValue = 99999999.99;
            if ($totalPrice > $maxValue) {
                \Log::error('Total price exceeds maximum value', [
                    'total_price' => $totalPrice,
                    'max_value' => $maxValue,
                    'reservation_price' => $reservationPrice,
                    'meal_total' => $mealTotal,
                    'organization_price' => $organizationPrice,
                    'extra_total' => $extraTotal,
                ]);
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Toplam tutar çok büyük! Maksimum tutar: ' . number_format($maxValue, 2, ',', '.') . ' ₺');
            }

            $depositPaid = 0;
            $remainingAmount = $totalPrice; 

            $code = 'REZ-' . strtoupper(uniqid());

            $reservationDate = date('Y-m-d', strtotime($validated['date']));

            $reservation = Reservation::create([
                'code' => $code,
                'customer_id' => $validated['customer_id'],
                'salon_id' => $validated['salon_id'],
                'date' => $reservationDate,
                'reservation_date' => $reservationDate,
                'start_time' => $validated['start_time'],
                'end_time' => $validated['end_time'],
                'guest_count' => $validated['guest_count'] ?? 1,
                'total_price' => $totalPrice,
                'deposit_paid' => $depositPaid,
                'remaining_amount' => $remainingAmount,
                'status' => 'confirmed', 
                'notes' => $validated['notes'] ?? null,
                'created_by' => $user->id,
            ]);

            $itemsData = [];

            if ($request->filled('meals') && is_array($request->meals)) {
                $itemsData['meals'] = [];
                foreach ($request->meals as $meal) {
                    if (isset($meal['name']) && !empty(trim($meal['name'])) && isset($meal['price']) && (float)$meal['price'] > 0) {
                        $itemsData['meals'][] = [
                            'name' => trim($meal['name']),
                            'quantity' => isset($meal['quantity']) && (int)$meal['quantity'] > 0 ? (int)$meal['quantity'] : 1,
                            'price' => (float)$meal['price']
                        ];
                    }
                }
            }

            if ($request->filled('organizations') && is_array($request->organizations)) {
                $itemsData['organizations'] = [];
                foreach ($request->organizations as $org) {
                    if (isset($org['name']) && !empty(trim($org['name'])) && isset($org['price']) && (float)$org['price'] > 0) {
                        $itemsData['organizations'][] = [
                            'name' => trim($org['name']),
                            'quantity' => isset($org['quantity']) && (int)$org['quantity'] > 0 ? (int)$org['quantity'] : 1,
                            'price' => (float)$org['price']
                        ];
                    }
                }
            }

            if ($request->filled('extras') && is_array($request->extras)) {
                $itemsData['extras'] = [];
                foreach ($request->extras as $extra) {
                    if (isset($extra['name']) && !empty(trim($extra['name'])) && isset($extra['price']) && (float)$extra['price'] > 0) {
                        $itemsData['extras'][] = [
                            'name' => trim($extra['name']),
                            'price' => (float)$extra['price']
                        ];
                    }
                }
            }

            if (!empty($itemsData)) {
                $reservation->items_data = json_encode($itemsData);
                $reservation->save();
            }

            try {
                $customer = Customer::find($validated['customer_id']);
                if ($customer && $totalPrice > 0) {
                    $customer->increment('balance', $totalPrice);
                    \Log::info('Müşteri bakiyesine borç eklendi', [
                        'customer_id' => $customer->id,
                        'amount' => $totalPrice,
                        'reservation_id' => $reservation->id,
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Müşteri bakiyesi güncellenemedi: ' . $e->getMessage());
            }

            $pdfLinks = null;
            try {
                $pdfLinks = $this->generateInvoiceAndContract($reservation);
                \Log::info('PDF linkleri oluşturuldu', [
                    'reservation_id' => $reservation->id,
                    'invoice_url' => $pdfLinks['invoice_url'] ?? null,
                    'contract_url' => $pdfLinks['contract_url'] ?? null,
                ]);
            } catch (\Exception $e) {
                \Log::error('PDF oluşturma hatası: ' . $e->getMessage(), [
                    'reservation_id' => $reservation->id,
                    'trace' => $e->getTraceAsString(),
                ]);
            }

            $invoiceUrl = null;
            if ($request->input('create_invoice') == '1') {
                try {
                    $invoice = $this->createInvoiceForReservation($reservation);
                    $invoiceUrl = $invoice->public_url; 
                    \Log::info('Fatura oluşturuldu (SMS için)', [
                        'reservation_id' => $reservation->id,
                        'invoice_id' => $invoice->id,
                        'public_url' => $invoiceUrl,
                    ]);
                } catch (\Exception $e) {
                    \Log::error('Fatura oluşturma hatası (SMS için): ' . $e->getMessage());
                    
                }
            }

            $reservationDate = $reservation->date ? $reservation->date->format('Y-m-d') : null;
            $reservationTime = '';
            if ($reservation->start_time && $reservation->end_time) {
                $reservationTime = $reservation->start_time . ' - ' . $reservation->end_time;
            } elseif ($reservation->start_time) {
                $reservationTime = $reservation->start_time;
            }

            $defaultContract = Contract::getDefaultForCompany(auth()->user()->company_id ?? null);
            $contractUrl = $defaultContract ? $defaultContract->public_url : ($pdfLinks['contract_url'] ?? null);

            $smsInvoiceUrl = $invoiceUrl ?? ($pdfLinks['invoice_url'] ?? null);
            
            \Log::info('WhatsApp mesaj gönderme hazırlığı', [
                'reservation_id' => $reservation->id,
                'has_customer' => $reservation->customer ? true : false,
                'customer_phone' => $reservation->customer->phone ?? null,
                'has_pdf_links' => $pdfLinks !== null,
                'invoice_url' => $smsInvoiceUrl,
                'contract_url' => $contractUrl,
                'is_view_page' => $invoiceUrl !== null,
            ]);

            $smsBalanceLow = false;
            $user = Auth::user();
            if ($user && !$user->isSuperAdmin()) {
                
                $hasCustomerPhone = $reservation->customer && $reservation->customer->phone;

                $requiredSms = 0;
                if ($hasCustomerPhone) $requiredSms += 1;
                
                if ($requiredSms > 0 && !$user->hasSmsBalance($requiredSms)) {
                    $smsBalanceLow = true;
                    \Log::warning('⚠️ SMS bakiyesi yetersiz', [
                        'user_id' => $user->id,
                        'sms_balance' => $user->sms_balance,
                        'has_customer_phone' => $hasCustomerPhone,
                        
                    ]);
                }
            }

            $customerWhatsAppSent = false;
            $customerSmsSent = false;
            $smsBalanceDeductedCount = 0; 
            
            $smsService = app(SmsService::class);

            if ($reservation->customer && $reservation->customer->phone) {
                try {
                    
                    $salonName = $user->salon_name ?? $user->name;
                    $salonAddress = $user->salon_address ?? '';

                    $whatsappMessage = $salonName . "\n\n";
                    if ($salonAddress) {
                        $whatsappMessage .= "Adres: " . $salonAddress . "\n\n";
                    }
                    $whatsappMessage .= $reservationDate . " tarihinde, " . $reservationTime . " saatinde randevunuz oluşturulmuştur.\n";
                    $whatsappMessage .= "Randevu saatinizden 5 dakika önce salonda hazır bulunmanızı rica ederiz.\n\n";

                    if ($smsInvoiceUrl) {
                        $whatsappMessage .= "Fatura:\n";
                        $whatsappMessage .= $smsInvoiceUrl . "\n\n";
                    }

                    if ($contractUrl) {
                        $whatsappMessage .= "Sözleşme:\n";
                        $whatsappMessage .= $contractUrl . "\n\n";
                    }
                    
                    $whatsappMessage .= "salonay.com iyi günler diler.";

                    $whatsappResult = $smsService->send(
                        $reservation->customer->phone,
                        $whatsappMessage,
                        null,
                        ['purpose' => 'appointment', 'user_id' => $user->id]
                    );
                    
                    if ($whatsappResult && $whatsappResult->success) {
                        $customerWhatsAppSent = true;
                        \Log::info('✅ Randevu oluşturulduğunda müşteriye WhatsApp mesajı gönderildi', [
                            'reservation_id' => $reservation->id,
                            'phone' => $reservation->customer->phone,
                            'invoice_url' => $smsInvoiceUrl,
                            'contract_url' => $contractUrl,
                        ]);
                    } else {
                        \Log::warning('⚠️ Randevu oluşturulduğunda müşteriye WhatsApp mesajı gönderilemedi', [
                            'reservation_id' => $reservation->id,
                            'phone' => $reservation->customer->phone,
                        ]);
                    }
                } catch (\Exception $e) {
                    \Log::error('❌ Müşteriye WhatsApp mesaj gönderme hatası: ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'phone' => $reservation->customer->phone,
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }

            if ($reservation->customer && $reservation->customer->phone) {
                
                $customerPhone = preg_replace('/[^0-9]/', '', $reservation->customer->phone);
                
                if (strlen($customerPhone) >= 10) {
                    if (!$smsBalanceLow) {
                        try {
                            $serviceName = $reservation->appointmentRequest && $reservation->appointmentRequest->service ? $reservation->appointmentRequest->service->name : 'Randevu';
                            $personelName = $reservation->personel ? $reservation->personel->name : '';
                            
                            $smsMessage = "Sayın {$reservation->customer->name}, {$reservationDate} tarihinde {$reservationTime} saatinde {$serviceName} randevunuz onaylanmıştır. ";
                            if ($personelName) {
                                $smsMessage .= "Personel: {$personelName}. ";
                            }
                            if ($smsInvoiceUrl) {
                                $smsMessage .= "Fatura: {$smsInvoiceUrl} ";
                            }
                            if ($contractUrl) {
                                $smsMessage .= "Sözleşme: {$contractUrl} ";
                            }
                            $smsMessage .= "Salonay";
                            
                            $result = $smsService->send(
                                $customerPhone,
                                $smsMessage,
                                null,
                                [
                                    'purpose' => 'appointment',
                                    'user_id' => $user->id
                                ]
                            );
                            
                            if ($result && $result->success) {
                                $customerSmsSent = true;
                                $smsBalanceDeductedCount += 1;
                                \Log::info('✅ Randevu oluşturulduğunda müşteriye SMS mesajı gönderildi', [
                                    'reservation_id' => $reservation->id,
                                    'phone' => $customerPhone,
                                    'invoice_url' => $smsInvoiceUrl,
                                    'contract_url' => $contractUrl,
                                    'sms_balance_deducted' => 1,
                                ]);
                            } else {
                                \Log::warning('⚠️ Randevu oluşturulduğunda müşteriye SMS mesajı gönderilemedi', [
                                    'reservation_id' => $reservation->id,
                                    'phone' => $customerPhone,
                                ]);
                            }
                        } catch (\Exception $e) {
                            \Log::error('❌ Müşteriye SMS mesaj gönderme hatası: ' . $e->getMessage(), [
                                'reservation_id' => $reservation->id,
                                'phone' => $customerPhone,
                                'trace' => $e->getTraceAsString()
                            ]);
                        }
                    } else {
                        \Log::warning('⚠️ SMS bakiyesi yetersiz, müşteriye SMS mesajı gönderilemedi', [
                            'reservation_id' => $reservation->id,
                            'customer_id' => $reservation->customer_id,
                            'phone' => $customerPhone,
                        ]);
                    }
                } else {
                    \Log::warning('⚠️ Müşteri telefon numarası geçersiz', [
                        'reservation_id' => $reservation->id,
                        'customer_id' => $reservation->customer_id,
                        'phone' => $customerPhone,
                    ]);
                }
            } else {
                \Log::warning('⚠️ Müşteri telefon numarası yok, mesaj gönderilemedi', [
                    'reservation_id' => $reservation->id,
                    'customer_id' => $reservation->customer_id,
                ]);
            }

            try {
                ActivityLog::create([
                    'user_id' => $user->id,
                    'action' => 'create_reservation',
                    'description' => "Yeni randevu oluşturuldu: {$reservation->code}",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('Activity log kaydedilemedi: ' . $e->getMessage());
            }

            $invoiceCreated = false;
            if ($request->input('create_invoice') == '1' && !$invoiceUrl) {
                
                try {
                    $invoice = $this->createInvoiceForReservation($reservation);
                    $invoiceCreated = true;
                    $invoiceUrl = $invoice->public_url;
                } catch (\Exception $e) {
                    \Log::error('Fatura oluşturma hatası: ' . $e->getMessage());
                }
            } elseif ($invoiceUrl) {
                
                $invoiceCreated = true;
            }

            $successMessage = 'Randevu başarıyla oluşturuldu.';
            if ($invoiceCreated) {
                $successMessage .= ' Fatura da oluşturuldu ve müşteriye gönderildi.';
            }

            if ($smsBalanceLow) {
                $smsWarningMessage = 'SMS bakiyeniz kalmamıştır. SMS gönderilemedi.';
                return redirect()->route('admin.appointments.index')
                    ->with('success', $successMessage)
                    ->with('sms_warning', $smsWarningMessage);
            }

            return redirect()->route('admin.appointments.index')
                ->with('success', $successMessage);
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Reservation Store Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->back()
                ->withInput()
                ->with('error', 'Randevu oluşturulurken bir hata oluştu: ' . $e->getMessage());
        }
    }

    private function createInvoiceForReservation($reservation, $service = null, $customer = null, $user = null)
    {
        try {
            
            if (!$user) {
                $user = Auth::user();
            }
            
            if (!$customer) {
                $reservation->load('customer');
                $customer = $reservation->customer;
            }

            if (!$reservation->relationLoaded('customer')) {
                $reservation->load('customer');
            }

            $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);

            $publicToken = Str::random(32);

            $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
            $logo = $settings['invoice_logo'] ?? null;
            $taxOffice = $settings['invoice_tax_office'] ?? '';
            $taxNumber = $settings['invoice_tax_number'] ?? '';

            if ($user && ($user->role === 'güzellik_salonu' || $user->isBeautySalon())) {
                $address = $user->salon_address ?? ($settings['invoice_address'] ?? '');
                $companyName = $user->salon_name ?? $user->name;
            } else {
                
                $reservation->load('salon');
                $address = $reservation->salon->address ?? ($settings['invoice_address'] ?? '');
                $companyName = $reservation->salon->name ?? 'Salon';
            }

            $items = [];

            $actualPrice = (float)($reservation->total_price ?? 0);

            // Hizmet bilgisini bul
            $serviceName = 'Hizmet';
            $serviceFound = false;

            if ($service) {
                $serviceName = $service->name;
                $serviceFound = true;
            } else {
                // Reservation'dan appointmentRequest'i yükle
                $reservation->load('appointmentRequest.service');
                
                if ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
                    $serviceName = $reservation->appointmentRequest->service->name;
                    $serviceFound = true;
                } else {
                    // items_data'dan hizmet bilgisini kontrol et
                    $itemsData = json_decode($reservation->items_data ?? '{}', true);
                    if (isset($itemsData['service']) && is_array($itemsData['service'])) {
                        $serviceData = $itemsData['service'];
                        $serviceName = $serviceData['name'] ?? 'Hizmet';
                        $serviceFound = true;
                    } elseif (isset($itemsData['services']) && is_array($itemsData['services']) && !empty($itemsData['services'])) {
                        // Birden fazla hizmet varsa ilkini al
                        $firstService = $itemsData['services'][0];
                        $serviceName = $firstService['name'] ?? 'Hizmet';
                        $serviceFound = true;
                    }
                }
            }

            // Fatura kalemi oluştur
            $items[] = [
                'name' => $serviceName,
                'quantity' => 1,
                'price' => $actualPrice,
                'total' => $actualPrice,
            ];

            $invoiceData = [
                'reservation' => $reservation,
                'customer' => $customer,
                'invoice' => null,
                'items' => $items,
                'subtotal' => $actualPrice,
                'grandTotal' => $actualPrice,
                'depositPaid' => $reservation->deposit_paid ?? 0,
                'logo' => $logo,
                'address' => $address,
                'salonName' => $companyName,
                'salonAddress' => $address,
                'taxOffice' => $taxOffice,
                'taxNumber' => $taxNumber,
                'customerName' => $customer ? $customer->name : ($reservation->customer ? $reservation->customer->name : 'Müşteri'),
                'invoiceDate' => now()->format('d.m.Y'),
            ];

            $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
            $pdf->setOption('fontDir', public_path('fonts'));
            $pdf->setOption('defaultFont', 'DejaVu Sans');

            $pdfPath = 'invoices/' . $invoiceNumber . '.pdf';
            Storage::disk('public')->put($pdfPath, $pdf->output());

            $publicUrl = $this->getInvoicePublicUrl($publicToken);

            $invoice = Invoice::create([
                'reservation_id' => $reservation->id,
                'customer_id' => $customer ? $customer->id : $reservation->customer_id,
                'invoice_number' => $invoiceNumber,
                'total_amount' => $reservation->total_price,
                'deposit_paid' => $reservation->deposit_paid ?? 0,
                'remaining_amount' => $reservation->total_price - ($reservation->deposit_paid ?? 0),
                'invoice_date' => now(),
                'pdf_path' => $pdfPath,
                'public_token' => $publicToken,
                'public_url' => $publicUrl,
                'status' => 'sent',
                'items_json' => json_encode($items),
                'created_by' => $user ? $user->id : Auth::id(),
            ]);
            
            return $invoice;
        } catch (\Exception $e) {
            \Log::error('Fatura oluşturma hatası', [
                'reservation_id' => $reservation->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    public function checkAvailableTimes(Request $request)
    {
        try {
            
            if ($request->isJson()) {
                $date = $request->json('date');
                $salonId = $request->json('salon_id');
            } else {
                $request->validate([
                    'date' => 'required|date',
                    'salon_id' => 'required|exists:salons,id',
                ]);
                $date = $request->date;
                $salonId = $request->salon_id;
            }

            if (!$date || !$salonId) {
                return response()->json([
                    'success' => false,
                    'booked_slots' => [],
                    'error' => 'Tarih ve salon ID gerekli'
                ], 400);
            }

            $dateColumn = $this->getReservationDateColumn();

            $user = Auth::user();
            $query = Reservation::where($dateColumn, $date)
                ->where('salon_id', $salonId)
                ->where('status', '!=', 'cancelled');

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $appointments = $query->get(['id', 'start_time', 'end_time', 'status']);

            $bookedSlots = [];
            foreach ($appointments as $reservation) {
                try {
                    if (!$reservation->start_time || !$reservation->end_time) {
                        continue;
                    }
                    
                    $start = strtotime($reservation->start_time);
                    $end = strtotime($reservation->end_time);
                    
                    if ($start === false || $end === false || $start >= $end) {
                        continue;
                    }

                    for ($time = $start; $time < $end; $time += 1800) { 
                        $slot = date('H:i', $time);
                        if (!in_array($slot, $bookedSlots)) {
                            $bookedSlots[] = $slot;
                        }
                    }
                } catch (\Exception $e) {
                    continue;
                }
            }

            sort($bookedSlots);
            $bookedSlots = array_values(array_unique($bookedSlots));

            return response()->json([
                'success' => true,
                'booked_slots' => $bookedSlots
            ]);
        } catch (\Exception $e) {
            \Log::error('Check Available Times Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'success' => false,
                'booked_slots' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $user = Auth::user();

            if ($user->role === 'güzellik_salonu') {
                $query = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room']);
                $query->where('beauty_salon_id', $user->id);
            } else {
                
                $query = Reservation::with(['customer', 'salon', 'appointmentRequest.service', 'personel', 'room']);
                
                if ($user->role === 'randevucu' || $user->role === 'admin') {
                    $query->where('created_by', $user->id);
                }
            }
            
            $reservation = $query->findOrFail($id);

            if (request()->ajax() || request()->wantsJson()) {
                
                $itemsData = json_decode($reservation->items_data ?? '{}', true);
                if (!is_array($itemsData) || empty($itemsData)) {
                    
                    $itemsData = json_decode($reservation->notes ?? '{}', true);
                    if (!is_array($itemsData)) {
                        $itemsData = [];
                    }
                }

                $items = [];

                if ($user->role === 'güzellik_salonu') {
                    
                    if (isset($itemsData['service']) && is_array($itemsData['service'])) {
                        
                        $serviceData = $itemsData['service'];
                        $serviceName = $serviceData['name'] ?? 'Hizmet';
                        $servicePrice = $serviceData['price'] ?? $reservation->total_price ?? 0;
                        $items[] = [
                            'name' => $serviceName,
                            'quantity' => '1',
                            'price' => $servicePrice,
                            'total' => $servicePrice
                        ];
                    } elseif (isset($itemsData['service_id']) || isset($itemsData['service_name'])) {
                        
                        $serviceName = $itemsData['service_name'] ?? 'Hizmet';
                        $servicePrice = $itemsData['service_price'] ?? $reservation->total_price ?? 0;
                        $items[] = [
                            'name' => $serviceName,
                            'quantity' => '1',
                            'price' => $servicePrice,
                            'total' => $servicePrice
                        ];
                    } elseif ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
                        
                        $service = $reservation->appointmentRequest->service;
                        $items[] = [
                            'name' => $service->name,
                            'quantity' => '1',
                            'price' => $service->price,
                            'total' => $service->price
                        ];
                    } else {
                        
                        $items[] = [
                            'name' => 'Hizmet',
                            'quantity' => '1',
                            'price' => $reservation->total_price ?? 0,
                            'total' => $reservation->total_price ?? 0
                        ];
                    }
                } else {
                    
                    $mealTotal = 0;
                    $organizationPrice = $itemsData['organization_price'] ?? 0;
                    $extraTotal = 0;
                    
                    if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
                        foreach ($itemsData['meals'] as $meal) {
                            if (isset($meal['quantity']) && isset($meal['price'])) {
                                $mealTotal += $meal['quantity'] * $meal['price'];
                            }
                        }
                    }
                    
                    if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
                        foreach ($itemsData['extras'] as $extra) {
                            if (isset($extra['price'])) {
                                $extraTotal += $extra['price'];
                            }
                        }
                    }
                    
                    $salonPrice = $reservation->total_price - ($mealTotal + $organizationPrice + $extraTotal);
                    
                    if ($salonPrice > 0) {
                        $items[] = ['name' => 'Salon Ücreti', 'quantity' => '-', 'price' => $salonPrice, 'total' => $salonPrice];
                    }
                    
                    if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
                        foreach ($itemsData['meals'] as $meal) {
                            if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                                $items[] = [
                                    'name' => $meal['name'],
                                    'quantity' => $meal['quantity'] . ' Kişi',
                                    'price' => $meal['price'],
                                    'total' => $meal['quantity'] * $meal['price']
                                ];
                            }
                        }
                    }
                    
                    if ($organizationPrice > 0) {
                        $items[] = ['name' => 'Organizasyon', 'quantity' => '-', 'price' => $organizationPrice, 'total' => $organizationPrice];
                    }
                    
                    if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
                        foreach ($itemsData['extras'] as $extra) {
                            if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                                $items[] = [
                                    'name' => $extra['name'],
                                    'quantity' => '-',
                                    'price' => $extra['price'],
                                    'total' => $extra['price']
                                ];
                            }
                        }
                    }
                }

                $brideGroomName = $itemsData['bride_groom_name'] ?? null;
                $brideName = $itemsData['bride_name'] ?? null;
                $groomName = $itemsData['groom_name'] ?? null;
                $brideGroomPhone = $itemsData['bride_groom_phone'] ?? null;

                $reservation->load('appointmentRequest.service', 'personel', 'room');

                $formattedDate = null;
                if ($reservation->date) {
                    try {
                        if (is_string($reservation->date)) {
                            $formattedDate = $reservation->date;
                        } elseif (method_exists($reservation->date, 'format')) {
                            $formattedDate = $reservation->date->format('Y-m-d');
                        } else {
                            $formattedDate = (string) $reservation->date;
                        }
                    } catch (\Exception $e) {
                        $formattedDate = is_string($reservation->date) ? $reservation->date : null;
                    }
                }
                
                return response()->json([
                    'reservation' => [
                        'id' => $reservation->id,
                        'code' => $reservation->code ?? null,
                        'date' => $formattedDate,
                        'start_time' => $reservation->start_time,
                        'end_time' => $reservation->end_time,
                        'status' => $reservation->status,
                        'total_price' => $reservation->total_price,
                        'deposit_paid' => $reservation->deposit_paid ?? 0,
                        'remaining_amount' => $reservation->remaining_amount ?? 0,
                        'notes' => $reservation->notes,
                        'personel' => $reservation->personel ? ['id' => $reservation->personel->id, 'name' => $reservation->personel->name] : null,
                        'room' => $reservation->room ? ['id' => $reservation->room->id, 'name' => $reservation->room->name] : null,
                        'appointment_request' => $reservation->appointmentRequest ? [
                            'service' => $reservation->appointmentRequest->service ? [
                                'name' => $reservation->appointmentRequest->service->name,
                                'price' => $reservation->appointmentRequest->service->price
                            ] : null
                        ] : null
                    ],
                    'customer' => $reservation->customer ? [
                        'id' => $reservation->customer->id,
                        'name' => $reservation->customer->name,
                        'surname' => $reservation->customer->surname,
                        'phone' => $reservation->customer->phone,
                        'email' => $reservation->customer->email
                    ] : null,
                    'salon' => $reservation->salon ? [
                        'id' => $reservation->salon->id,
                        'name' => $reservation->salon->name,
                        'address' => $reservation->salon->address ?? null
                    ] : null,
                    'items' => $items,
                    'total_price' => $reservation->total_price,
                    'deposit_paid' => $reservation->deposit_paid ?? 0,
                    'remaining_amount' => $reservation->remaining_amount ?? 0,
                    
                ]);
            }

            return view('admin.appointments.show', compact('reservation'));
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            \Log::error('Reservation Not Found: ' . $e->getMessage(), [
                'id' => $id,
                'user_id' => Auth::id(),
            ]);

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'error' => 'Randevu bulunamadı.',
                    'message' => 'Bu randevuu görüntüleme yetkiniz olmayabilir veya randevu silinmiş olabilir.'
                ], 404);
            }
            
            abort(404, 'Randevu bulunamadı.');
        } catch (\Exception $e) {
            \Log::error('Reservation Show Error: ' . $e->getMessage(), [
                'id' => $id,
                'user_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'error' => 'Randevu bilgileri yüklenirken bir hata oluştu.',
                    'message' => $e->getMessage()
                ], 500);
            }
            
            return redirect()->route('admin.appointments.index')->with('error', 'Randevu bilgileri yüklenirken bir hata oluştu.');
        }
    }
    
    public function confirm($id)
    {
        try {
            $user = Auth::user();
            $query = Reservation::with(['customer', 'salon', 'appointmentRequest.service', 'personel', 'room']);

            if ($user->role === 'randevucu') {
                $query->where('created_by', $user->id);
            }

            if ($user->role === 'güzellik_salonu') {
                $query->where('beauty_salon_id', $user->id);
            }
            
            $reservation = $query->findOrFail($id);
            
            $reservation->update([
                'status' => 'confirmed'
            ]);

            if ($reservation->beauty_salon_id && $reservation->customer_id) {
                try {
                    $salon = \App\Models\User::find($reservation->beauty_salon_id);
                    $customer = $reservation->customer;
                    
                    if ($salon && $customer && $salon->salon_slug) {
                        
                        $existingRating = \App\Models\SalonRating::where('reservation_id', $reservation->id)
                            ->where('salon_id', $reservation->beauty_salon_id)
                            ->first();
                        
                        if (!$existingRating) {
                            
                            $token = \App\Models\SalonRating::generateToken();
                            $referralCode = \App\Models\SalonRating::generateReferralCode();
                            $expiresAt = now()->addDays(30); 
                            
                            $rating = \App\Models\SalonRating::create([
                                'salon_id' => $reservation->beauty_salon_id,
                                'customer_id' => $reservation->customer_id,
                                'reservation_id' => $reservation->id,
                                'token' => $token,
                                'referral_code' => $referralCode,
                                'expires_at' => $expiresAt,
                                'is_used' => false,
                                'customer_name' => $customer->name,
                                'customer_phone' => $customer->phone,
                            ]);

                            try {
                                $ratingUrl = route('public.salon.rating', [
                                    'slug' => $salon->salon_slug,
                                    'token' => $token
                                ]);
                                
                                $salonName = $salon->salon_name ?? $salon->name ?? 'Salon';
                                
                                $whatsappMessage = "{$salonName}\n\n";
                                $whatsappMessage .= "Randevunuz başarıyla tamamlanmıştır bizi oylamak için aşağıdaki linke ulaşabilirsiniz.\n\n";
                                $whatsappMessage .= "{$ratingUrl}\n\n";
                                $whatsappMessage .= "🎁 Referans Kodunuz: {$referralCode}\n";
                                $whatsappMessage .= "Bu kodu arkadaşlarınızla paylaşın, onlar randevu aldığında size 3 puan kazandırır!";
                                
                                $smsService = app(\App\Services\SmsService::class);
                                $whatsappResult = $smsService->send(
                                    $customer->phone,
                                    $whatsappMessage,
                                    null,
                                    [
                                        'purpose' => 'appointment',
                                        'user_id' => $reservation->beauty_salon_id
                                    ]
                                );
                                
                                if ($whatsappResult && $whatsappResult->success) {
                                    \Log::info('Yorum linki WhatsApp ile gönderildi', [
                                        'reservation_id' => $reservation->id,
                                        'customer_phone' => $customer->phone,
                                        'rating_url' => $ratingUrl
                                    ]);
                                } else {
                                    \Log::warning('Yorum linki WhatsApp ile gönderilemedi', [
                                        'reservation_id' => $reservation->id,
                                        'customer_phone' => $customer->phone,
                                        'error' => $whatsappResult->message ?? 'Bilinmeyen hata'
                                    ]);
                                }
                            } catch (\Exception $whatsappError) {
                                \Log::warning('Yorum linki WhatsApp gönderilirken hata oluştu: ' . $whatsappError->getMessage(), [
                                    'reservation_id' => $reservation->id,
                                    'trace' => $whatsappError->getTraceAsString()
                                ]);
                            }
                        } else {
                            
                            try {
                                $ratingUrl = route('public.salon.rating', [
                                    'slug' => $salon->salon_slug,
                                    'token' => $existingRating->token
                                ]);
                                
                                $salonName = $salon->salon_name ?? $salon->name ?? 'Salon';
                                
                                $whatsappMessage = "{$salonName}\n\n";
                                $whatsappMessage .= "Randevunuz başarıyla tamamlanmıştır bizi oylamak için aşağıdaki linke ulaşabilirsiniz.\n\n";
                                $whatsappMessage .= "{$ratingUrl}";
                                
                                $smsService = app(\App\Services\SmsService::class);
                                $whatsappResult = $smsService->send(
                                    $customer->phone,
                                    $whatsappMessage,
                                    null,
                                    [
                                        'purpose' => 'appointment',
                                        'user_id' => $reservation->beauty_salon_id
                                    ]
                                );
                                
                                if ($whatsappResult && $whatsappResult->success) {
                                    \Log::info('Mevcut yorum linki WhatsApp ile gönderildi', [
                                        'reservation_id' => $reservation->id,
                                        'customer_phone' => $customer->phone,
                                        'rating_url' => $ratingUrl
                                    ]);
                                }
                            } catch (\Exception $whatsappError) {
                                \Log::warning('Mevcut yorum linki WhatsApp gönderilirken hata oluştu: ' . $whatsappError->getMessage());
                            }
                        }
                    }
                } catch (\Exception $ratingError) {
                    \Log::warning('Yorum linki oluşturulamadı: ' . $ratingError->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'trace' => $ratingError->getTraceAsString()
                    ]);
                }
            }
            
            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'confirm_reservation',
                    'description' => "Randevu onaylandı: ID {$reservation->id}",
                    'ip_address' => request()->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }
            
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Randevu başarıyla onaylandı.',
                ]);
            }
            
            return redirect()->back()->with('success', 'Randevu başarıyla onaylandı.');
        } catch (\Exception $e) {
            \Log::error('Randevu onaylama hatası: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Randevu onaylanırken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()->with('error', 'Randevu onaylanırken bir hata oluştu.');
        }
    }
    
    public function complete($id)
    {
        try {
            $user = Auth::user();
            $query = Reservation::with(['customer', 'salon', 'appointmentRequest.service', 'personel', 'room']);

            if ($user->role === 'randevucu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }

            if ($user->role === 'güzellik_salonu') {
                $query->where('beauty_salon_id', $user->id);
            }
            
            $reservation = $query->findOrFail($id);

            $totalPrice = $reservation->total_price ?? 0;

            if ($totalPrice > 0) {
                try {
                    
                    $cashAccount = Account::getCashAccount($user->id);

                    $description = "Randevu Ödemesi - Randevu #{$reservation->code}";
                    if ($reservation->customer) {
                        $description .= " | Müşteri: {$reservation->customer->name}";
                    }
                    if ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
                        $description .= " | Hizmet: {$reservation->appointmentRequest->service->name}";
                    }
                    
                    Transaction::create([
                        'account_id' => $cashAccount->id,
                        'reservation_id' => $reservation->id,
                        'customer_id' => $reservation->customer_id,
                        'type' => 'income',
                        'amount' => $totalPrice,
                        'date' => now()->format('Y-m-d'),
                        'description' => $description,
                        'created_by' => $user->id,
                    ]);

                    $cashAccount->increment('balance', $totalPrice);

                    if ($reservation->customer && $reservation->customer->balance > 0) {
                        $reservation->customer->decrement('balance', min($totalPrice, $reservation->customer->balance));
                    }

                    $reservation->update([
                        'remaining_amount' => 0,
                        'deposit_paid' => 0
                    ]);
                    
                    \Log::info('Ödeme kasaya eklendi', [
                        'reservation_id' => $reservation->id,
                        'amount' => $totalPrice,
                        'cash_account_id' => $cashAccount->id,
                    ]);
                    
                } catch (\Exception $e) {
                    \Log::error('Ödeme kaydedilemedi: ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            } else {
                
                if ($totalPrice > 0) {
                    try {
                        
                        $cashAccount = Account::getCashAccount($user->id);

                        $description = "Randevu Ödemesi - Randevu #{$reservation->code}";
                        if ($reservation->customer) {
                            $description .= " | Müşteri: {$reservation->customer->name}";
                        }
                        if ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
                            $description .= " | Hizmet: {$reservation->appointmentRequest->service->name}";
                        }
                        
                        Transaction::create([
                            'account_id' => $cashAccount->id,
                            'reservation_id' => $reservation->id,
                            'customer_id' => $reservation->customer_id,
                            'type' => 'income',
                            'amount' => $totalPrice,
                            'date' => now()->format('Y-m-d'),
                            'description' => $description,
                            'created_by' => $user->id,
                        ]);

                        $cashAccount->increment('balance', $totalPrice);

                        if ($reservation->customer && $reservation->customer->balance > 0) {
                            $reservation->customer->decrement('balance', min($totalPrice, $reservation->customer->balance));
                        }

                        $reservation->update([
                            'remaining_amount' => 0,
                            'deposit_paid' => 0
                        ]);
                        
                        \Log::info('Ödeme kasaya eklendi (remainingAmount <= 0 ama totalPrice > 0)', [
                            'reservation_id' => $reservation->id,
                            'amount' => $totalPrice,
                            'cash_account_id' => $cashAccount->id,
                        ]);
                        
                    } catch (\Exception $e) {
                        \Log::error('Ödeme kaydedilemedi: ' . $e->getMessage(), [
                            'reservation_id' => $reservation->id,
                            'trace' => $e->getTraceAsString()
                        ]);
                    }
                }
            }
            
            // Referans ödülü kontrolü - randevu tamamlandığında
            if ($reservation->customer_id) {
                try {
                    // AppointmentRequest'ten referral_code kontrolü
                    $appointmentRequest = \App\Models\AppointmentRequest::where('reservation_id', $reservation->id)->first();
                    
                    if ($appointmentRequest && $appointmentRequest->referral_code) {
                        // Daha önce bu referans kodu için ödül verilmiş mi kontrol et
                        $existingReferral = \App\Models\Referral::where('referral_code', $appointmentRequest->referral_code)
                            ->where('referred_id', $reservation->customer_id)
                            ->where('beauty_salon_id', $reservation->beauty_salon_id)
                            ->first();
                        
                        if (!$existingReferral) {
                            // Referans koduna sahip müşteriyi bul
                            $referrerRating = \App\Models\SalonRating::where('referral_code', $appointmentRequest->referral_code)
                                ->where('salon_id', $reservation->beauty_salon_id)
                                ->first();
                            
                            if ($referrerRating && $referrerRating->customer_id) {
                                $referrerCustomer = \App\Models\Customer::find($referrerRating->customer_id);
                                
                                if ($referrerCustomer) {
                                    // Referans eden müşterinin bakiyesine 3 puan ekle
                                    $referrerCustomer->increment('balance', 3);
                                    
                                    // Referral kaydı oluştur
                                    \App\Models\Referral::create([
                                        'beauty_salon_id' => $reservation->beauty_salon_id,
                                        'referrer_id' => $referrerCustomer->id,
                                        'referred_id' => $reservation->customer_id,
                                        'referral_code' => $appointmentRequest->referral_code,
                                        'status' => 'completed',
                                        'first_appointment_id' => $reservation->id,
                                        'first_visit_at' => now(),
                                        'rewarded_at' => now(),
                                    ]);
                                    
                                    \Log::info('Referans ödülü verildi (complete)', [
                                        'referrer_id' => $referrerCustomer->id,
                                        'referred_id' => $reservation->customer_id,
                                        'referral_code' => $appointmentRequest->referral_code,
                                        'reward_amount' => 3,
                                    ]);
                                }
                            }
                        }
                    }
                } catch (\Exception $referralError) {
                    \Log::warning('Referans ödülü verilirken hata (complete): ' . $referralError->getMessage());
                }
            }

            $reservation->update([
                'status' => 'completed'
            ]);

            // Premium özellikler entegrasyonu
            try {
                $premiumService = app(PremiumFeaturesIntegrationService::class);
                $premiumService->onAppointmentCompleted($reservation);
            } catch (\Exception $e) {
                \Log::error('Premium özellik entegrasyonu hatası (complete): ' . $e->getMessage(), [
                    'reservation_id' => $reservation->id,
                    'trace' => $e->getTraceAsString(),
                ]);
            }

            if ($reservation->beauty_salon_id && $reservation->customer_id) {
                try {
                    $salon = \App\Models\User::find($reservation->beauty_salon_id);
                    $customer = $reservation->customer;
                    
                    if ($salon && $customer && $salon->salon_slug) {
                        $token = \App\Models\SalonRating::generateToken();
                        $referralCode = \App\Models\SalonRating::generateReferralCode();
                        $expiresAt = now()->addDays(30); 
                        
                        \App\Models\SalonRating::create([
                            'salon_id' => $reservation->beauty_salon_id,
                            'customer_id' => $reservation->customer_id,
                            'reservation_id' => $reservation->id,
                            'token' => $token,
                            'referral_code' => $referralCode,
                            'expires_at' => $expiresAt,
                            'is_used' => false,
                            'customer_name' => $customer->name,
                            'customer_phone' => $customer->phone,
                        ]);

                        try {
                            $ratingUrl = route('public.salon.rating', [
                                'slug' => $salon->salon_slug,
                                'token' => $token
                            ]);
                            
                            $salonName = $salon->salon_name ?? $salon->name ?? 'Salon';
                            
                            $whatsappMessage = "{$salonName}\n\n";
                            $whatsappMessage .= "Randevunuz başarıyla tamamlanmıştır bizi oylamak için aşağıdaki linke ulaşabilirsiniz.\n\n";
                            $whatsappMessage .= "{$ratingUrl}\n\n";
                            $whatsappMessage .= "🎁 Referans Kodunuz: {$referralCode}\n";
                            $whatsappMessage .= "Bu kodu arkadaşlarınızla paylaşın, onlar randevu aldığında size 3 puan kazandırır!";
                            
                            $smsService = app(\App\Services\SmsService::class);
                            $whatsappResult = $smsService->send(
                                $customer->phone,
                                $whatsappMessage,
                                null,
                                [
                                    'purpose' => 'appointment',
                                    'user_id' => $reservation->beauty_salon_id
                                ]
                            );
                            
                            if ($whatsappResult && $whatsappResult->success) {
                                \Log::info('Yorum linki WhatsApp ile gönderildi (complete)', [
                                    'reservation_id' => $reservation->id,
                                    'customer_phone' => $customer->phone,
                                    'rating_url' => $ratingUrl
                                ]);
                            } else {
                                \Log::warning('Yorum linki WhatsApp ile gönderilemedi (complete)', [
                                    'reservation_id' => $reservation->id,
                                    'customer_phone' => $customer->phone,
                                    'error' => $whatsappResult->message ?? 'Bilinmeyen hata'
                                ]);
                            }
                        } catch (\Exception $smsError) {
                            \Log::warning('Rating linki gönderilirken hata oluştu (complete): ' . $smsError->getMessage(), [
                                'reservation_id' => $reservation->id,
                                'trace' => $smsError->getTraceAsString()
                            ]);
                        }
                    }
                } catch (\Exception $ratingError) {
                    \Log::warning('Rating linki oluşturulamadı: ' . $ratingError->getMessage());
                }
            }
            
            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'complete_reservation',
                    'description' => "Randevu tamamlandı: #{$reservation->code}" . ($totalPrice > 0 ? " - Ödeme: " . number_format($totalPrice, 2, ',', '.') . " ₺" : ""),
                    'ip_address' => request()->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }
            
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Randevu başarıyla tamamlandı.' . ($totalPrice > 0 ? ' ' . number_format($totalPrice, 2, ',', '.') . ' ₺ kasaya eklendi.' : ''),
                ]);
            }
            
            return redirect()->back()->with('success', 'Randevu başarıyla tamamlandı.' . ($totalPrice > 0 ? ' ' . number_format($totalPrice, 2, ',', '.') . ' ₺ kasaya eklendi.' : ''));
        } catch (\Exception $e) {
            \Log::error('Randevu tamamlama hatası: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Randevu tamamlanırken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()->with('error', 'Randevu tamamlanırken bir hata oluştu.');
        }
    }

    public function edit($id)
    {
        $user = Auth::user();
        $query = Reservation::with(['extras', 'customer', 'appointmentRequest.service', 'personel', 'room']);

        if ($user->role === 'randevucu') {
            $query->where('created_by', $user->id);
        }

        if ($user->role === 'güzellik_salonu') {
            $query->where('beauty_salon_id', $user->id);
        }
        
        $reservation = $query->findOrFail($id);

        $customerQuery = Customer::query();
        if ($user->role === 'randevucu') {
            $customerQuery->where('created_by', $user->id);
        } elseif ($user->role === 'güzellik_salonu') {
            if (\Schema::hasColumn('customers', 'beauty_salon_id')) {
                $customerQuery->where('beauty_salon_id', $user->id);
            } else {
                $customerQuery->where('created_by', $user->id);
            }
        }
        $customers = $customerQuery->orderBy('name')->get();

        if ($user->role === 'güzellik_salonu') {
            $personels = User::where('beauty_salon_id', $user->id)
                ->where('role', 'personel')
                ->orderBy('name')
                ->get();
            
            $rooms = \App\Models\Room::where('beauty_salon_id', $user->id)
                ->orderBy('name')
                ->get();
            
            $services = \App\Models\Service::where('beauty_salon_id', $user->id)
                ->orderBy('name')
                ->get();
            
            return view('admin.appointments.edit', compact('reservation', 'customers', 'personels', 'rooms', 'services'));
        }

        $salonQuery = Salon::where('is_active', true);
        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $salonQuery->where('created_by', $user->id);
        }
        $salons = $salonQuery->get();
        
        $menus = Menu::where('is_active', true)->get();
        $extras = Extra::where('is_active', true)->get();

        return view('admin.appointments.edit', compact('reservation', 'customers', 'salons', 'menus', 'extras'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = Reservation::query();

        if ($user->role === 'randevucu') {
            $query->where('created_by', $user->id);
        }

        if ($user->role === 'güzellik_salonu') {
            $query->where('beauty_salon_id', $user->id);
        }
        
        $reservation = $query->findOrFail($id);

        if ($user->role === 'randevucu') {
            $customer = Customer::where('id', $request->customer_id)
                ->where('created_by', $user->id)
                ->firstOrFail();
        }

        if ($user->role === 'güzellik_salonu') {
            $validated = $request->validate([
                'customer_id' => 'required|exists:customers,id',
                'service_id' => 'nullable|exists:services,id',
                'personel_id' => 'nullable|exists:users,id',
                'room_id' => 'nullable|exists:rooms,id',
                'date' => 'required|date',
                'start_time' => 'required',
                'end_time' => 'nullable',
                'guest_count' => 'nullable|integer|min:1',
                'total_price' => 'nullable|numeric|min:0',
                'deposit_paid' => 'nullable|numeric|min:0',
                'status' => 'required|in:open,confirmed,cancelled,completed',
                'notes' => 'nullable|string',
            ]);

            if (empty($validated['end_time']) && !empty($validated['start_time'])) {
                // Bitiş saati her zaman 30 dakika sonrası olarak ayarla
                try {
                    $startTime = \Carbon\Carbon::parse($validated['start_time']);
                    $validated['end_time'] = $startTime->addMinutes(30)->format('H:i:s');
                } catch (\Exception $e) {
                    
                }
            }

            if (empty($validated['total_price']) && !empty($validated['service_id'])) {
                $service = \App\Models\Service::find($validated['service_id']);
                if ($service) {
                    $validated['total_price'] = $service->price;
                }
            }

            $itemsData = json_decode($reservation->items_data ?? '{}', true);
            if (!empty($validated['service_id'])) {
                $service = \App\Models\Service::find($validated['service_id']);
                if ($service) {
                    $itemsData['service_id'] = $service->id;
                    $itemsData['service_name'] = $service->name;
                    $itemsData['service_price'] = $service->price;
                }
            }
            if (!empty($validated['personel_id'])) {
                $itemsData['personel_id'] = $validated['personel_id'];
            }
            if (!empty($validated['room_id'])) {
                $itemsData['room_id'] = $validated['room_id'];
            }
            $validated['items_data'] = json_encode($itemsData);
        } else {
            
            $validated = $request->validate([
                'customer_id' => 'required|exists:customers,id',
                'salon_id' => 'required|exists:salons,id',
                'date' => 'required|date',
                'start_time' => 'required',
                'end_time' => 'required',
                'guest_count' => 'required|integer|min:1',
                'total_price' => 'required|numeric|min:0',
                'deposit_paid' => 'nullable|numeric|min:0',
                'status' => 'required|in:open,confirmed,cancelled,completed',
                'notes' => 'nullable|string',
                'extras' => 'nullable|array',
                
            ]);

        }

        $totalPrice = $validated['total_price'] ?? $reservation->total_price ?? 0;
        $depositPaid = $validated['deposit_paid'] ?? $reservation->deposit_paid ?? 0;
        $validated['remaining_amount'] = $totalPrice - $depositPaid;

        if ($user->role === 'güzellik_salonu') {
            $validated['beauty_salon_id'] = $user->id;
            $validated['salon_id'] = null; 
        }

        if (!Schema::hasColumn('reservations', 'updated_by')) {
            unset($validated['updated_by']);
        } else {
            $validated['updated_by'] = Auth::id();
        }

        $reservation->update($validated);

        $reservation->extras()->detach();
        if ($request->filled('extras')) {
            foreach ($request->extras as $extraId => $data) {
                if (isset($data['quantity']) && $data['quantity'] > 0) {
                    $reservation->extras()->attach($extraId, [
                        'quantity' => $data['quantity'],
                        'price' => $data['price'] ?? 0,
                    ]);
                }
            }
        }

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'update_reservation',
            'description' => "Randevu güncellendi: ID {$reservation->id}",
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('admin.appointments.index')
            ->with('success', 'Randevu başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $query = Reservation::query();

        if ($user->role === 'randevucu') {
            $query->where('created_by', $user->id);
        }
        
        $reservation = $query->findOrFail($id);
        $id = $reservation->id;
        $reservation->delete();

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'delete_reservation',
            'description' => "Randevu silindi: ID {$id}",
            'ip_address' => request()->ip(),
        ]);

        return redirect()->route('admin.appointments.index')
            ->with('success', 'Randevu başarıyla silindi.');
    }

    public function daily(Request $request)
    {
        try {
            $date = $request->get('date', now()->format('Y-m-d'));
            $dateColumn = $this->getReservationDateColumn();
            $appointments = Reservation::whereDate($dateColumn, $date)
                ->with(['customer', 'salon'])
                ->orderBy('start_time')
                ->get();

            return view('admin.appointments.daily', compact('appointments', 'date'));
        } catch (\Exception $e) {
            \Log::error('Reservation Daily Error: ' . $e->getMessage());
            return view('admin.appointments.daily', [
                'appointments' => collect([]),
                'date' => $request->get('date', now()->format('Y-m-d'))
            ])->with('error', 'Randevular yüklenirken bir hata oluştu.');
        }
    }

    public function cancelled()
    {
        $appointments = Reservation::where('status', 'cancelled')
            ->with(['customer', 'salon'])
            ->orderBy('date', 'desc')
            ->paginate(20);

        return view('admin.appointments.cancelled', compact('appointments'));
    }

    public function open()
    {
        $appointments = Reservation::where('status', 'open')
            ->with(['customer', 'salon'])
            ->orderBy('date', 'asc')
            ->paginate(20);

        return view('admin.appointments.open', compact('appointments'));
    }

    private function generateInvoiceAndContract($reservation)
    {
        
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $reservation->salon->address ?? ($settings['invoice_address'] ?? '');
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';

        $invoiceData = $this->prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber);

        $invoicePdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        $invoicePdf->setOption('fontDir', storage_path('fonts'));
        $invoicePdf->setOption('defaultFont', 'dejavu sans');
        $invoicePath = 'invoices/reservation-' . $reservation->id . '-' . time() . '.pdf';
        Storage::disk('public')->put($invoicePath, $invoicePdf->output());

        $defaultContract = Contract::getDefaultForCompany(auth()->user()->company_id);
        $contractUrl = null;
        
        if ($defaultContract) {
            
            $contractUrl = $defaultContract->public_url;
        } else {
            
            $contractPdf = $this->generateContractPdf($reservation, $logo, $address);
            $contractPath = 'contracts/reservation-' . $reservation->id . '-' . time() . '.pdf';
            Storage::disk('public')->put($contractPath, $contractPdf->output());
            $contractUrl = url('/storage/' . $contractPath);
        }

        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if (!is_array($itemsData)) {
            $itemsData = [];
        }
        $itemsData['invoice_pdf'] = $invoicePath;
        $itemsData['invoice_url'] = url('/storage/' . $invoicePath);
        $itemsData['contract_url'] = $contractUrl;
        
        $reservation->items_data = json_encode($itemsData);
        $reservation->save();
        
        return [
            'invoice_url' => $itemsData['invoice_url'],
            'contract_url' => $contractUrl
        ];
    }

    private function generateContractPdf($reservation, $logo, $address)
    {
        // Salon bilgilerini al
        $salonName = 'Güzellik Salonu';
        if ($reservation->beauty_salon_id) {
            $beautySalon = \App\Models\User::find($reservation->beauty_salon_id);
            if ($beautySalon) {
                $salonName = $beautySalon->salon_name ?? $beautySalon->name ?? 'Güzellik Salonu';
            }
        } elseif ($reservation->salon) {
            $salonName = $reservation->salon->name ?? 'Güzellik Salonu';
        }
        
        $contractContent = "RANDEVU SÖZLEŞMESİ\n\n";
        $contractContent .= "Bu sözleşme, " . $reservation->customer->name . " ile " . $salonName . " arasında yapılmıştır.\n\n";
        $contractContent .= "Randevu Detayları:\n";
        $contractContent .= "Tarih: " . $reservation->date->format('d.m.Y') . "\n";
        $contractContent .= "Saat: " . $reservation->start_time . " - " . $reservation->end_time . "\n";
        $contractContent .= "Güzellik Salonu: " . $salonName . "\n";
        $contractContent .= "Toplam Tutar: " . number_format($reservation->total_price, 2, ',', '.') . " ₺\n";
        $contractContent .= "Alınan Kapora: " . number_format($reservation->deposit_paid ?? 0, 2, ',', '.') . " ₺\n\n";
        $contractContent .= "Bu sözleşme Salonay güzellik salonu randevu yönetim sistemi üzerinden oluşturulmuştur.\n\n";
        $contractContent .= "Taraflar bu sözleşmeyi kabul etmişlerdir.\n\n";
        $contractContent .= "Müşteri İmza: ___________________\n";
        $contractContent .= "Güzellik Salonu İmza: ___________________\n";
        $contractContent .= "Tarih: " . now()->format('d.m.Y');
        
        $data = [
            'reservation' => $reservation,
            'content' => $contractContent,
            'logo' => $logo,
            'address' => $address,
        ];
        
        $pdf = Pdf::loadView('admin.contracts.pdf', $data);
        $pdf->setOption('fontDir', storage_path('fonts'));
        $pdf->setOption('defaultFont', 'dejavu sans');
        return $pdf;
    }

    private function prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber, $invoice = null)
    {
        $items = [];

        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if (!is_array($itemsData) || empty($itemsData)) {
            $itemsData = json_decode($reservation->notes ?? '{}', true);
            if (!is_array($itemsData)) {
                $itemsData = [];
            }
        }
        
        $mealTotal = 0;
        $organizationTotal = 0;
        $extraTotal = 0;

        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $mealTotal += $total;
                }
            }
        }

        if (isset($itemsData['organizations']) && is_array($itemsData['organizations'])) {
            foreach ($itemsData['organizations'] as $org) {
                if (isset($org['quantity']) && isset($org['price'])) {
                    $total = $org['quantity'] * $org['price'];
                    $organizationTotal += $total;
                }
            }
        }

        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $extraTotal += $extra['price'];
                }
            }
        }

        $salonPrice = $reservation->total_price - ($mealTotal + $organizationTotal + $extraTotal);
        
        if ($salonPrice > 0) {
            $items[] = [
                'name' => 'Salon Ucreti',
                'quantity' => '-',
                'price' => $salonPrice,
                'total' => $salonPrice
            ];
        }

        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $items[] = [
                        'name' => $meal['name'],
                        'quantity' => $meal['quantity'] . ' Kisi',
                        'price' => $meal['price'],
                        'total' => $total
                    ];
                }
            }
        }

        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $items[] = [
                        'name' => $extra['name'],
                        'quantity' => '-',
                        'price' => $extra['price'],
                        'total' => $extra['price']
                    ];
                }
            }
        }

        $grandTotal = $reservation->total_price;
        $depositPaid = $reservation->deposit_paid ?? 0;

        $salonName = $reservation->salon->name ?? 'Güzellik Salonu';
        $salonAddress = $reservation->salon->address ?? $address;
        
        return [
            'reservation' => $reservation,
            'invoice' => $invoice,
            'items' => $items,
            'subtotal' => $grandTotal,
            'grandTotal' => $grandTotal,
            'depositPaid' => $depositPaid,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $salonName,
            'salonAddress' => $salonAddress,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
        ];
    }

    public function updateStatus(Request $request, $id)
    {
        try {
            $user = Auth::user();
            $query = Reservation::query();

            if ($user->role !== 'super_admin') {
                if ($user->role === 'güzellik_salonu') {
                    $query->where('beauty_salon_id', $user->id);
                } else {
                    $query->where('created_by', $user->id);
                }
            }
            
            $reservation = $query->findOrFail($id);
            
            $validated = $request->validate([
                'status' => 'required|in:open,confirmed,completed,cancelled',
                'mark_paid' => 'nullable|boolean'
            ]);
            
            $oldStatus = $reservation->status;
            $oldDepositPaid = $reservation->deposit_paid;

            $markPaid = $request->input('mark_paid', false);
            if ($request->isJson()) {
                $markPaid = $request->json('mark_paid', false);
            }
            $markPaid = filter_var($markPaid, FILTER_VALIDATE_BOOLEAN);

            if ($markPaid && ($validated['status'] === 'completed' || $validated['status'] === 'confirmed')) {
                
                $reservation->load('customer');

                if ($validated['status'] === 'completed') {
                    $amountToAdd = $reservation->total_price - ($reservation->deposit_paid ?? 0);
                } else {
                    
                    $amountToAdd = $reservation->total_price;
                }
                
                if ($amountToAdd > 0) {
                    try {
                        
                        $cashAccount = \App\Models\Account::getCashAccount($user->id);

                        $description = "Randevu #{$reservation->code}";
                        if ($validated['status'] === 'completed') {
                            $description .= " kalan ödeme alındı";
                        } else {
                            $description .= " onaylandı - ödeme alındı";
                        }
                        $description .= " - " . ($reservation->customer->name ?? 'Müşteri');
                        
                        \App\Models\Transaction::create([
                            'account_id' => $cashAccount->id,
                            'reservation_id' => $reservation->id,
                            'customer_id' => $reservation->customer_id,
                            'type' => 'income',
                            'amount' => $amountToAdd,
                            'description' => $description,
                            'date' => now()->format('Y-m-d'),
                            'created_by' => $user->id
                        ]);

                        $cashAccount->increment('balance', $amountToAdd);

                        if ($reservation->customer_id) {
                            $customer = \App\Models\Customer::find($reservation->customer_id);
                            if ($customer && $customer->balance !== null && $customer->balance > 0) {
                                $customer->decrement('balance', min($amountToAdd, $customer->balance));
                            }
                        }

                        $reservation->remaining_amount = 0;
                        $reservation->deposit_paid = 0;
                        
                        \Log::info('Ödeme kasaya eklendi - randevu güncellendi', [
                            'reservation_id' => $reservation->id,
                            'status' => $validated['status'],
                            'amount' => $amountToAdd,
                            'new_remaining' => 0,
                            'new_deposit_paid' => 0,
                        ]);
                        
                    } catch (\Exception $e) {
                        
                        \Log::error('Ödeme işlemi hatası: ' . $e->getMessage(), [
                            'reservation_id' => $reservation->id,
                            'trace' => $e->getTraceAsString()
                        ]);
                        
                        $reservation->remaining_amount = 0;
                        $reservation->deposit_paid = 0;
                    }
                }
            }
            
            $reservation->status = $validated['status'];
            $reservation->save();

            // Referans ödülü kontrolü - randevu onaylandığında veya tamamlandığında
            if (($validated['status'] === 'confirmed' || $validated['status'] === 'completed') && 
                ($oldStatus !== 'confirmed' && $oldStatus !== 'completed') && 
                $reservation->customer_id) {
                try {
                    // AppointmentRequest'ten referral_code kontrolü
                    $appointmentRequest = \App\Models\AppointmentRequest::where('reservation_id', $reservation->id)->first();
                    
                    if ($appointmentRequest && $appointmentRequest->referral_code) {
                        // Daha önce bu referans kodu için ödül verilmiş mi kontrol et
                        $existingReferral = \App\Models\Referral::where('referral_code', $appointmentRequest->referral_code)
                            ->where('referred_id', $reservation->customer_id)
                            ->where('beauty_salon_id', $reservation->beauty_salon_id)
                            ->first();
                        
                        if (!$existingReferral) {
                            // Referans koduna sahip müşteriyi bul
                            $referrerRating = \App\Models\SalonRating::where('referral_code', $appointmentRequest->referral_code)
                                ->where('salon_id', $reservation->beauty_salon_id)
                                ->first();
                            
                            if ($referrerRating && $referrerRating->customer_id) {
                                $referrerCustomer = \App\Models\Customer::find($referrerRating->customer_id);
                                
                                if ($referrerCustomer) {
                                    // Referans eden müşterinin bakiyesine 3 puan ekle
                                    $referrerCustomer->increment('balance', 3);
                                    
                                    // Referral kaydı oluştur
                                    \App\Models\Referral::create([
                                        'beauty_salon_id' => $reservation->beauty_salon_id,
                                        'referrer_id' => $referrerCustomer->id,
                                        'referred_id' => $reservation->customer_id,
                                        'referral_code' => $appointmentRequest->referral_code,
                                        'status' => $validated['status'] === 'completed' ? 'completed' : 'confirmed',
                                        'first_appointment_id' => $reservation->id,
                                        'first_visit_at' => now(),
                                        'rewarded_at' => now(),
                                    ]);
                                    
                                    \Log::info('Referans ödülü verildi', [
                                        'referrer_id' => $referrerCustomer->id,
                                        'referred_id' => $reservation->customer_id,
                                        'referral_code' => $appointmentRequest->referral_code,
                                        'reward_amount' => 3,
                                        'status' => $validated['status'],
                                    ]);
                                }
                            }
                        }
                    }
                } catch (\Exception $referralError) {
                    \Log::warning('Referans ödülü verilirken hata: ' . $referralError->getMessage());
                }
            }
            
            // Premium özellikler entegrasyonu - Randevu tamamlandığında
            if ($validated['status'] === 'completed' && $oldStatus !== 'completed') {
                try {
                    $premiumService = app(PremiumFeaturesIntegrationService::class);
                    $premiumService->onAppointmentCompleted($reservation);
                } catch (\Exception $e) {
                    \Log::error('Premium özellik entegrasyonu hatası (updateStatus): ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'trace' => $e->getTraceAsString(),
                    ]);
                }
            }

            // Yorum linki sadece 'completed' durumunda gönderilmeli
            if ($validated['status'] === 'completed' && $oldStatus !== 'completed' && $reservation->beauty_salon_id && $reservation->customer_id) {
                try {
                    $reservation->load(['customer']);
                    $salon = \App\Models\User::find($reservation->beauty_salon_id);
                    $customer = $reservation->customer;
                    
                    if ($salon && $customer && $salon->salon_slug) {
                        
                        $existingRating = \App\Models\SalonRating::where('reservation_id', $reservation->id)
                            ->where('salon_id', $reservation->beauty_salon_id)
                            ->first();
                        
                        if (!$existingRating) {
                            
                            $token = \App\Models\SalonRating::generateToken();
                            $referralCode = \App\Models\SalonRating::generateReferralCode();
                            $expiresAt = now()->addDays(30); 
                            
                            $rating = \App\Models\SalonRating::create([
                                'salon_id' => $reservation->beauty_salon_id,
                                'customer_id' => $reservation->customer_id,
                                'reservation_id' => $reservation->id,
                                'token' => $token,
                                'referral_code' => $referralCode,
                                'expires_at' => $expiresAt,
                                'is_used' => false,
                                'customer_name' => $customer->name,
                                'customer_phone' => $customer->phone,
                            ]);
                            
                            $ratingUrl = route('public.salon.rating', [
                                'slug' => $salon->salon_slug,
                                'token' => $token
                            ]);
                        } else {
                            $ratingUrl = route('public.salon.rating', [
                                'slug' => $salon->salon_slug,
                                'token' => $existingRating->token
                            ]);
                        }

                        try {
                            $salonName = $salon->salon_name ?? $salon->name ?? 'Salon';
                            
                            $whatsappMessage = "{$salonName}\n\n";
                            $whatsappMessage .= "Randevunuz başarıyla tamamlanmıştır bizi oylamak için aşağıdaki linke ulaşabilirsiniz.\n\n";
                            $whatsappMessage .= "{$ratingUrl}\n\n";
                            if (isset($referralCode)) {
                                $whatsappMessage .= "🎁 Referans Kodunuz: {$referralCode}\n";
                                $whatsappMessage .= "Bu kodu arkadaşlarınızla paylaşın, onlar randevu aldığında size 3 puan kazandırır!";
                            }
                            
                            $smsService = app(\App\Services\SmsService::class);
                            $whatsappResult = $smsService->send(
                                $customer->phone,
                                $whatsappMessage,
                                null,
                                [
                                    'purpose' => 'appointment',
                                    'user_id' => $reservation->beauty_salon_id
                                ]
                            );
                            
                            if ($whatsappResult && $whatsappResult->success) {
                                \Log::info('Yorum linki WhatsApp ile gönderildi (updateStatus)', [
                                    'reservation_id' => $reservation->id,
                                    'customer_phone' => $customer->phone,
                                    'rating_url' => $ratingUrl
                                ]);
                            } else {
                                \Log::warning('Yorum linki WhatsApp ile gönderilemedi (updateStatus)', [
                                    'reservation_id' => $reservation->id,
                                    'customer_phone' => $customer->phone,
                                    'error' => $whatsappResult->message ?? 'Bilinmeyen hata'
                                ]);
                            }
                        } catch (\Exception $whatsappError) {
                            \Log::warning('Yorum linki WhatsApp gönderilirken hata oluştu (updateStatus): ' . $whatsappError->getMessage(), [
                                'reservation_id' => $reservation->id,
                                'trace' => $whatsappError->getTraceAsString()
                            ]);
                        }
                    }
                } catch (\Exception $ratingError) {
                    \Log::warning('Yorum linki oluşturulamadı (updateStatus): ' . $ratingError->getMessage(), [
                        'reservation_id' => $reservation->id,
                        'trace' => $ratingError->getTraceAsString()
                    ]);
                }
            }

            $description = "Randevu #{$reservation->id} durumu '{$validated['status']}' olarak güncellendi";
            if ($markPaid) {
                $description .= " (Ödeme alındı)";
            }
            
            try {
                ActivityLog::create([
                    'user_id' => $user->id,
                    'action' => 'reservation_status_updated',
                    'model_type' => 'Reservation',
                    'model_id' => $reservation->id,
                    'description' => $description,
                    'properties' => json_encode([
                        'old_status' => $oldStatus,
                        'new_status' => $validated['status'],
                        'mark_paid' => $markPaid,
                        'old_deposit' => $oldDepositPaid,
                        'new_deposit' => $reservation->deposit_paid
                    ])
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }
            
            return response()->json([
                'success' => true,
                'message' => $markPaid ? 'Ödeme alındı ve randevu tamamlandı.' : 'Randevu durumu güncellendi.',
                'status' => $reservation->status
            ]);
        } catch (\Exception $e) {
            \Log::error('Randevu durum güncelleme hatası: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Randevu durumu güncellenirken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    public function createRecurring(Request $request)
    {
        try {
            $user = Auth::user();
            
            $validated = $request->validate([
                'reservation_id' => 'required|exists:reservations,id',
                'recurring_type' => 'required|in:monthly,weekly',
                'day_of_month' => 'nullable|integer|min:1|max:31',
                'day_of_week' => 'nullable|integer|min:1|max:7',
                'recurring_count' => 'required|integer|min:1|max:24',
                'start_date' => 'required|date',
            ]);

            $originalReservation = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                ->where('id', $validated['reservation_id'])
                ->where('beauty_salon_id', $user->id)
                ->firstOrFail();
            
            $createdReservations = [];
            $startDate = \Carbon\Carbon::parse($validated['start_date']);
            $currentDate = $startDate->copy();

            for ($i = 0; $i < $validated['recurring_count']; $i++) {
                if ($validated['recurring_type'] === 'monthly') {
                    
                    $targetDay = $validated['day_of_month'];

                    if ($i === 0) {
                        $currentDate->day($targetDay);
                        
                        if ($currentDate->lt($startDate)) {
                            $currentDate->addMonth();
                            $currentDate->day($targetDay);
                        }
                    } else {
                        $currentDate->addMonth();
                        $currentDate->day($targetDay);
                    }
                } else {
                    
                    $targetDayOfWeek = $validated['day_of_week'];
                    
                    if ($i === 0) {
                        
                        $currentDate->next($targetDayOfWeek === 7 ? \Carbon\Carbon::SUNDAY : $targetDayOfWeek);
                    } else {
                        $currentDate->addWeek();
                    }
                }

                $code = 'REZ-' . strtoupper(uniqid());

                $newReservation = Reservation::create([
                    'code' => $code,
                    'customer_id' => $originalReservation->customer_id,
                    'beauty_salon_id' => $user->id,
                    'date' => $currentDate->format('Y-m-d'),
                    'start_time' => $originalReservation->start_time,
                    'end_time' => $originalReservation->end_time,
                    'personel_id' => $originalReservation->personel_id,
                    'room_id' => $originalReservation->room_id,
                    'status' => 'confirmed',
                    'total_price' => $originalReservation->total_price,
                    'notes' => $originalReservation->notes . ' (Tekrarlanan Seans)',
                    'items_data' => $originalReservation->items_data,
                    'created_by' => $user->id,
                ]);

                if ($originalReservation->appointmentRequest) {
                    $newAppointmentRequest = \App\Models\AppointmentRequest::create([
                        'beauty_salon_id' => $user->id,
                        'customer_id' => $originalReservation->customer_id,
                        'service_id' => $originalReservation->appointmentRequest->service_id,
                        'name' => $originalReservation->appointmentRequest->name,
                        'surname' => $originalReservation->appointmentRequest->surname,
                        'phone' => $originalReservation->appointmentRequest->phone,
                        'age' => $originalReservation->appointmentRequest->age,
                        'appointment_date' => $currentDate->format('Y-m-d'),
                        'appointment_time' => $originalReservation->start_time,
                        'personel_id' => $originalReservation->personel_id,
                        'room_id' => $originalReservation->room_id,
                        'status' => 'approved',
                        'reservation_id' => $newReservation->id,
                        'notes' => $originalReservation->appointmentRequest->notes,
                    ]);
                }
                
                $createdReservations[] = $newReservation;
            }
            
            \Log::info('Tekrarlanan seanslar oluşturuldu', [
                'original_reservation_id' => $originalReservation->id,
                'recurring_type' => $validated['recurring_type'],
                'count' => $validated['recurring_count'],
                'created_count' => count($createdReservations),
            ]);
            
            return response()->json([
                'success' => true,
                'message' => count($createdReservations) . ' adet tekrarlanan seans oluşturuldu.',
                'reservations' => $createdReservations->map(function($r) {
                    return [
                        'id' => $r->id,
                        'date' => $r->date,
                        'start_time' => $r->start_time,
                    ];
                })
            ]);
        } catch (\Exception $e) {
            \Log::error('Tekrarlanan seans oluşturma hatası: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Tekrarlanan seanslar oluşturulurken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    private function getInvoicePublicUrl($token)
    {
        $url = url('/fatura/' . $token);
        
        // localhost için salonay/public yolunu kaldır
        if (strpos($url, 'localhost') !== false || strpos($url, '127.0.0.1') !== false) {
            $url = str_replace('/salonay/public', '', $url);
        }
        return $url;
    }

    protected function getReservationDateColumn(): string
    {
        if (\Schema::hasColumn('reservations', 'date')) {
            return 'date';
        }
        return 'appointment_date'; 
    }

    public function sendSms(Request $request, $id)
    {
        $reservation = Reservation::with(['customer', 'appointmentRequest.service', 'personel'])->findOrFail($id);
        
        if (!$reservation->customer || !$reservation->customer->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Müşterinin telefon numarası bulunamadı.'
            ], 400);
        }
        
        $user = Auth::user();
        
        try {
            $smsService = app(SmsService::class);

            $invoice = Invoice::where('reservation_id', $reservation->id)->first();
            $invoiceUrl = $invoice ? $invoice->public_url : '';
            
            $contract = Contract::where('user_id', $reservation->beauty_salon_id ?? $user->id)
                ->where('is_default', true)
                ->where('is_active', true)
                ->first();
            
            if (!$contract) {
                $contract = Contract::where('user_id', $reservation->beauty_salon_id ?? $user->id)
                    ->first();
            }
            
            $contractUrl = $contract ? $contract->public_url : '';

            $reservationDate = \Carbon\Carbon::parse($reservation->date ?? $reservation->appointment_date)->format('d/m/Y');
            $reservationTime = $reservation->start_time ?? $reservation->appointment_time ?? '';

            $customerName = $reservation->customer->name;
            $salonName = $user->salon_name ?? $user->name;
            $salonAddress = $user->salon_address ?? '';
            $serviceName = $reservation->appointmentRequest && $reservation->appointmentRequest->service ? $reservation->appointmentRequest->service->name : 'Randevu';
            $personelName = $reservation->personel ? $reservation->personel->name : '';

            $message = $salonName . "\n\n";
            if ($salonAddress) {
                $message .= "Adres: " . $salonAddress . "\n\n";
            }
            $message .= "Sayın {$customerName}, {$reservationDate} tarihinde {$reservationTime} saatinde {$serviceName} randevunuz onaylanmıştır.\n";
            if ($personelName) {
                $message .= "Personel: {$personelName}\n";
            }
            $message .= "Randevu saatinizden 5 dakika önce salonda hazır bulunmanızı rica ederiz.\n\n";
            
            if ($invoiceUrl) {
                $message .= "Fatura:\n";
                $message .= $invoiceUrl . "\n\n";
            }
            
            if ($contractUrl) {
                $message .= "Sözleşme:\n";
                $message .= $contractUrl . "\n\n";
            }
            
            $message .= "salonay.com iyi günler diler.";

            $result = $smsService->send(
                $reservation->customer->phone,
                $message,
                null,
                [
                    'purpose' => 'default',
                    'user_id' => $user->id
                ]
            );
            
            if ($result && $result->success) {
                return response()->json([
                    'success' => true,
                    'message' => 'SMS mesajı başarıyla gönderildi.'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result->message ?? 'SMS mesajı gönderilemedi. Lütfen tekrar deneyin.'
                ], 500);
            }
        } catch (\Exception $e) {
            \Log::error('❌ Randevu SMS gönderme hatası: ' . $e->getMessage(), [
                'reservation_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'SMS mesajı gönderilirken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }
}

