<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Room;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RoomController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $rooms = Room::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get();
        
        return view('admin.rooms.index', compact('rooms'));
    }

    public function create()
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        return view('admin.rooms.create');
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'capacity' => 'nullable|integer|min:1',
            'color' => 'nullable|string|max:7',
            'tags' => 'nullable|array',
        ]);
        
        $room = Room::create([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'capacity' => $validated['capacity'] ?? null,
            'color' => $validated['color'] ?? '#10b981',
            'tags' => $validated['tags'] ?? null,
            'beauty_salon_id' => $user->id,
        ]);

        if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
            return response()->json([
                'success' => true,
                'message' => 'Oda başarıyla eklendi.',
                'room' => [
                    'id' => $room->id,
                    'name' => $room->name,
                    'description' => $room->description ?? '',
                ]
            ], 200);
        }
        
        return redirect()->route('admin.beauty-salon.rooms.index')
            ->with('success', 'Oda başarıyla eklendi.');
    }

    public function show($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $room = Room::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
        
        return view('admin.rooms.show', compact('room'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $room = Room::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
        
        return view('admin.rooms.edit', compact('room'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $room = Room::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'capacity' => 'nullable|integer|min:1',
            'color' => 'nullable|string|max:7',
            'tags' => 'nullable|array',
        ]);
        
        $room->update($validated);
        
        return redirect()->route('admin.beauty-salon.rooms.index')
            ->with('success', 'Oda başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $room = Room::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
        
        $room->delete();
        
        return redirect()->route('admin.beauty-salon.rooms.index')
            ->with('success', 'Oda başarıyla silindi.');
    }
}

