<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Salon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SalonController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $query = Salon::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }

        $salon = $query->orderBy('name')->first();
        
        if (!$salon) {
            return redirect()->route('admin.dashboard')
                ->with('error', 'Henüz salon oluşturulmamış. İlk kayıt sırasında salon oluşturulmalıdır.');
        }
        
        return view('admin.settings.salons.index', compact('salon'));
    }

    public function create()
    {
        
        return redirect()->route('admin.settings.salons.index')
            ->with('error', 'Yeni salon oluşturma özelliği kaldırılmıştır. Sadece ilk kayıtta salon oluşturulabilir.');
    }

    public function store(Request $request)
    {
        $user = Auth::user();

        $existingSalon = Salon::where('created_by', $user->id)->first();

        if ($existingSalon && !$request->has('is_setup_wizard')) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Yeni salon oluşturma özelliği kaldırılmıştır. Sadece ilk kayıtta salon oluşturulabilir.'
                ], 403);
            }
            
            return redirect()->route('admin.settings.salons.index')
                ->with('error', 'Yeni salon oluşturma özelliği kaldırılmıştır. Sadece ilk kayıtta salon oluşturulabilir.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'capacity' => 'required|integer|min:1',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'district' => 'required|string|max:100',
            'image_path' => 'nullable|string',
            'is_active' => 'boolean',
        ], [
            'name.required' => 'Salon adı zorunludur.',
            'phone.required' => 'Telefon numarası zorunludur.',
            'capacity.required' => 'Salon kapasitesi zorunludur.',
            'capacity.min' => 'Kapasite en az 1 olmalıdır.',
            'address.required' => 'Adres zorunludur.',
            'city.required' => 'İl seçimi zorunludur.',
            'district.required' => 'İlçe seçimi zorunludur.',
        ]);

        $validated['description'] = ''; 

        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['created_by'] = Auth::id();

        Salon::create($validated);

        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Salon oluşturuldu.']);
        }

        return redirect()->route('admin.settings.salons.index')
            ->with('success', 'Salon oluşturuldu.');
    }

    public function edit($id)
    {
        $user = Auth::user();
        $query = Salon::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $salon = $query->findOrFail($id);
        return view('admin.settings.salons.edit', compact('salon'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = Salon::query();

        if ($user->role === 'randevucu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $salon = $query->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'capacity' => 'required|integer|min:1',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'district' => 'required|string|max:100',
            'is_active' => 'boolean',
        ], [
            'name.required' => 'Salon adı zorunludur.',
            'phone.required' => 'Telefon numarası zorunludur.',
            'capacity.required' => 'Salon kapasitesi zorunludur.',
            'capacity.min' => 'Kapasite en az 1 olmalıdır.',
            'address.required' => 'Adres zorunludur.',
            'city.required' => 'İl seçimi zorunludur.',
            'district.required' => 'İlçe seçimi zorunludur.',
        ]);
        
        $validated['is_active'] = $request->has('is_active') ? true : false;

        $salon->update($validated);

        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Salon güncellendi.']);
        }

        return redirect()->route('admin.settings.salons.index')
            ->with('success', 'Salon güncellendi.');
    }

    public function destroy(Request $request, $id)
    {
        
        if ($request->ajax()) {
            return response()->json([
                'success' => false, 
                'message' => 'Salon silme özelliği kaldırılmıştır.'
            ], 403);
        }
        
        return redirect()->route('admin.settings.salons.index')
            ->with('error', 'Salon silme özelliği kaldırılmıştır.');
    }
}

