<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use App\Models\Company;
use App\Models\User;
use App\Models\Salon;
use App\Models\Extra;
use Illuminate\Http\Request;

class SearchController extends Controller
{
    public function search(Request $request)
    {
        
        if (!auth()->check()) {
            return response()->json([
                'appointments' => [],
                'customers' => [],
                'companies' => [],
                'users' => []
            ], 401);
        }
        
        $type = $request->get('type');

        if ($type === 'salons') {
            $user = auth()->user();
            $query = Salon::where('is_active', true);

            if ($user && ($user->role === 'randevucu' || $user->role === 'admin')) {
                $query->where('created_by', $user->id);
            }
            
            $salons = $query->orderBy('name')
                ->get()
                ->map(function($salon) {
                    return [
                        'id' => $salon->id,
                        'title' => $salon->name,
                        'subtitle' => $salon->capacity ? "Kapasite: {$salon->capacity} kişi" : null
                    ];
                });
            
            return response()->json(['salons' => $salons]);
        }

        if ($type === 'extras') {
            $extras = Extra::where('is_active', true)
                ->orderBy('name')
                ->get()
                ->map(function($extra) {
                    return [
                        'id' => $extra->id,
                        'title' => $extra->name,
                        'price' => $extra->price ?? 0
                    ];
                });
            
            return response()->json(['extras' => $extras]);
        }
        
        $query = $request->get('q', '');
        
        if (strlen($query) < 2) {
            return response()->json([
                'customers' => []
            ]);
        }

        $results = [
            'customers' => []
        ];

        $currentUser = auth()->user();

        // Personel için sadece kendi randevularını göster
        if ($currentUser->isPersonel()) {
            $reservations = Reservation::with(['customer', 'appointmentRequest.service', 'room'])
                ->where('personel_id', $currentUser->id)
                ->where(function($q) use ($query) {
                    $q->whereHas('customer', function($customerQuery) use ($query) {
                        $customerQuery->where('name', 'like', "%{$query}%")
                            ->orWhere('phone', 'like', "%{$query}%")
                            ->orWhere('email', 'like', "%{$query}%");
                    })
                    ->orWhere('date', 'like', "%{$query}%")
                    ->orWhere('start_time', 'like', "%{$query}%");
                })
                ->orderBy('date', 'desc')
                ->orderBy('start_time', 'asc')
                ->limit(20)
                ->get();

            $results = ['customers' => []];
            
            foreach ($reservations as $reservation) {
                $customer = $reservation->customer;
                $statusText = match($reservation->status) {
                    'open' => 'Açık',
                    'confirmed' => 'Onaylı',
                    'cancelled' => 'İptal',
                    'completed' => 'Tamamlandı',
                    default => ucfirst($reservation->status ?? 'Bilinmiyor')
                };

                $serviceName = $reservation->appointmentRequest && $reservation->appointmentRequest->service 
                    ? $reservation->appointmentRequest->service->name 
                    : 'Hizmet';
                
                $title = ($customer ? $customer->name : 'Müşteri') . ' - ' . $serviceName;
                $subtitle = ($reservation->date ? \Carbon\Carbon::parse($reservation->date)->format('d.m.Y') : '') . 
                           ($reservation->start_time ? ' ' . $reservation->start_time : '');
                
                $results['customers'][] = [
                    'id' => $reservation->id,
                    'title' => $title,
                    'url' => '#', // Modal açılacak
                    'type' => $statusText,
                    'icon' => 'bi-calendar-event',
                    'subtitle' => $subtitle,
                    'is_customer' => false,
                    'reservation_id' => $reservation->id
                ];
            }

            return response()->json($results);
        }

        try {
            
            if ($currentUser->role === 'güzellik_salonu') {
                $customerQuery = Customer::with(['reservations' => function($q) use ($currentUser) {
                    $q->with(['appointmentRequest.service', 'personel', 'room'])
                      ->where('beauty_salon_id', $currentUser->id)
                      ->orderBy('date', 'desc')
                      ->limit(5);
                }])
                ->where('beauty_salon_id', $currentUser->id);
            } else {
                
                $customerQuery = Customer::with(['reservations' => function($q) use ($currentUser) {
                    $q->with('salon')->orderBy('date', 'desc')->limit(5);
                    if ($currentUser->role === 'randevucu' || $currentUser->role === 'admin') {
                        $q->where('created_by', $currentUser->id);
                    }
                }]);

                if ($currentUser->role === 'randevucu' || $currentUser->role === 'admin') {
                    $customerQuery->where('created_by', $currentUser->id);
                } elseif (!$currentUser->isSuperAdmin()) {
                    $companyId = $currentUser->companies->first()?->id;
                    if ($companyId) {
                        $companyUserIds = \App\Models\User::whereHas('companies', function($q) use ($companyId) {
                            $q->where('companies.id', $companyId);
                        })->pluck('id');
                        $customerQuery->whereIn('created_by', $companyUserIds);
                    } else {
                        $customerQuery->where('created_by', $currentUser->id);
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::error('SearchController Customer Query Error: ' . $e->getMessage());
            $customerQuery = Customer::query();
            if ($currentUser->role === 'güzellik_salonu') {
                $customerQuery->where('beauty_salon_id', $currentUser->id);
            } elseif ($currentUser->role === 'randevucu' || $currentUser->role === 'admin') {
                $customerQuery->where('created_by', $currentUser->id);
            }
        }
        
        $customers = $customerQuery
            ->where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('phone', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            })
            ->orderBy('name')
            ->limit(20)
            ->get();

        try {
            if ($currentUser->role === 'güzellik_salonu') {
                $reservationQuery = Reservation::with(['customer', 'appointmentRequest.service', 'personel', 'room'])
                    ->where('beauty_salon_id', $currentUser->id)
                    ->where(function($q) use ($query) {

                        $q->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.bride_name')) LIKE ?", ["%{$query}%"])
                          ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.groom_name')) LIKE ?", ["%{$query}%"])
                          ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.bride_groom_name')) LIKE ?", ["%{$query}%"]);
                    });
            } else {
                $reservationQuery = Reservation::with(['customer', 'salon'])
                    ->where(function($q) use ($query) {

                        $q->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.bride_name')) LIKE ?", ["%{$query}%"])
                          ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.groom_name')) LIKE ?", ["%{$query}%"])
                          ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(items_data, '$.bride_groom_name')) LIKE ?", ["%{$query}%"]);
                    });

                if ($currentUser->role === 'randevucu' || $currentUser->role === 'admin') {
                    $reservationQuery->where('created_by', $currentUser->id);
                } elseif (!$currentUser->isSuperAdmin()) {
                    $companyId = $currentUser->companies->first()?->id;
                    if ($companyId) {
                        $companyUserIds = \App\Models\User::whereHas('companies', function($q) use ($companyId) {
                            $q->where('companies.id', $companyId);
                        })->pluck('id');
                        $reservationQuery->whereIn('created_by', $companyUserIds);
                    } else {
                        $reservationQuery->where('created_by', $currentUser->id);
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::error('SearchController Reservation Query Error: ' . $e->getMessage());
            $reservationQuery = Reservation::query();
            if ($currentUser->role === 'güzellik_salonu') {
                $reservationQuery->where('beauty_salon_id', $currentUser->id);
            } elseif ($currentUser->role === 'randevucu' || $currentUser->role === 'admin') {
                $reservationQuery->where('created_by', $currentUser->id);
            }
            $reservationQuery->whereRaw('1 = 0'); 
        }
        
        $appointmentsByBrideGroom = $reservationQuery
            ->orderBy('date', 'desc')
            ->limit(10)
            ->get();

        foreach ($customers as $customer) {
            try {
                
                $results['customers'][] = [
                    'id' => $customer->id,
                    'title' => $customer->name,
                    'url' => route('admin.customers.show', $customer->id),
                    'type' => 'Müşteri',
                    'icon' => 'bi-person-fill',
                    'subtitle' => $customer->phone ?? '',
                    'is_customer' => true
                ];

                $reservations = $customer->reservations ?? collect([]);
                foreach ($reservations as $reservation) {
                    try {
                        $statusText = match($reservation->status) {
                            'open' => 'Açık',
                            'confirmed' => 'Onaylı',
                            'cancelled' => 'İptal',
                            'completed' => 'Tamamlandı',
                            default => ucfirst($reservation->status ?? 'Bilinmiyor')
                        };

                        if ($currentUser->role === 'güzellik_salonu') {
                            $serviceName = $reservation->appointmentRequest && $reservation->appointmentRequest->service 
                                ? $reservation->appointmentRequest->service->name 
                                : 'Hizmet';
                            $personelName = $reservation->personel ? $reservation->personel->name : '';
                            $title = $serviceName . ($personelName ? ' - ' . $personelName : '') . ' - ' . ($reservation->date ? $reservation->date->format('d.m.Y') : '');
                        } else {
                            $title = ($reservation->salon->name ?? 'Salon') . ' - ' . ($reservation->date ? $reservation->date->format('d.m.Y') : '');
                        }
                        
                        $results['customers'][] = [
                            'id' => $reservation->id,
                            'title' => $title,
                            'url' => route('admin.appointments.show', $reservation->id),
                            'type' => $statusText,
                            'icon' => 'bi-calendar-event',
                            'subtitle' => $customer->name,
                            'is_customer' => false,
                            'customer_id' => $customer->id
                        ];
                    } catch (\Exception $e) {
                        \Log::warning('SearchController: Reservation processing error: ' . $e->getMessage());
                        continue;
                    }
                }
            } catch (\Exception $e) {
                \Log::warning('SearchController: Customer processing error: ' . $e->getMessage());
                continue;
            }
        }

        try {
            foreach ($appointmentsByBrideGroom as $reservation) {
                try {
                    $itemsData = json_decode($reservation->items_data ?? '{}', true);
                    $brideName = $itemsData['bride_name'] ?? '';
                    $groomName = $itemsData['groom_name'] ?? '';
                    $brideGroomName = $itemsData['bride_groom_name'] ?? '';

                    $alreadyAdded = false;
                    foreach ($results['customers'] as $result) {
                        if (isset($result['id']) && $result['id'] == $reservation->id && !$result['is_customer']) {
                            $alreadyAdded = true;
                            break;
                        }
                    }
                    
                    if (!$alreadyAdded) {
                        $statusText = match($reservation->status) {
                            'open' => 'Açık',
                            'confirmed' => 'Onaylı',
                            'cancelled' => 'İptal',
                            'completed' => 'Tamamlandı',
                            default => ucfirst($reservation->status ?? 'Bilinmiyor')
                        };
                        
                        $subtitle = '';
                        if ($brideGroomName) {
                            $subtitle = $brideGroomName;
                        } elseif ($brideName && $groomName) {
                            $subtitle = $groomName . ' & ' . $brideName;
                        } elseif ($groomName) {
                            $subtitle = $groomName;
                        } elseif ($brideName) {
                            $subtitle = $brideName;
                        } else {
                            $subtitle = $reservation->customer->name ?? 'Müşteri';
                        }

                        if ($currentUser->role === 'güzellik_salonu') {
                            $serviceName = $reservation->appointmentRequest && $reservation->appointmentRequest->service 
                                ? $reservation->appointmentRequest->service->name 
                                : 'Hizmet';
                            $personelName = $reservation->personel ? $reservation->personel->name : '';
                            $title = $serviceName . ($personelName ? ' - ' . $personelName : '') . ' - ' . ($reservation->date ? $reservation->date->format('d.m.Y') : '');
                        } else {
                            $title = ($reservation->salon->name ?? 'Salon') . ' - ' . ($reservation->date ? $reservation->date->format('d.m.Y') : '');
                        }
                        
                        $results['customers'][] = [
                            'id' => $reservation->id,
                            'title' => $title,
                            'url' => route('admin.appointments.show', $reservation->id),
                            'type' => $statusText . ' (Gelin/Damat)',
                            'icon' => 'bi-heart-fill',
                            'subtitle' => $subtitle,
                            'is_customer' => false,
                            'customer_id' => $reservation->customer_id
                        ];
                    }
                } catch (\Exception $e) {
                    \Log::warning('SearchController: Bride/Groom reservation processing error: ' . $e->getMessage());
                    continue;
                }
            }
        } catch (\Exception $e) {
            \Log::error('SearchController: Bride/Groom search error: ' . $e->getMessage());
        }

        try {
            return response()->json($results);
        } catch (\Exception $e) {
            \Log::error('SearchController: JSON response error: ' . $e->getMessage());
            return response()->json([
                'customers' => [],
                'error' => 'Arama sırasında bir hata oluştu'
            ], 500);
        }
    }
}

