<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ServiceCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ServiceCategoryController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $categories = ServiceCategory::where('beauty_salon_id', $user->id)
            ->withCount('services')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();
        
        return view('admin.services.index', compact('categories'));
    }

    public function create()
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        return view('admin.services.categories.create');
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'color' => 'nullable|string|max:7',
        ]);

        $category = ServiceCategory::create([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'color' => $validated['color'] ?? '#ec4899',
            'beauty_salon_id' => $user->id,
        ]);

        return redirect()->route('admin.beauty-salon.service-categories.index')
            ->with('success', 'Kategori başarıyla oluşturuldu.');
    }

    public function edit($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $category = ServiceCategory::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
        
        return view('admin.services.categories.edit', compact('category'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $category = ServiceCategory::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'color' => 'nullable|string|max:7',
        ]);
        
        $category->update($validated);

        return redirect()->route('admin.beauty-salon.service-categories.index')
            ->with('success', 'Kategori başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $category = ServiceCategory::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();

        $category->services()->update(['service_category_id' => null]);
        
        $category->delete();

        return redirect()->route('admin.beauty-salon.service-categories.index')
            ->with('success', 'Kategori başarıyla silindi.');
    }
}
