<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Service;
use App\Models\ServiceCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ServiceController extends Controller
{
    public function index()
    {
        return app(\App\Http\Controllers\Admin\ServiceCategoryController::class)->index();
    }

    public function indexByCategory($categoryId)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $category = ServiceCategory::where('id', $categoryId)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();
        
        $services = Service::where('beauty_salon_id', $user->id)
            ->where('service_category_id', $categoryId)
            ->orderBy('name')
            ->get();
        
        $categories = ServiceCategory::where('beauty_salon_id', $user->id)
            ->withCount('services')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();
        
        return view('admin.services.index-by-category', compact('category', 'services', 'categories'));
    }

    public function create(Request $request, $categoryId = null)
    {
        $user = Auth::user();
        
        if ($user->role !== 'güzellik_salonu') {
            abort(403);
        }
        
        $category = null;
        if ($categoryId) {
            $category = ServiceCategory::where('id', $categoryId)
                ->where('beauty_salon_id', $user->id)
                ->firstOrFail();
        }
        
        $categories = ServiceCategory::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get();
        
        return view('admin.services.create', compact('category', 'categories'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'nullable|numeric|min:0',
            'duration' => 'nullable|integer|min:1',
            'service_category_id' => 'required|exists:service_categories,id',
        ]);

        $serviceData = $validated;
        $serviceData['beauty_salon_id'] = $user->id;

        $category = ServiceCategory::where('id', $validated['service_category_id'])
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();

        $service = Service::create($serviceData);

        if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
            return response()->json([
                'success' => true,
                'message' => 'Hizmet başarıyla oluşturuldu.',
                'service' => [
                    'id' => $service->id,
                    'name' => $service->name,
                    'price' => $service->price ?? 0,
                    'duration' => $service->duration ?? 60,
                    'description' => $service->description ?? '',
                    'service_category_id' => $service->service_category_id,
                ]
            ], 200);
        }

        return redirect()->route('admin.beauty-salon.services.index-by-category', $category->id)
            ->with('success', 'Hizmet başarıyla oluşturuldu.');
    }

    public function edit($id)
    {
        $user = Auth::user();
        
        $service = Service::findOrFail($id);

        if ($user->role === 'güzellik_salonu' && $service->beauty_salon_id !== $user->id) {
            abort(403);
        }
        
        $categories = ServiceCategory::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get();
        
        return view('admin.services.edit', compact('service', 'categories'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        $service = Service::findOrFail($id);

        if ($user->role === 'güzellik_salonu' && $service->beauty_salon_id !== $user->id) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'nullable|numeric|min:0',
            'duration' => 'nullable|integer|min:1',
            'service_category_id' => 'required|exists:service_categories,id',
        ]);
        
        $service->update($validated);

        $categoryId = $service->service_category_id;
        return redirect()->route('admin.beauty-salon.services.index-by-category', $categoryId)
            ->with('success', 'Hizmet başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        
        $service = Service::findOrFail($id);

        if ($user->role === 'güzellik_salonu' && $service->beauty_salon_id !== $user->id) {
            abort(403);
        }
        
        $categoryId = $service->service_category_id;
        $service->delete();

        if ($categoryId) {
            return redirect()->route('admin.beauty-salon.services.index-by-category', $categoryId)
                ->with('success', 'Hizmet başarıyla silindi.');
        }
        
        return redirect()->route('admin.beauty-salon.services.index')
            ->with('success', 'Hizmet başarıyla silindi.');
    }
}

