<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class SettingsController extends Controller
{
    public function membership()
    {
        $user = auth()->user();
        $settings = Setting::where('group', 'membership')->pluck('value', 'key');

        $licenseExpiresAt = $user->license_expires_at;
        $daysRemaining = null;
        $isUnlimited = false;
        
        if ($licenseExpiresAt) {
            $now = \Carbon\Carbon::now();
            $expires = \Carbon\Carbon::parse($licenseExpiresAt);
            $daysRemaining = floor($now->diffInDays($expires, false)); 
        } else {
            $isUnlimited = true;
        }
        
        return view('admin.settings.membership', compact('settings', 'licenseExpiresAt', 'daysRemaining', 'isUnlimited'));
    }

    public function updateMembership(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::updateOrCreate(
                ['key' => $key, 'group' => 'membership'],
                ['value' => $value]
            );
        }

        return redirect()->back()->with('success', 'Ayarlar kaydedildi.');
    }

    public function accounting()
    {
        $settings = Setting::where('group', 'accounting')->pluck('value', 'key');
        return view('admin.settings.accounting', compact('settings'));
    }

    public function updateAccounting(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::updateOrCreate(
                ['key' => $key, 'group' => 'accounting'],
                ['value' => $value]
            );
        }

        return redirect()->back()->with('success', 'Ayarlar kaydedildi.');
    }

    public function reservationForm()
    {
        $settings = Setting::where('group', 'reservation_form')->pluck('value', 'key');
        return view('admin.settings.reservation-form', compact('settings'));
    }

    public function updateReservationForm(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::updateOrCreate(
                ['key' => $key, 'group' => 'reservation_form'],
                ['value' => $value]
            );
        }

        return redirect()->back()->with('success', 'Ayarlar kaydedildi.');
    }

    public function calendar()
    {
        $settings = Setting::where('group', 'calendar')->pluck('value', 'key');
        return view('admin.settings.calendar', compact('settings'));
    }

    public function updateCalendar(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::updateOrCreate(
                ['key' => $key, 'group' => 'calendar'],
                ['value' => $value]
            );
        }

        return redirect()->back()->with('success', 'Ayarlar kaydedildi.');
    }

    public function sms()
    {
        
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        $settings = Setting::where('group', 'sms')->pluck('value', 'key');
        return view('admin.settings.sms', compact('settings'));
    }

    public function invoiceInfo()
    {
        $user = auth()->user();
        return view('admin.settings.invoice-info', compact('user'));
    }

    public function updateInvoiceInfo(Request $request)
    {
        $user = auth()->user();
        $data = $request->validate([
            'invoice_type' => 'nullable|in:bireysel,kurumsal',
            'invoice_name' => 'nullable|string|max:255',
            'invoice_tax_number' => 'nullable|string|max:20',
            'invoice_tax_office' => 'nullable|string|max:255',
            'invoice_address' => 'nullable|string',
            'invoice_city' => 'nullable|string|max:100',
            'invoice_district' => 'nullable|string|max:100',
            'invoice_postal_code' => 'nullable|string|max:10',
        ]);
        // Boş stringleri null yap
        $data = array_map(fn($v) => $v === '' ? null : $v, $data);
        $user->update($data);
        return redirect()->back()->with('success', 'Fatura bilgileri kaydedildi.');
    }

    public function personalInvoices()
    {
        $user = auth()->user();
        
        try {
            $query = \App\Models\Transaction::where('user_id', $user->id)
                ->whereIn('type', ['income'])
                ->where(function($q) {
                    $q->where('description', 'like', '%Paketi Satın Alma%')
                      ->orWhere('description', 'like', '%SMS Paketi Satın Alma%')
                      ->orWhere('description', 'like', '%Paket Satın Alma%');
                });
            
            $transactions = $query->orderBy('created_at', 'desc')->paginate(15);
        } catch (\Exception $e) {
            \Log::error('personalInvoices error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return back()->with('error', 'Faturalar yüklenirken bir hata oluştu. Lütfen tekrar deneyin.');
        }
        
        return view('admin.settings.personal-invoices', compact('transactions'));
    }

    public function updateSms(Request $request)
    {
        
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        foreach ($request->except('_token') as $key => $value) {
            Setting::updateOrCreate(
                ['key' => $key, 'group' => 'sms'],
                ['value' => $value]
            );
        }

        return redirect()->back()->with('success', 'Ayarlar kaydedildi.');
    }

    public function testSms(Request $request)
    {
        
        if (!auth()->user()->isSuperAdmin()) {
            return response()->json([
                'success' => false,
                'message' => 'Bu işlemi yapma yetkiniz yok.'
            ], 403);
        }
        
        try {
            $request->validate([
                'test_phone' => 'required|string|min:10',
                'test_message' => 'nullable|string|max:500',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz telefon numarası. Lütfen en az 10 haneli bir numara girin.',
                'errors' => $e->errors()
            ], 422);
        }
        
        try {
            $smsService = new \App\Services\SmsService();
            $phone = $request->test_phone;
            $message = $request->test_message ?? 'Bu bir test SMS mesajıdır. Salonay SMS API testi.';
            
            $result = $smsService->sendSms($phone, $message);
            
            if ($result) {
                return response()->json([
                    'success' => true,
                    'message' => 'Test SMS başarıyla gönderildi!'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'SMS gönderilemedi. Lütfen API ayarlarınızı ve log dosyasını kontrol edin.'
                ], 400);
            }
        } catch (\Exception $e) {
            \Log::error('Test SMS error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'SMS gönderilirken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    public function email()
    {
        $settings = Setting::where('group', 'email')->pluck('value', 'key');
        return view('admin.settings.email', compact('settings'));
    }

    public function updateEmail(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::updateOrCreate(
                ['key' => $key, 'group' => 'email'],
                ['value' => $value]
            );
        }

        return redirect()->back()->with('success', 'Ayarlar kaydedildi.');
    }

    public function whatsappInvoice()
    {
        $whatsappSettings = Setting::where('group', 'whatsapp')->pluck('value', 'key');
        $invoiceSettings = Setting::where('group', 'invoice')->pluck('value', 'key');
        return view('admin.settings.whatsapp-invoice', compact('whatsappSettings', 'invoiceSettings'));
    }

    public function updateWhatsappInvoice(Request $request)
    {
        try {
            
            $validated = $request->validate([
                'whatsapp_api_token' => 'nullable|string|max:255',
                'whatsapp_session_id' => 'nullable|string|max:255',
                'invoice_address' => 'nullable|string',
                'invoice_tax_office' => 'nullable|string|max:255',
                'invoice_tax_number' => 'nullable|string|max:255',
                'invoice_logo' => 'nullable|image|mimes:jpeg,jpg,png,gif,webp|max:2048', 
            ], [
                'invoice_logo.image' => 'Yüklenen dosya bir resim olmalıdır.',
                'invoice_logo.mimes' => 'Resim formatı jpeg, jpg, png, gif veya webp olmalıdır.',
                'invoice_logo.max' => 'Resim boyutu en fazla 2MB olabilir.',
            ]);

            if ($request->filled('whatsapp_api_token')) {
                Setting::updateOrCreate(
                    ['key' => 'whatsapp_api_token', 'group' => 'whatsapp'],
                    ['value' => $request->whatsapp_api_token]
                );
            }
            if ($request->filled('whatsapp_session_id')) {
                Setting::updateOrCreate(
                    ['key' => 'whatsapp_session_id', 'group' => 'whatsapp'],
                    ['value' => $request->whatsapp_session_id]
                );
            }

            if ($request->filled('invoice_address')) {
                Setting::updateOrCreate(
                    ['key' => 'invoice_address', 'group' => 'invoice'],
                    ['value' => $request->invoice_address]
                );
            }
            if ($request->filled('invoice_tax_office')) {
                Setting::updateOrCreate(
                    ['key' => 'invoice_tax_office', 'group' => 'invoice'],
                    ['value' => $request->invoice_tax_office]
                );
            }
            if ($request->filled('invoice_tax_number')) {
                Setting::updateOrCreate(
                    ['key' => 'invoice_tax_number', 'group' => 'invoice'],
                    ['value' => $request->invoice_tax_number]
                );
            }

            if ($request->hasFile('invoice_logo')) {
                try {
                    $logo = $request->file('invoice_logo');

                    $logosPath = storage_path('app/public/logos');
                    if (!file_exists($logosPath)) {
                        File::makeDirectory($logosPath, 0755, true);
                    }

                    $oldLogo = Setting::where('key', 'invoice_logo')->where('group', 'invoice')->first();
                    if ($oldLogo && $oldLogo->value) {
                        $oldLogoPath = storage_path('app/public/' . $oldLogo->value);
                        if (file_exists($oldLogoPath)) {
                            @unlink($oldLogoPath);
                        }
                    }

                    $logoPath = $logo->store('logos', 'public');
                    
                    if ($logoPath) {
                        Setting::updateOrCreate(
                            ['key' => 'invoice_logo', 'group' => 'invoice'],
                            ['value' => $logoPath]
                        );
                    } else {
                        return redirect()->back()->with('error', 'Logo kaydedilemedi. Lütfen tekrar deneyin.');
                    }
                } catch (\Exception $e) {
                    \Log::error('Logo yükleme hatası: ' . $e->getMessage(), [
                        'file' => $e->getFile(),
                        'line' => $e->getLine(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    return redirect()->back()->with('error', 'Logo yüklenirken bir hata oluştu: ' . $e->getMessage());
                }
            }

            return redirect()->back()->with('success', 'Ayarlar başarıyla kaydedildi.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('WhatsApp/Fatura ayarları güncelleme hatası: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Ayarlar kaydedilirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function homepage()
    {
        
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        $settings = Setting::where('group', 'homepage')->pluck('value', 'key');
        return view('admin.settings.homepage', compact('settings'));
    }

    public function updateHomepage(Request $request)
    {
        
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        foreach ($request->except('_token') as $key => $value) {
            Setting::updateOrCreate(
                ['key' => $key, 'group' => 'homepage'],
                ['value' => $value]
            );
        }

        return redirect()->back()->with('success', 'Ana sayfa ayarları kaydedildi.');
    }

    public function general()
    {
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        $generalSettings = Setting::where('group', 'general')->pluck('value', 'key');
        $paymentSettings = Setting::where('group', 'payment')->pluck('value', 'key');
        $settings = $generalSettings->merge($paymentSettings);
        return view('admin.settings.general', compact('settings'));
    }

    public function updateGeneral(Request $request)
    {
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        foreach ($request->except(['_token', 'android_download_url', 'ios_download_url']) as $key => $value) {
            if (strpos($key, 'paytr_') === 0 || strpos($key, 'iyzico_') === 0 || $key === 'payment_provider') {
                Setting::updateOrCreate(
                    ['key' => $key, 'group' => 'payment'],
                    ['value' => $value]
                );
            } else {
                Setting::updateOrCreate(
                    ['key' => $key, 'group' => 'general'],
                    ['value' => $value]
                );
            }
        }
        
        if ($request->has('android_download_url')) {
            Setting::updateOrCreate(
                ['key' => 'android_download_url', 'group' => 'apps'],
                ['value' => $request->android_download_url]
            );
        }
        
        if ($request->has('ios_download_url')) {
            Setting::updateOrCreate(
                ['key' => 'ios_download_url', 'group' => 'apps'],
                ['value' => $request->ios_download_url]
            );
        }

        return redirect()->back()->with('success', 'Ayarlar kaydedildi.');
    }

}

