<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SmartDiscount;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SmartDiscountController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $discounts = SmartDiscount::where('beauty_salon_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $services = Service::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('admin.smart-discount.index', compact('discounts', 'services'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        // İndirim tipini belirle
        $discountType = $request->input('discount_type') ?? $request->input('campaign_discount_type');
        
        $rules = [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'campaign_type' => 'required|in:discount,campaign',
            'days_of_week' => 'required|array|min:1',
            'days_of_week.*' => 'in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'service_ids' => 'nullable|array',
            'service_ids.*' => 'exists:services,id',
            'target_segments' => 'nullable|array',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i|after:start_time',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'is_active' => 'boolean',
            'status' => 'required|in:draft,active,paused,completed,cancelled',
            'auto_apply' => 'boolean',
            'budget' => 'nullable|numeric|min:0',
            'max_redemptions' => 'nullable|integer|min:1',
            'auto_stop_on_loss' => 'boolean',
            'min_roi' => 'nullable|numeric|min:0|max:100',
        ];

        // İndirim tipine göre validation
        if ($discountType === 'percentage') {
            $rules['discount_percentage'] = 'required|numeric|min:1|max:100';
            $rules['discount_amount'] = 'nullable|numeric|min:0';
        } elseif ($discountType === 'amount') {
            $rules['discount_percentage'] = 'nullable|numeric|min:0|max:100';
            $rules['discount_amount'] = 'required|numeric|min:0';
        } else {
            // Varsayılan olarak yüzde veya tutar biri olmalı
            $rules['discount_percentage'] = 'nullable|numeric|min:0|max:100';
            $rules['discount_amount'] = 'nullable|numeric|min:0';
        }

        $validated = $request->validate($rules);

        $validated['beauty_salon_id'] = $user->id;
        $validated['is_active'] = $request->has('is_active');
        $validated['auto_apply'] = $request->has('auto_apply');
        $validated['auto_stop_on_loss'] = $request->has('auto_stop_on_loss');
        $validated['service_ids'] = $validated['service_ids'] ?? null;
        $validated['target_segments'] = $validated['target_segments'] ?? null;
        $validated['budget_used'] = 0;
        $validated['redemptions_count'] = 0;

        SmartDiscount::create($validated);

        $message = $validated['campaign_type'] === 'campaign' 
            ? 'Kampanya başarıyla oluşturuldu.' 
            : 'Akıllı indirim başarıyla oluşturuldu.';

        return redirect()->route('admin.beauty-salon.smart-discount.index')
            ->with('success', $message);
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $discount = SmartDiscount::where('beauty_salon_id', $user->id)
            ->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'campaign_type' => 'required|in:discount,campaign',
            'days_of_week' => 'required|array|min:1',
            'days_of_week.*' => 'in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'discount_amount' => 'nullable|numeric|min:0',
            'service_ids' => 'nullable|array',
            'service_ids.*' => 'exists:services,id',
            'target_segments' => 'nullable|array',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i|after:start_time',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'is_active' => 'boolean',
            'status' => 'required|in:draft,active,paused,completed,cancelled',
            'auto_apply' => 'boolean',
            'budget' => 'nullable|numeric|min:0',
            'max_redemptions' => 'nullable|integer|min:1',
            'auto_stop_on_loss' => 'boolean',
            'min_roi' => 'nullable|numeric|min:0|max:100',
        ]);

        $validated['is_active'] = $request->has('is_active');
        $validated['auto_apply'] = $request->has('auto_apply');
        $validated['auto_stop_on_loss'] = $request->has('auto_stop_on_loss');
        $validated['service_ids'] = $validated['service_ids'] ?? null;
        $validated['target_segments'] = $validated['target_segments'] ?? null;

        $discount->update($validated);

        return redirect()->route('admin.beauty-salon.smart-discount.index')
            ->with('success', 'Akıllı indirim başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $discount = SmartDiscount::where('beauty_salon_id', $user->id)
            ->findOrFail($id);

        $discount->delete();

        return redirect()->route('admin.beauty-salon.smart-discount.index')
            ->with('success', 'Akıllı indirim başarıyla silindi.');
    }
}
