<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PricingRule;
use App\Models\DynamicPrice;
use App\Models\PriceTest;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class SmartPricingController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $pricingRules = PricingRule::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('priority', 'desc')
            ->get();

        $services = Service::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('admin.smart-pricing.index', compact('pricingRules', 'services'));
    }

    public function rules()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $rules = PricingRule::where('beauty_salon_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $services = Service::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('admin.smart-pricing.rules', compact('rules', 'services'));
    }

    public function tests()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $tests = PriceTest::where('beauty_salon_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $services = Service::where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('admin.smart-pricing.tests', compact('tests', 'services'));
    }

    public function storeRule(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'rule_type' => 'required|in:time_based,demand_based,staff_based,seasonal,day_of_week',
            'pricing_type' => 'required|in:percentage,fixed,multiplier',
            'price_adjustment' => 'required|numeric',
            'multiplier' => 'nullable|numeric|min:0',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i|after:start_time',
            'days_of_week' => 'nullable|array',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'demand_threshold' => 'nullable|integer|min:0',
            'staff_id' => 'nullable|exists:users,id',
            'service_id' => 'nullable|exists:services,id',
            'max_bookings' => 'nullable|integer|min:0',
            'priority' => 'required|integer|min:0|max:100',
            'is_active' => 'boolean',
        ]);

        $validated['beauty_salon_id'] = $user->id;
        $validated['is_active'] = $request->has('is_active');
        
        // Eğer pricing_type percentage ise, price_adjustment'ı yüzde olarak kaydet
        if ($validated['pricing_type'] === 'percentage') {
            // Zaten yüzde olarak geliyor, değişiklik yok
        } elseif ($validated['pricing_type'] === 'multiplier' && !isset($validated['multiplier'])) {
            // Eğer multiplier seçildiyse ama değer verilmediyse, percentage'den hesapla
            if (isset($validated['price_adjustment'])) {
                $validated['multiplier'] = 1 + ($validated['price_adjustment'] / 100);
            }
        }

        PricingRule::create($validated);

        return redirect()->route('admin.beauty-salon.smart-pricing.rules')
            ->with('success', 'Fiyatlandırma kuralı başarıyla oluşturuldu.');
    }

    public function storeTest(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'service_id' => 'required|exists:services,id',
            'test_name' => 'required|string|max:255',
            'control_price' => 'required|numeric|min:0',
            'test_price' => 'required|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
        ]);

        $validated['beauty_salon_id'] = $user->id;
        $validated['status'] = 'running';
        $validated['control_group_size'] = 0;
        $validated['test_group_size'] = 0;
        $validated['control_conversions'] = 0;
        $validated['test_conversions'] = 0;
        $validated['control_revenue'] = 0;
        $validated['test_revenue'] = 0;
        $validated['conversion_rate_control'] = 0;
        $validated['conversion_rate_test'] = 0;

        PriceTest::create($validated);

        return redirect()->route('admin.beauty-salon.smart-pricing.tests')
            ->with('success', 'Fiyat testi başarıyla oluşturuldu.');
    }
}
