<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\StaffPerformanceMetric;
use App\Models\StaffCommission;
use App\Models\StaffBonus;
use App\Models\Reservation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class StaffPerformanceController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $staff = User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->with(['services'])
            ->orderBy('name')
            ->get();

        // Son 30 günlük performans metrikleri
        $currentMonth = Carbon::now()->format('Y-m');
        $performanceMetrics = StaffPerformanceMetric::where('beauty_salon_id', $user->id)
            ->where('period_date', 'like', $currentMonth . '%')
            ->with('staff')
            ->get()
            ->keyBy('staff_id');

        return view('admin.staff-performance.index', compact('staff', 'performanceMetrics'));
    }

    public function commissions()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $commissions = StaffCommission::where('beauty_salon_id', $user->id)
            ->with('staff')
            ->orderBy('created_at', 'desc')
            ->get();

        $staff = User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->orderBy('name')
            ->get();

        return view('admin.staff-performance.commissions', compact('commissions', 'staff'));
    }

    public function bonuses()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $bonuses = StaffBonus::where('beauty_salon_id', $user->id)
            ->with('staff')
            ->orderBy('created_at', 'desc')
            ->get();

        $staff = User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->orderBy('name')
            ->get();

        return view('admin.staff-performance.bonuses', compact('bonuses', 'staff'));
    }

    public function storeCommission(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'staff_id' => 'required|exists:users,id',
            'commission_type' => 'required|in:percentage,fixed,tiered',
            'commission_percentage' => 'nullable|numeric|min:0|max:100',
            'commission_amount' => 'nullable|numeric|min:0',
            'min_threshold' => 'nullable|numeric|min:0',
            'max_threshold' => 'nullable|numeric|min:0',
        ]);

        $validated['beauty_salon_id'] = $user->id;
        $validated['is_active'] = true;

        StaffCommission::create($validated);

        return redirect()->route('admin.beauty-salon.staff-performance.commissions')
            ->with('success', 'Komisyon ayarı başarıyla oluşturuldu.');
    }

    public function storeBonus(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'staff_id' => 'required|exists:users,id',
            'bonus_name' => 'required|string|max:255',
            'bonus_type' => 'required|in:target_based,performance_based,special',
            'target_amount' => 'nullable|numeric|min:0',
            'bonus_amount' => 'required|numeric|min:0',
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
        ]);

        $validated['beauty_salon_id'] = $user->id;
        $validated['status'] = 'active';

        StaffBonus::create($validated);

        return redirect()->route('admin.beauty-salon.staff-performance.bonuses')
            ->with('success', 'Bonus başarıyla oluşturuldu.');
    }
}
