<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StaffSalary;
use App\Models\StaffServiceTracking;
use App\Models\StaffCommission;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class StaffSalaryController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $salaries = StaffSalary::where('beauty_salon_id', $user->id)
            ->with('staff')
            ->orderBy('period_start', 'desc')
            ->paginate(20);

        return view('admin.staff-salary.index', compact('salaries'));
    }

    public function calculate(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $validated = $request->validate([
            'staff_id' => 'required|exists:users,id',
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
        ]);

        $staff = User::where('id', $validated['staff_id'])
            ->where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->firstOrFail();

        // Personel komisyon ayarlarını al
        $commission = StaffCommission::where('staff_id', $staff->id)
            ->where('beauty_salon_id', $user->id)
            ->where('is_active', true)
            ->first();

        // Dönem içindeki hizmetleri al
        $serviceTrackings = StaffServiceTracking::where('staff_id', $staff->id)
            ->where('beauty_salon_id', $user->id)
            ->whereBetween('service_date', [$validated['period_start'], $validated['period_end']])
            ->where('status', 'completed')
            ->get();

        $totalRevenue = $serviceTrackings->sum('service_price');
        $totalCommission = $serviceTrackings->sum('commission_amount');
        $totalAppointments = $serviceTrackings->count();

        // Sabit maaş (varsayılan 0, ayarlanabilir)
        $baseSalary = 0; // Bu değer staff_salaries tablosundan veya ayarlardan alınabilir

        // Bonus hesaplama (performans bazlı)
        $bonusAmount = 0; // Bu değer performans metriklerine göre hesaplanabilir

        $totalEarnings = $baseSalary + $totalCommission + $bonusAmount;

        // Maaş kaydı oluştur
        $salary = StaffSalary::create([
            'staff_id' => $staff->id,
            'beauty_salon_id' => $user->id,
            'base_salary' => $baseSalary,
            'commission_rate' => $commission ? $commission->commission_percentage : 0,
            'bonus_amount' => $bonusAmount,
            'period_start' => $validated['period_start'],
            'period_end' => $validated['period_end'],
            'total_earnings' => $totalEarnings,
            'total_commission' => $totalCommission,
            'total_bonus' => $bonusAmount,
            'total_appointments' => $totalAppointments,
            'total_revenue' => $totalRevenue,
            'status' => 'calculated',
        ]);

        return redirect()->route('admin.beauty-salon.staff-salary.index')
            ->with('success', 'Maaş başarıyla hesaplandı.');
    }

    public function markAsPaid($id)
    {
        $user = Auth::user();
        
        if (!$user->isBeautySalon()) {
            abort(403);
        }

        $salary = StaffSalary::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->firstOrFail();

        $salary->update([
            'status' => 'paid',
            'paid_at' => now(),
        ]);

        return redirect()->back()
            ->with('success', 'Maaş ödendi olarak işaretlendi.');
    }
}

