<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Laravel\Sanctum\PersonalAccessToken;

class AuthController extends Controller
{
    public function login(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required_without:phone|email',
            'phone' => 'required_without:email|string',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = null;
        if ($request->has('email')) {
            $user = User::where('email', $request->email)->first();
        } elseif ($request->has('phone')) {
            $user = User::where('phone', $request->phone)->first();
        }

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'E-posta veya şifre hatalı',
                'code' => 'INVALID_CREDENTIALS',
            ], 401);
        }

        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Hesabınız askıya alınmış',
                'code' => 'ACCOUNT_SUSPENDED',
            ], 403);
        }

        if ($user->license_expires_at && $user->license_expires_at->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'Lisans süreniz dolmuş',
                'code' => 'LICENSE_EXPIRED',
            ], 403);
        }

        $user->tokens()->delete();

        $token = $user->createToken('mobile-app', ['*'], now()->addDays(30))->plainTextToken;

        return response()->json([
            'success' => true,
            'data' => [
                'access_token' => $token,
                'token_type' => 'Bearer',
                'expires_in' => 30 * 24 * 60 * 60, 
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'role' => $user->role,
                    'salon_id' => $user->beauty_salon_id,
                    'salon_name' => $user->salon_name,
                ],
            ],
        ]);
    }

    public function logout(Request $request): JsonResponse
    {
        
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Başarıyla çıkış yapıldı',
        ]);
    }

    public function me(Request $request): JsonResponse
    {
        $user = $request->user();

        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Hesabınız askıya alınmış',
                'code' => 'ACCOUNT_SUSPENDED',
            ], 403);
        }

        if ($user->license_expires_at && $user->license_expires_at->isPast()) {
            return response()->json([
                'success' => false,
                'message' => 'Lisans süreniz dolmuş',
                'code' => 'LICENSE_EXPIRED',
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'role' => $user->role,
                'salon_id' => $user->beauty_salon_id,
                'salon_name' => $user->salon_name,
                'avatar' => $user->salon_profile_image ? asset('storage/' . $user->salon_profile_image) : null,
            ],
        ]);
    }

    public function refreshToken(Request $request): JsonResponse
    {
        $user = $request->user();

        $request->user()->currentAccessToken()->delete();

        $token = $user->createToken('mobile-app', ['*'], now()->addDays(30))->plainTextToken;

        return response()->json([
            'success' => true,
            'data' => [
                'access_token' => $token,
                'token_type' => 'Bearer',
                'expires_in' => 30 * 24 * 60 * 60,
            ],
        ]);
    }

    public function sendOTP(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|regex:/^[0-9+\-\s()]+$/',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz telefon numarası',
                'errors' => $validator->errors(),
            ], 422);
        }

        $phone = preg_replace('/[^0-9+]/', '', $request->phone);

        $otp = str_pad((string) rand(0, 999999), 6, '0', STR_PAD_LEFT);

        cache()->put("otp_{$phone}", $otp, now()->addMinutes(10));

        try {

            \Log::info("OTP for {$phone}: {$otp}");
        } catch (\Exception $e) {
            \Log::error("Failed to send OTP: " . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'OTP kodu gönderildi',
            
            'otp' => app()->environment('local') ? $otp : null,
        ]);
    }

    public function verifyOTP(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'otp' => 'required|string|size:6',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $phone = preg_replace('/[^0-9+]/', '', $request->phone);
        $storedOtp = cache()->get("otp_{$phone}");

        if (!$storedOtp || $storedOtp !== $request->otp) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz OTP kodu',
                'code' => 'INVALID_OTP',
            ], 401);
        }

        $user = User::where('phone', $phone)->first();

        if (!$user) {
            
            $user = User::create([
                'name' => 'Müşteri',
                'phone' => $phone,
                'role' => 'customer', 
                'password' => Hash::make(uniqid()), 
                'is_active' => true,
            ]);
        }

        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Hesabınız askıya alınmış',
                'code' => 'ACCOUNT_SUSPENDED',
            ], 403);
        }

        $user->tokens()->delete();

        $token = $user->createToken('mobile-app', ['*'], now()->addDays(30))->plainTextToken;

        cache()->forget("otp_{$phone}");

        return response()->json([
            'success' => true,
            'data' => [
                'access_token' => $token,
                'token_type' => 'Bearer',
                'expires_in' => 30 * 24 * 60 * 60,
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'role' => $user->role,
                    'salon_id' => $user->beauty_salon_id,
                    'salon_name' => $user->salon_name,
                ],
            ],
        ]);
    }
}

