<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Services\Sms\OtpService;
use App\Services\SmsService;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    public function login(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required_without:phone|email',
            'phone' => 'required_without:email|string',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = null;
        if ($request->has('email')) {
            $user = User::where('email', $request->email)->first();
        } elseif ($request->has('phone')) {
            $phone = preg_replace('/[^0-9]/', '', $request->phone);
            $user = User::where('phone', $phone)->first();
        }

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'E-posta veya şifre hatalı',
                'code' => 'INVALID_CREDENTIALS',
            ], 401);
        }

        // Sadece admin ve personel giriş yapabilir
        if (!in_array($user->role, ['güzellik_salonu', 'personel', 'admin', 'super_admin'])) {
            return response()->json([
                'success' => false,
                'message' => 'Bu hesap türü mobil uygulamaya giriş yapamaz',
                'code' => 'INVALID_ROLE',
            ], 403);
        }

        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Hesabınız askıya alınmış',
                'code' => 'ACCOUNT_SUSPENDED',
            ], 403);
        }

        // Lisans/trial süresi dolmuş olsa bile girişe izin ver; uygulama içinde paket seçimine yönlendirilebilir
        // if ($user->license_expires_at && $user->license_expires_at->isPast()) { ... }

        $user->tokens()->delete();
        $token = $user->createToken('ios-app', ['*'], now()->addDays(180))->plainTextToken;

        return response()->json([
            'success' => true,
            'data' => [
                'access_token' => $token,
                'token_type' => 'Bearer',
                'expires_in' => 180 * 24 * 60 * 60,
                'user' => $this->mapUserPayload($user),
            ],
        ]);
    }

    public function logout(Request $request): JsonResponse
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Başarıyla çıkış yapıldı',
        ]);
    }

    public function me(Request $request): JsonResponse
    {
        $user = $request->user();

        if (!$user->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Hesabınız askıya alınmış',
                'code' => 'ACCOUNT_SUSPENDED',
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => $this->mapUserPayload($user),
        ]);
    }

    public function updateProfile(Request $request): JsonResponse
    {
        $user = $request->user();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|max:255|unique:users,email,' . $user->id,
            'phone' => 'sometimes|string|max:20|unique:users,phone,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
            'salon_name' => 'sometimes|string|max:255',
            'salon_short_name' => 'sometimes|string|max:50',
            'short_url' => 'sometimes|nullable|string|max:50|unique:users,short_url,' . $user->id,
            'salon_phone' => 'sometimes|string|min:10|max:20',
            'salon_city' => 'sometimes|string|max:255',
            'salon_district' => 'sometimes|string|max:255',
            'salon_address' => 'sometimes|string',
            'salon_description' => 'sometimes|nullable|string',
            'salon_color' => 'sometimes|in:pink,blue,purple,orange,green',
            'salon_working_hours' => 'sometimes|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $payload = $request->only(['name', 'email', 'phone']);
        if ($request->filled('password')) {
            $payload['password'] = Hash::make($request->password);
        }

        $hasSalonSettingsInRequest = $request->hasAny([
            'salon_name',
            'salon_short_name',
            'short_url',
            'salon_phone',
            'salon_city',
            'salon_district',
            'salon_address',
            'salon_description',
            'salon_color',
            'salon_working_hours',
        ]);

        if ($hasSalonSettingsInRequest) {
            if (!in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'], true)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Salon ayarlarını güncelleme yetkiniz yok.',
                ], 403);
            }

            if ($request->has('salon_phone')) {
                $payload['salon_phone'] = $this->normalizePhone((string) $request->input('salon_phone'));
            }
            if ($request->has('salon_name')) {
                $payload['salon_name'] = $request->input('salon_name');
            }
            if ($request->has('salon_city')) {
                $payload['salon_city'] = $request->input('salon_city');
            }
            if ($request->has('salon_district')) {
                $payload['salon_district'] = $request->input('salon_district');
            }
            if ($request->has('salon_address')) {
                $payload['salon_address'] = $request->input('salon_address');
            }
            if ($request->has('salon_description')) {
                $payload['salon_description'] = $request->input('salon_description');
            }
            if ($request->has('salon_color')) {
                $payload['salon_color'] = $request->input('salon_color');
            }

            if ($request->has('salon_short_name')) {
                $salonShortName = trim((string) $request->input('salon_short_name'));
                $payload['salon_short_name'] = $salonShortName;
                $payload['salon_slug'] = $this->generateUniqueSalonSlug($salonShortName, $user->id);
            }

            if ($request->has('short_url')) {
                $short = trim((string) $request->input('short_url', ''));
                if ($short === '') {
                    $payload['short_url'] = null;
                } else {
                    $shortSlug = Str::slug($short);
                    if ($shortSlug === '') {
                        return response()->json([
                            'success' => false,
                            'message' => 'Kısa URL geçersiz.',
                        ], 422);
                    }
                    $payload['short_url'] = $this->generateUniqueShortUrl($shortSlug, $user->id);
                }
            }

            if ($request->has('salon_working_hours')) {
                $workingHoursValidation = $this->normalizeSalonWorkingHours(
                    (array) $request->input('salon_working_hours', [])
                );
                if (!$workingHoursValidation['ok']) {
                    return response()->json([
                        'success' => false,
                        'message' => $workingHoursValidation['message'],
                        'errors' => [
                            'salon_working_hours' => [$workingHoursValidation['message']],
                        ],
                    ], 422);
                }
                $payload['salon_working_hours'] = $workingHoursValidation['data'];
            }
        }

        $user->update($payload);
        $user->refresh();

        if ($hasSalonSettingsInRequest && in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'], true)) {
            $this->syncSalonModel($user);
        }

        return response()->json([
            'success' => true,
            'message' => 'Profil başarıyla güncellendi',
            'data' => $this->mapUserPayload($user),
        ]);
    }

    public function sendDeleteOtp(Request $request, SmsService $smsService): JsonResponse
    {
        $user = $request->user();

        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil. Hesap silme için önce telefon numaranızı güncelleyin.',
            ], 400);
        }

        $result = $smsService->sendOtp($user->phone, 'account_deletion');
        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Doğrulama kodu telefon numaranıza gönderildi.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
        ], 400);
    }

    public function verifyDeleteOtp(Request $request, OtpService $otpService): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'otp_code' => 'required|string|size:6',
        ], [
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();
        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil.',
            ], 400);
        }

        $otpCode = (int) $request->otp_code;
        if (!$otpService->verify((string) $user->phone, $otpCode, 'account_deletion')) {
            return response()->json([
                'success' => false,
                'message' => 'Doğrulama kodu hatalı veya süresi dolmuş.',
            ], 422);
        }

        return response()->json([
            'success' => true,
            'message' => 'Doğrulama başarılı. Hesabı silme işlemini tamamlayabilirsiniz.',
        ]);
    }

    public function deleteAccount(Request $request, OtpService $otpService): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'otp_code' => 'required|string|size:6',
            'password' => 'required|string',
        ], [
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
            'password.required' => 'Şifre zorunludur.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil.',
            ], 400);
        }

        if (!Hash::check((string) $request->password, (string) $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Şifre hatalı.',
            ], 422);
        }

        $otpCode = (int) $request->otp_code;
        if (!$otpService->verify((string) $user->phone, $otpCode, 'account_deletion')) {
            return response()->json([
                'success' => false,
                'message' => 'Doğrulama kodu hatalı veya süresi dolmuş.',
            ], 422);
        }

        try {
            DB::beginTransaction();

            $user->tokens()->delete();
            $deleted = $user->delete();
            if (!$deleted) {
                throw new \RuntimeException('Kullanıcı silinemedi.');
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Hesabınız başarıyla silindi.',
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('iOS hesap silme hatası', [
                'user_id' => $user->id ?? null,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Hesap silinirken bir hata oluştu. Lütfen destek ile iletişime geçin.',
            ], 500);
        }
    }

    private function mapUserPayload(User $user): array
    {
        return [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'phone' => $user->phone,
            'role' => $user->role,
            'is_personel_admin' => $user->is_personel_admin ?? false,
            'salon_id' => $user->beauty_salon_id,
            'salon_name' => $user->salon_name,
            'salon_short_name' => $user->salon_short_name,
            'salon_slug' => $user->salon_slug,
            'short_url' => $user->short_url,
            'salon_phone' => $user->salon_phone,
            'salon_city' => $user->salon_city,
            'salon_district' => $user->salon_district,
            'salon_address' => $user->salon_address,
            'salon_description' => $user->salon_description,
            'salon_color' => $user->salon_color,
            'salon_theme' => $user->salon_theme,
            'salon_working_hours' => $user->salon_working_hours ?? [],
            'avatar' => $user->profile_image ? asset('storage/' . $user->profile_image) : null,
            'salon_profile_image' => $user->salon_profile_image ? asset('storage/' . $user->salon_profile_image) : null,
            'salon_banner_image' => $user->salon_banner_image ? asset('storage/' . $user->salon_banner_image) : null,
            'salon_gallery' => is_array($user->salon_gallery) ? array_map(function ($item) {
                return asset('storage/' . ltrim((string) $item, '/'));
            }, $user->salon_gallery) : [],
        ];
    }

    private function normalizePhone(string $phone): string
    {
        $normalized = preg_replace('/[^0-9]/', '', $phone) ?? '';
        if ($normalized !== '' && !str_starts_with($normalized, '0')) {
            $normalized = '0' . $normalized;
        }
        return $normalized;
    }

    private function generateUniqueSalonSlug(string $source, int $exceptUserId): string
    {
        $slug = Str::slug($source);
        if ($slug === '') {
            $slug = 'salon';
        }

        $candidate = $slug;
        $counter = 1;
        while (User::where('salon_slug', $candidate)->where('id', '!=', $exceptUserId)->exists()) {
            $candidate = $slug . '-' . $counter;
            $counter++;
        }

        return $candidate;
    }

    private function generateUniqueShortUrl(string $base, int $exceptUserId): string
    {
        $candidate = $base;
        $counter = 1;
        while (User::where('short_url', $candidate)->where('id', '!=', $exceptUserId)->exists()) {
            $candidate = $base . $counter;
            $counter++;
        }

        return $candidate;
    }

    private function normalizeSalonWorkingHours(array $input): array
    {
        $days = [
            'monday' => 'Pazartesi',
            'tuesday' => 'Salı',
            'wednesday' => 'Çarşamba',
            'thursday' => 'Perşembe',
            'friday' => 'Cuma',
            'saturday' => 'Cumartesi',
            'sunday' => 'Pazar',
        ];

        $normalized = [];
        $openDayCount = 0;

        foreach ($days as $dayKey => $dayLabel) {
            $dayData = (array) ($input[$dayKey] ?? []);
            $isOpen = filter_var($dayData['is_open'] ?? false, FILTER_VALIDATE_BOOLEAN);

            $normalized[$dayKey] = [
                'is_open' => $isOpen,
                'open_time' => null,
                'close_time' => null,
            ];

            if (!$isOpen) {
                continue;
            }

            $openTime = trim((string) ($dayData['open_time'] ?? ''));
            $closeTime = trim((string) ($dayData['close_time'] ?? ''));

            if (!preg_match('/^\d{2}:\d{2}$/', $openTime) || !preg_match('/^\d{2}:\d{2}$/', $closeTime)) {
                return [
                    'ok' => false,
                    'message' => $dayLabel . ' için açılış ve kapanış saatini seçmelisiniz.',
                ];
            }

            if ($openTime >= $closeTime) {
                return [
                    'ok' => false,
                    'message' => $dayLabel . ' için kapanış saati açılış saatinden sonra olmalıdır.',
                ];
            }

            $normalized[$dayKey]['open_time'] = $openTime;
            $normalized[$dayKey]['close_time'] = $closeTime;
            $openDayCount++;
        }

        if ($openDayCount === 0) {
            return [
                'ok' => false,
                'message' => 'En az bir günü açık olarak seçmelisiniz.',
            ];
        }

        return [
            'ok' => true,
            'data' => $normalized,
        ];
    }

    private function syncSalonModel(User $user): void
    {
        $salon = \App\Models\Salon::where('created_by', $user->id)->first();

        if (!$salon) {
            \App\Models\Salon::create([
                'name' => $user->salon_name ?: $user->name,
                'created_by' => $user->id,
                'is_active' => true,
                'capacity' => 100,
                'address' => $user->salon_address,
                'phone' => $user->salon_phone,
                'city' => $user->salon_city,
                'district' => $user->salon_district,
            ]);
            return;
        }

        $salon->name = $user->salon_name ?: $salon->name;
        $salon->address = $user->salon_address;
        $salon->phone = $user->salon_phone;
        $salon->city = $user->salon_city;
        $salon->district = $user->salon_district;
        $salon->save();
    }
}

