<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Menu;
use App\Models\Reservation;
use App\Models\Transaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CustomerController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = Customer::query();
        
        // Personel için müşteri listesi gösterilmez veya sadece kendi randevularındaki müşteriler
        if ($user->role === 'personel') {
            $personelCustomerIds = Reservation::where('personel_id', $user->id)
                ->distinct()
                ->pluck('customer_id')
                ->filter();
            $query->whereIn('id', $personelCustomerIds);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }
        
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }
        
        $customers = $query->orderBy('name', 'asc')
            ->paginate($request->get('per_page', 20));
        
        return response()->json([
            'success' => true,
            'data' => collect($customers->items())->map(fn($customer) => $this->formatCustomer($customer, $user))->values(),
            'meta' => [
                'current_page' => $customers->currentPage(),
                'last_page' => $customers->lastPage(),
                'per_page' => $customers->perPage(),
                'total' => $customers->total(),
            ],
        ]);
    }
    
    public function search(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $request->validate([
            'q' => 'required|string|min:2',
        ]);
        
        $query = Customer::query();
        
        if ($user->role === 'personel') {
            $personelCustomerIds = Reservation::where('personel_id', $user->id)
                ->distinct()
                ->pluck('customer_id')
                ->filter();
            $query->whereIn('id', $personelCustomerIds);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }
        
        $search = $request->q;
        $customers = $query->where(function($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%");
        })
        ->limit(20)
        ->get();
        
        return response()->json([
            'success' => true,
            'data' => $customers->map(fn($customer) => $this->formatCustomer($customer, $user)),
        ]);
    }
    
    public function show(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        $customer = Customer::find($id);
        
        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Müşteri bulunamadı',
            ], 404);
        }
        
        // Personel kontrolü
        if ($user->role === 'personel') {
            $hasReservation = Reservation::where('customer_id', $customer->id)
                ->where('personel_id', $user->id)
                ->exists();
            if (!$hasReservation) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu müşteriye erişim yetkiniz yok',
                ], 403);
            }
        }
        
        return response()->json([
            'success' => true,
            'data' => $this->formatCustomer($customer, $user, true),
        ]);
    }
    
    public function store(Request $request): JsonResponse
    {
        $user = $request->user();
        
        // Personel müşteri oluşturamaz
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Müşteri oluşturma yetkiniz yok',
            ], 403);
        }
        
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string|max:500',
            'notes' => 'nullable|string|max:1000',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }
        
        $customer = Customer::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'address' => $request->address,
            'notes' => $request->notes,
            'beauty_salon_id' => $user->beauty_salon_id ?? $user->id,
            'created_by' => $user->id,
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Müşteri oluşturuldu',
            'data' => $this->formatCustomer($customer, $user),
        ], 201);
    }
    
    public function update(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        // Personel müşteri güncelleyemez
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Müşteri güncelleme yetkiniz yok',
            ], 403);
        }
        
        $customer = Customer::find($id);
        
        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Müşteri bulunamadı',
            ], 404);
        }
        
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'phone' => 'sometimes|required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string|max:500',
            'notes' => 'nullable|string|max:1000',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }
        
        $customer->update($request->only(['name', 'phone', 'email', 'address', 'notes']));
        
        return response()->json([
            'success' => true,
            'message' => 'Müşteri güncellendi',
            'data' => $this->formatCustomer($customer, $user),
        ]);
    }
    
    private function formatCustomer(Customer $customer, $user, $detailed = false): array
    {
        $data = [
            'id' => $customer->id,
            'name' => $customer->name,
            'phone' => $customer->phone,
            'email' => $customer->email,
            'address' => $customer->address,
            'balance' => (float) ($customer->balance ?? 0),
        ];
        
        if ($detailed && $user->role !== 'personel') {
            // Toplam borç ve ödenen hesapla
            $reservationQuery = Reservation::where('customer_id', $customer->id);
            if ($user->role === 'personel') {
                $reservationQuery->where('personel_id', $user->id);
            }
            
            $totalDebt = $reservationQuery->sum('total_price') ?? 0;
            $totalDepositPaid = $reservationQuery->sum('deposit_paid') ?? 0;
            
            $transactionQuery = Transaction::where('customer_id', $customer->id);
            if ($user->role === 'personel') {
                $transactionQuery->where('created_by', $user->id);
            }
            $totalIncomeTransactions = (clone $transactionQuery)->where('type', 'income')->sum('amount') ?? 0;
            $totalPaid = $totalDepositPaid + $totalIncomeTransactions;
            $totalRemaining = max(0, $totalDebt - $totalPaid);
            
            $data['total_debt'] = (float) $totalDebt;
            $data['total_paid'] = (float) $totalPaid;
            $data['total_remaining'] = (float) $totalRemaining;
            $data['notes'] = $customer->notes;
            $data['reservations_count'] = $reservationQuery->count();

            // Müşteri geçmişi: randevular
            $reservations = (clone $reservationQuery)
                ->with(['personel:id,name'])
                ->orderByDesc('date')
                ->orderByDesc('start_time')
                ->limit(100)
                ->get([
                    'id',
                    'menu_id',
                    'personel_id',
                    'date',
                    'start_time',
                    'end_time',
                    'status',
                    'total_price',
                    'deposit_paid',
                    'remaining_amount',
                    'notes',
                    'items_data',
                ]);

            $menuIds = $reservations->pluck('menu_id')->filter()->unique()->values();
            $menusById = $menuIds->isNotEmpty()
                ? Menu::whereIn('id', $menuIds)->pluck('name', 'id')
                : collect();

            $data['reservations'] = $reservations->map(function (Reservation $reservation) use ($menusById) {
                $serviceName = $menusById[$reservation->menu_id] ?? null;

                if (!$serviceName && !empty($reservation->items_data)) {
                    $decoded = json_decode($reservation->items_data, true);
                    if (is_array($decoded) && isset($decoded[0])) {
                        $firstItem = $decoded[0];
                        if (is_array($firstItem) && !empty($firstItem['name'])) {
                            $serviceName = $firstItem['name'];
                        }
                    }
                }

                return [
                    'id' => $reservation->id,
                    'date' => optional($reservation->date)->format('Y-m-d'),
                    'start_time' => $reservation->start_time,
                    'end_time' => $reservation->end_time,
                    'status' => (string) $reservation->status,
                    'service_name' => $serviceName,
                    'staff_name' => optional($reservation->personel)->name,
                    'total_price' => (float) ($reservation->total_price ?? 0),
                    'deposit_paid' => (float) ($reservation->deposit_paid ?? 0),
                    'remaining_amount' => (float) ($reservation->remaining_amount ?? 0),
                    'notes' => $reservation->notes,
                ];
            })->values();

            // Müşteri geçmişi: finans hareketleri
            $transactions = Transaction::where('customer_id', $customer->id)
                ->orderByDesc('date')
                ->orderByDesc('id')
                ->limit(150)
                ->get([
                    'id',
                    'reservation_id',
                    'type',
                    'amount',
                    'date',
                    'description',
                    'status',
                    'payment_method',
                    'payment_reference',
                    'created_at',
                ]);

            $data['transactions'] = $transactions->map(function (Transaction $transaction) {
                return [
                    'id' => $transaction->id,
                    'reservation_id' => $transaction->reservation_id,
                    'type' => (string) $transaction->type,
                    'amount' => (float) ($transaction->amount ?? 0),
                    'date' => optional($transaction->date)->format('Y-m-d'),
                    'description' => $transaction->description,
                    'status' => $transaction->status,
                    'payment_method' => $transaction->payment_method,
                    'payment_reference' => $transaction->payment_reference,
                    'created_at' => optional($transaction->created_at)->toDateTimeString(),
                ];
            })->values();

            $data['activity_summary'] = [
                'total_reservations' => (int) $reservations->count(),
                'completed_reservations' => (int) $reservations->whereIn('status', ['completed', 'tamamlandi'])->count(),
                'cancelled_reservations' => (int) $reservations->whereIn('status', ['cancelled', 'iptal'])->count(),
                'open_reservations' => (int) $reservations->whereIn('status', ['open', 'beklemede'])->count(),
                'confirmed_reservations' => (int) $reservations->whereIn('status', ['confirmed', 'onaylandi'])->count(),
                'total_transactions' => (int) $transactions->count(),
            ];
        }
        
        return $data;
    }
}

