<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        $today = Carbon::today();
        $cancelStatuses = ['iptal', 'cancelled'];
        $completedStatuses = ['tamamlandi', 'completed'];
        $confirmedStatuses = ['onaylandi', 'confirmed'];
        $openStatuses = ['beklemede', 'open'];

        $reservationQuery = Reservation::query();
        
        // Personel sadece kendi randevularını görür
        if ($user->role === 'personel') {
            $reservationQuery->where('personel_id', $user->id);
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            // Admin/salon sahibi tüm randevuları görür
            if ($user->beauty_salon_id) {
                $reservationQuery->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $reservationQuery->where('beauty_salon_id', $user->id);
            }
        }

        $todayAppointments = (clone $reservationQuery)
            ->whereDate('date', $today)
            ->whereNotIn('status', $cancelStatuses)
            ->count();

        $totalAppointments = (clone $reservationQuery)
            ->whereNotIn('status', $cancelStatuses)
            ->count();

        $upcomingAppointments = (clone $reservationQuery)
            ->whereIn('status', array_merge($confirmedStatuses, $openStatuses))
            ->whereDate('date', '>=', $today)
            ->orderBy('date', 'asc')
            ->orderBy('start_time', 'asc')
            ->limit(5)
            ->get()
            ->map(function ($apt) {
                $dateValue = $apt->date ? Carbon::parse($apt->date)->format('Y-m-d') : null;
                return [
                    'id' => $apt->id,
                    'customer_name' => optional($apt->customer)->name ?? 'Müşteri',
                    'service_name' => optional($apt->service)->name ?? 'Hizmet',
                    'date' => $dateValue ?? Carbon::today()->format('Y-m-d'),
                    'start_time' => $apt->start_time ?? '--:--',
                    'status' => $apt->status,
                ];
            });

        $customerQuery = Customer::query();
        if ($user->role === 'personel') {
            // Personel için müşteri sayısı gösterilmez veya sadece kendi randevularındaki müşteriler
        } elseif (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $customerQuery->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $customerQuery->where('beauty_salon_id', $user->id);
            }
        }
        $totalCustomers = $customerQuery->count();

        $revenueQuery = (clone $reservationQuery)
            ->whereIn('status', $completedStatuses)
            ->whereMonth('date', $today->month)
            ->whereYear('date', $today->year);
        
        $monthlyRevenue = $revenueQuery->sum('total_price') ?? 0.0;

        // Haftalık gelir
        $weekStart = $today->copy()->startOfWeek();
        $weeklyRevenue = (clone $reservationQuery)
            ->whereIn('status', $completedStatuses)
            ->whereBetween('date', [$weekStart, $today])
            ->sum('total_price') ?? 0.0;

        // Bugünkü gelir
        $todayRevenue = (clone $reservationQuery)
            ->whereIn('status', $completedStatuses)
            ->whereDate('date', $today)
            ->sum('total_price') ?? 0.0;

        // Durum bazlı istatistikler
        $statusStats = [
            'beklemede' => (clone $reservationQuery)->whereIn('status', $openStatuses)->count(),
            'onaylandi' => (clone $reservationQuery)->whereIn('status', $confirmedStatuses)->count(),
            'tamamlandi' => (clone $reservationQuery)->whereIn('status', $completedStatuses)->count(),
            'iptal' => (clone $reservationQuery)->whereIn('status', $cancelStatuses)->count(),
        ];

        // Son tamamlanan randevular
        $recentCompleted = (clone $reservationQuery)
            ->whereIn('status', $completedStatuses)
            ->orderBy('date', 'desc')
            ->orderBy('start_time', 'desc')
            ->limit(5)
            ->get()
            ->map(function ($apt) {
                $dateValue = $apt->date ? Carbon::parse($apt->date)->format('Y-m-d') : null;
                return [
                    'id' => $apt->id,
                    'customer_name' => optional($apt->customer)->name ?? 'Müşteri',
                    'service_name' => optional($apt->service)->name ?? 'Hizmet',
                    'date' => $dateValue ?? Carbon::today()->format('Y-m-d'),
                    'start_time' => $apt->start_time ?? '--:--',
                    'total_price' => (float) $apt->total_price,
                ];
            });

        // Bu hafta randevu sayıları (günlük)
        $weeklyAppointments = [];
        for ($i = 0; $i < 7; $i++) {
            $date = $weekStart->copy()->addDays($i);
            $count = (clone $reservationQuery)
                ->whereDate('date', $date)
                ->whereNotIn('status', $cancelStatuses)
                ->count();
            $weeklyAppointments[] = [
                'day' => $date->format('d.m'),
                'day_name' => $date->format('D'),
                'count' => $count,
            ];
        }

        return response()->json([
            'success' => true,
            'data' => [
                'today_appointments' => $todayAppointments,
                'total_appointments' => $totalAppointments,
                'total_customers' => $user->role === 'personel' ? null : $totalCustomers,
                'monthly_revenue' => (float) $monthlyRevenue,
                'weekly_revenue' => (float) $weeklyRevenue,
                'today_revenue' => (float) $todayRevenue,
                'status_stats' => $statusStats,
                'upcoming_appointments' => $upcomingAppointments,
                'recent_completed' => $recentCompleted,
                'weekly_appointments' => $weeklyAppointments,
            ],
        ]);
    }
}

