<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Notifications\DatabaseNotification;

class NotificationController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $notifications = $user->notifications()
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 50));
        
        return response()->json([
            'success' => true,
            'data' => collect($notifications->items())->map(function ($notification) {
                $data = is_string($notification->data) 
                    ? json_decode($notification->data, true) 
                    : $notification->data;
                
                return [
                    'id' => $notification->id,
                    'type' => $notification->type,
                    'title' => $data['title'] ?? null,
                    'message' => $data['message'] ?? null,
                    'data' => $data,
                    'read_at' => $notification->read_at?->toISOString(),
                    'created_at' => $notification->created_at?->toISOString(),
                ];
            })->values(),
            'meta' => [
                'current_page' => $notifications->currentPage(),
                'last_page' => $notifications->lastPage(),
                'per_page' => $notifications->perPage(),
                'total' => $notifications->total(),
            ],
        ]);
    }
    
    public function unreadCount(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $count = $user->unreadNotifications()->count();
        
        return response()->json([
            'success' => true,
            'data' => [
                'count' => $count,
            ],
        ]);
    }
    
    public function markAsRead(Request $request, string $id): JsonResponse
    {
        $user = $request->user();
        
        $notification = $user->notifications()->find($id);
        
        if (!$notification) {
            return response()->json([
                'success' => false,
                'message' => 'Bildirim bulunamadı',
            ], 404);
        }
        
        $notification->markAsRead();
        
        return response()->json([
            'success' => true,
            'message' => 'Bildirim okundu olarak işaretlendi',
        ]);
    }
    
    public function markAllAsRead(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $user->unreadNotifications->markAsRead();
        
        return response()->json([
            'success' => true,
            'message' => 'Tüm bildirimler okundu olarak işaretlendi',
        ]);
    }
}

