<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Room;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class RoomController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = Room::query();
        
        if (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        } elseif ($user->role === 'personel') {
            // Personel tüm odaları görebilir (randevu oluştururken gerekli)
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        }
        
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }
        
        $rooms = $query->orderBy('name', 'asc')
            ->paginate($request->get('per_page', 50));
        
        return response()->json([
            'success' => true,
            'data' => collect($rooms->items())->map(fn($room) => $this->formatRoom($room))->values(),
            'meta' => [
                'current_page' => $rooms->currentPage(),
                'last_page' => $rooms->lastPage(),
                'per_page' => $rooms->perPage(),
                'total' => $rooms->total(),
            ],
        ]);
    }
    
    public function show(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        $room = Room::find($id);
        
        if (!$room) {
            return response()->json([
                'success' => false,
                'message' => 'Oda bulunamadı',
            ], 404);
        }
        
        return response()->json([
            'success' => true,
            'data' => $this->formatRoom($room),
        ]);
    }
    
    public function store(Request $request): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Oda oluşturma yetkiniz yok',
            ], 403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'capacity' => 'nullable|integer|min:1',
            'color' => 'nullable|string|max:7',
        ]);
        
        $salonId = $user->beauty_salon_id ?? $user->id;
        
        $room = Room::create([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'capacity' => $validated['capacity'] ?? 1,
            'color' => $validated['color'] ?? '#ec4899',
            'beauty_salon_id' => $salonId,
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Oda başarıyla oluşturuldu',
            'data' => $this->formatRoom($room),
        ], 201);
    }
    
    public function update(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Oda güncelleme yetkiniz yok',
            ], 403);
        }

        $salonId = $user->beauty_salon_id ?? $user->id;
        $room = Room::where('beauty_salon_id', $salonId)->find($id);
        if (!$room) {
            return response()->json([
                'success' => false,
                'message' => 'Oda bulunamadı',
            ], 404);
        }

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'capacity' => 'nullable|integer|min:1',
            'color' => 'nullable|string|max:7',
        ]);

        $room->update([
            'name' => $validated['name'] ?? $room->name,
            'description' => array_key_exists('description', $validated) ? $validated['description'] : $room->description,
            'capacity' => $validated['capacity'] ?? $room->capacity,
            'color' => $validated['color'] ?? $room->color,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Oda başarıyla güncellendi',
            'data' => $this->formatRoom($room->fresh()),
        ]);
    }
    
    public function destroy(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Oda silme yetkiniz yok',
            ], 403);
        }

        $salonId = $user->beauty_salon_id ?? $user->id;
        $room = Room::where('beauty_salon_id', $salonId)->find($id);
        if (!$room) {
            return response()->json([
                'success' => false,
                'message' => 'Oda bulunamadı',
            ], 404);
        }

        $room->delete();

        return response()->json([
            'success' => true,
            'message' => 'Oda silindi',
        ]);
    }
    
    private function formatRoom(Room $room): array
    {
        return [
            'id' => $room->id,
            'name' => $room->name,
            'description' => $room->description,
            'capacity' => $room->capacity,
            'color' => $room->color,
            'tags' => $room->tags ?? [],
        ];
    }
}

