<?php

namespace App\Http\Controllers\Api\IosApi;

use App\Http\Controllers\Controller;
use App\Models\Service;
use App\Models\ServiceCategory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ServiceController extends Controller
{
    public function categories(Request $request): JsonResponse
    {
        $user = $request->user();
        $salonId = $user->beauty_salon_id ?? $user->id;

        $categories = ServiceCategory::query()
            ->where('beauty_salon_id', $salonId)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $categories->map(function (ServiceCategory $category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                ];
            })->values(),
        ]);
    }

    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = Service::with(['category', 'beautySalon']);
        
        if (in_array($user->role, ['güzellik_salonu', 'admin', 'super_admin'])) {
            if ($user->beauty_salon_id) {
                $query->where('beauty_salon_id', $user->beauty_salon_id);
            } else {
                $query->where('beauty_salon_id', $user->id);
            }
        } elseif ($user->role === 'personel') {
            // Personel sadece kendi hizmetlerini görür
            $query->whereHas('personels', function ($q) use ($user) {
                $q->where('personel_id', $user->id);
            });
        }
        
        if ($request->has('category_id')) {
            $query->where('service_category_id', $request->category_id);
        }
        
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }
        
        $services = $query->orderBy('name', 'asc')
            ->paginate($request->get('per_page', 50));
        
        return response()->json([
            'success' => true,
            'data' => collect($services->items())->map(fn($service) => $this->formatService($service))->values(),
            'meta' => [
                'current_page' => $services->currentPage(),
                'last_page' => $services->lastPage(),
                'per_page' => $services->perPage(),
                'total' => $services->total(),
            ],
        ]);
    }
    
    public function show(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        $service = Service::with(['category', 'beautySalon', 'personels'])->find($id);
        
        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Hizmet bulunamadı',
            ], 404);
        }
        
        // Personel kontrolü
        if ($user->role === 'personel') {
            $hasAccess = $service->personels()->where('personel_id', $user->id)->exists();
            if (!$hasAccess) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu hizmete erişim yetkiniz yok',
                ], 403);
            }
        }
        
        return response()->json([
            'success' => true,
            'data' => $this->formatService($service, true),
        ]);
    }
    
    public function store(Request $request): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Hizmet oluşturma yetkiniz yok',
            ], 403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration' => 'required|integer|min:1',
            'category_id' => 'nullable|exists:service_categories,id',
        ]);
        
        $salonId = $user->beauty_salon_id ?? $user->id;
        
        $service = Service::create([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'price' => $validated['price'],
            'duration' => $validated['duration'],
            'service_category_id' => $validated['category_id'] ?? null,
            'beauty_salon_id' => $salonId,
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Hizmet başarıyla oluşturuldu',
            'data' => $this->formatService($service),
        ], 201);
    }
    
    public function update(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Hizmet güncelleme yetkiniz yok',
            ], 403);
        }

        $salonId = $user->beauty_salon_id ?? $user->id;
        $service = Service::where('beauty_salon_id', $salonId)->find($id);
        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Hizmet bulunamadı',
            ], 404);
        }

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'price' => 'sometimes|numeric|min:0',
            'duration' => 'sometimes|integer|min:1',
            'category_id' => 'nullable|exists:service_categories,id',
        ]);

        $service->update([
            'name' => $validated['name'] ?? $service->name,
            'description' => array_key_exists('description', $validated) ? $validated['description'] : $service->description,
            'price' => $validated['price'] ?? $service->price,
            'duration' => $validated['duration'] ?? $service->duration,
            'service_category_id' => array_key_exists('category_id', $validated) ? $validated['category_id'] : $service->service_category_id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Hizmet başarıyla güncellendi',
            'data' => $this->formatService($service->fresh()),
        ]);
    }
    
    public function destroy(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        if ($user->role === 'personel') {
            return response()->json([
                'success' => false,
                'message' => 'Hizmet silme yetkiniz yok',
            ], 403);
        }

        $salonId = $user->beauty_salon_id ?? $user->id;
        $service = Service::where('beauty_salon_id', $salonId)->find($id);
        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Hizmet bulunamadı',
            ], 404);
        }

        $service->delete();

        return response()->json([
            'success' => true,
            'message' => 'Hizmet silindi',
        ]);
    }
    
    private function formatService(Service $service, $detailed = false): array
    {
        $data = [
            'id' => $service->id,
            'name' => $service->name,
            'description' => $service->description,
            'price' => (float) $service->price,
            'duration' => $service->duration,
            'category_id' => $service->service_category_id,
            'category_name' => $service->category->name ?? null,
        ];
        
        if ($detailed) {
            $data['personels'] = $service->personels->map(function ($personel) {
                return [
                    'id' => $personel->id,
                    'name' => $personel->name,
                ];
            });
        }
        
        return $data;
    }
}

