<?php

namespace App\Http\Controllers\Api\Mobile;

use App\Http\Controllers\Controller;
use App\Models\NotificationDevice;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DeviceRegistrationController extends Controller
{
    public function register(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'device_token' => 'required|string|max:255',
            'device_type' => 'required|in:ios,android',
            'app_version' => 'nullable|string|max:20',
            'os_version' => 'nullable|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $device = NotificationDevice::updateOrCreate(
            [
                'device_token' => $request->device_token,
            ],
            [
                'user_id' => $user->id,
                'platform' => $request->device_type,
                'app_version' => $request->app_version,
                'os_version' => $request->os_version,
                'last_seen_at' => now(),
                'is_active' => true,
            ]
        );

        return response()->json([
            'success' => true,
            'message' => 'Cihaz başarıyla kaydedildi',
            'data' => [
                'id' => $device->id,
            ],
        ]);
    }

    public function unregister(Request $request, string $token): JsonResponse
    {
        $device = NotificationDevice::where('device_token', $token)
            ->where('user_id', $request->user()->id)
            ->first();

        if ($device) {
            $device->update(['is_active' => false]);

        }

        return response()->json([
            'success' => true,
            'message' => 'Cihaz kaydı kaldırıldı',
        ]);
    }
}

